/**
 * Copyright (C) 2025 Finn Landweber and olell
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

import TherapistListEntry from "@/components/TherapistListEntry";
import TimeDivider from "@/components/TimeDivider";
import TimeslotEntity from "@/data/entity/TimeslotEntity";
import { getTherapistsDS, loadEnabledTimeslots } from "@/utils/storage";
import { getTherapistName, timeslotToString } from "@/utils/misc";
import { getReachabilities } from "@/utils/reachability";
import { useFocusEffect, useRouter } from "expo-router";
import { useCallback, useState } from "react";
import { View, StyleSheet, SectionList } from "react-native";
import { Text } from "react-native-paper";

interface CallSection {
  title: string;
  data: TimeslotEntity[];
}

export default function Tab() {
  const router = useRouter();
  const TherapistsDS = getTherapistsDS();

  const [sectionData, setSectionData] = useState<CallSection[]>([]);
  const [isRefreshing, setIsRefreshing] = useState(false);

  const updateSectionData = () => {
    setIsRefreshing(true);
    loadEnabledTimeslots(TherapistsDS)
      .then((timeslots) => {
        const reachabilites = getReachabilities(timeslots, 1);
        const data = [
          { title: "Aktuell erreichbar", data: reachabilites.currently },
          { title: "Bald erreichbar", data: reachabilites.soon },
          ...reachabilites.later.map((x, index) => ({
            title: `In ${(index + 1).toString()} Stunde${index + 1 > 1 ? "n" : ""} erreichbar`,
            data: x,
          })),
        ];
        setSectionData(data.filter((x) => x.data.length > 0));
        setIsRefreshing(false);
      })
      .catch((reason: unknown) => {
        console.log("Error refreshing sections", reason);
        setIsRefreshing(false);
      });
  };

  useFocusEffect(
    useCallback(() => {
      updateSectionData();
    }, []),
  );

  return (
    <View style={styles.container}>
      {sectionData.length > 0 ? (
        <SectionList
          sections={sectionData}
          renderSectionHeader={({ section: { title } }) => (
            <TimeDivider title={title} />
          )}
          onRefresh={updateSectionData}
          refreshing={isRefreshing}
          style={styles.sectionlist}
          renderItem={(item) => (
            <TherapistListEntry
              name={getTherapistName(item.item.therapist)}
              bubbleText={timeslotToString(item.item)}
              onPress={() => {
                router.push({
                  pathname: `/call/therapist`,
                  params: { id: item.item.therapist.id },
                });
              }}
              // rightContent={(reset) => (
              //   <Button
              //     icon="edit"
              //     style={{
              //       minHeight: "100%",
              //       backgroundColor: theme.colors.primary,
              //     }}
              //     onPress={() => {
              //       router.push({
              //         pathname: `/list/edit`,
              //         params: { id: item.item.therapist.id },
              //       });
              //       reset();
              //     }}
              //   >
              //     Bearbeiten
              //   </Button>
              // )}
            />
          )}
        />
      ) : (
        <Text>Aktuell keine verfügbaren Erreichbarkeiten</Text>
      )}
    </View>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, justifyContent: "center", alignItems: "center" },
  sectionlist: { flex: 1, width: "100%" },
});
