package org.sensors2.osc.bluetoothSensors;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothClass;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothGatt;
import android.bluetooth.BluetoothGattCallback;
import android.bluetooth.BluetoothGattCharacteristic;
import android.bluetooth.BluetoothGattDescriptor;
import android.bluetooth.BluetoothGattService;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.Log;

import com.google.gson.Gson;

import org.sensors2.osc.bluetoothSensors.sensorHandlers.GeneralHandler;
import org.sensors2.osc.bluetoothSensors.sensorHandlers.Helpers;
import org.sensors2.osc.bluetoothSensors.sensorHandlers.SensorHandler;
import org.sensors2.osc.bluetoothSensors.sensorHandlers.models.BluetoothOscData;
import org.sensors2.osc.bluetoothSensors.sensorHandlers.models.Characteristic;
import org.sensors2.osc.bluetoothSensors.sensorHandlers.models.Device;
import org.sensors2.osc.dispatch.OscDispatcher;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.annotation.RequiresPermission;
import androidx.core.app.ActivityCompat;

import static android.bluetooth.BluetoothDevice.BOND_BONDED;
import static org.sensors2.osc.dispatch.SensorService.BT_PERMISSION_REQUEST;
import static org.sensors2.osc.dispatch.SensorService.BT_REQUEST_CODE;

@SuppressLint("MissingPermission")
@RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
public class BluetoothConnectionManager {
    public static final UUID CLIENT_CHARACTERISTIC_CONFIG_UUID = new UUID(0x290200001000L, 0x800000805f9b34fbL);
    private static final String TAG = BluetoothConnectionManager.class.getSimpleName();
    public static final String DEVICE_ACTION_NAME = "org.sensors2.osc.BT_DEVICES";
    public static final String DEVICE_ACTION_CONTENT = "devices";
    private final Context context;
    private final SensorHandler sensorHandler = new GeneralHandler();
    private OscDispatcher dispatcher;
    private final List<Device> devices = new ArrayList<>();

    public BluetoothConnectionManager(Context context){
        this.context = context;
    }

    private BluetoothAdapter bluetoothAdapter;
    private final BluetoothGattCallback gattCallback = new BluetoothGattCallback() {
        @Override
        public void onConnectionStateChange(BluetoothGatt gatt, int status, int newState) {
            switch (newState) {
                case BluetoothProfile.STATE_CONNECTING:
                case BluetoothProfile.STATE_DISCONNECTING:
                    break;
                case BluetoothProfile.STATE_CONNECTED:
                    gatt.discoverServices();
                    break;
                case BluetoothProfile.STATE_DISCONNECTED:
                    String deviceId = gatt.getDevice().getAddress();
                    Iterator<Device> iter = devices.iterator();
                    boolean changeDetected = false;
                    while (iter.hasNext()) {
                        Device d = iter.next();
                        if (d.getAddress().equals(deviceId)) {
                            iter.remove();
                            changeDetected = true;
                        }
                    }
                    //This is also triggered, if no connection was established (ca. 30s)
                    gatt.connect();
                    if (changeDetected) {
                        broadcastDevices();
                    }
                    break;
            }
        }


        @Override
        public void onServicesDiscovered(@NonNull BluetoothGatt gatt, int status) {
            for (BluetoothGattService gattService : gatt.getServices()) {
                switch(Helpers.getSiginicantString(gattService.getUuid())) {
                    // Do not use generic services, like GAP and GATT
                    // see https://www.bluetooth.com/wp-content/uploads/Files/Specification/HTML/Assigned_Numbers/out/en/Assigned_Numbers.pdf
                    case "0x1800":
                    case "0x1801":
                        continue;
                }
                for (BluetoothGattCharacteristic characteristic: gattService.getCharacteristics()) {
                    // Register for updates, if possible
                    gatt.setCharacteristicNotification(characteristic, true);
                    BluetoothGattDescriptor descriptor = characteristic.getDescriptor(CLIENT_CHARACTERISTIC_CONFIG_UUID);
                    if (descriptor == null) {
                        continue;
                    }
                    if (!descriptor.setValue(BluetoothGattDescriptor.ENABLE_NOTIFICATION_VALUE)) {
                        continue;
                    }
                    if (!gatt.writeDescriptor(descriptor)) {
                        continue;
                    }
                    BluetoothDevice btDevice = gatt.getDevice();
                    Device device = findSendingConfiguration(btDevice);
                    if (device == null){
                        device = new Device( btDevice.getAddress(), btDevice.getName());
                        devices.add(device);
                    }
                    Log.d(TAG, "GATT service discovered: " + Helpers.getSiginicantString(gattService.getUuid()));
                    Log.d(TAG, "characteristic: " + Helpers.getSiginicantString(characteristic.getUuid()));
                    Log.d(TAG, "descriptor: " + Helpers.getSiginicantString(descriptor.getUuid()));
                    if (!device.containsCharacteristic(characteristic.getUuid())){
                        device.addCharacteristic(new Characteristic(characteristic.getUuid()));
                        sensorHandler.addGatt(gatt, gattService);
                    }
                }
            }
            broadcastDevices();
        }

        @Override
        public void onCharacteristicChanged(BluetoothGatt gatt, @NonNull BluetoothGattCharacteristic characteristic) {
            onCharacteristicChanged(gatt, characteristic, characteristic.getValue());
        }

        @Override
        public void onCharacteristicChanged(BluetoothGatt gatt, @NonNull BluetoothGattCharacteristic characteristic, @NonNull byte[] value) {
            UUID serviceUUID = characteristic.getUuid();
            BluetoothDevice btDevice = gatt.getDevice();
            Device device = findSendingConfiguration(btDevice);
            if (device == null || !device.getActive()) {
                return;
            }
            Log.d(TAG, btDevice + ": Received data with service " + serviceUUID + " and characteristics " + characteristic.getUuid());
            BluetoothOscData data = sensorHandler.getPayload(serviceUUID, gatt.getDevice().getAddress(), value);
            if (dispatcher != null && data != null) {
                dispatcher.dispatch(data);
            }
        }
    };

    private void broadcastDevices() {
        Intent intent = new Intent();
        intent.putExtra(DEVICE_ACTION_CONTENT, new Gson().toJson(devices));
        intent.setAction(DEVICE_ACTION_NAME);
        intent.setPackage("org.sensors2.osc");
        context.sendBroadcast(intent);
    }

    private Device findSendingConfiguration(BluetoothDevice btDevice) {
        for (Device listedDevice : devices) {
            if (Objects.equals(listedDevice.getAddress(), btDevice.getAddress())){
                return listedDevice;
            }
        }
        return null;
    }

    @RequiresPermission(Manifest.permission.BLUETOOTH_CONNECT)
    public synchronized void connect() {
        if (bluetoothAdapter == null){
            return;
        }
        Set<BluetoothDevice> pairedDevices = bluetoothAdapter.getBondedDevices();

        if (!pairedDevices.isEmpty()) {
            for (BluetoothDevice device : pairedDevices) {
                if (device.getBondState() == BOND_BONDED) {
                    BluetoothClass bluetoothClass = device.getBluetoothClass();
                    device.connectGatt(context, true, gattCallback);
                    String deviceName = device.getName();
                    Log.d(TAG, deviceName + ": " + bluetoothClass.getMajorDeviceClass() + "; " + bluetoothClass.getDeviceClass());
                }
            }
        }
    }

    public synchronized void disconnect() {
        sensorHandler.disconnect();
    }

    public void checkForPermissions(Activity activity) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            BluetoothManager bluetoothManager = activity.getSystemService(BluetoothManager.class);
            bluetoothAdapter = bluetoothManager.getAdapter();

            if (bluetoothAdapter != null) {
                if (!bluetoothAdapter.isEnabled()) {
                    Intent enableBtIntent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
                    activity.startActivityForResult(enableBtIntent, BT_REQUEST_CODE);
                } else {
                    if (ActivityCompat.checkSelfPermission(context, Manifest.permission.BLUETOOTH_CONNECT) != PackageManager.PERMISSION_GRANTED) {
                        ActivityCompat.requestPermissions(activity, new String[]{Manifest.permission.BLUETOOTH_CONNECT}, BT_PERMISSION_REQUEST);
                    }
                }
            }
        }
    }

    public void setDispatcher(OscDispatcher dispatcher) {
        this.dispatcher = dispatcher;
    }

    public void setDeviceStatus(String bluetoothAddress, boolean active){
        for (Device listedDevice : devices) {
            if (Objects.equals(listedDevice.getAddress(), bluetoothAddress)){
                listedDevice.setActive(active);
            }
        }
    }

    public List<Device> getDevices() {
        return devices;
    }
}
