package org.sensors2.osc.preferences;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.widget.SeekBar;
import android.widget.TextView;

import org.sensors2.osc.R;
import org.sensors2.osc.sensors.Settings;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.preference.PreferenceViewHolder;
import androidx.preference.SeekBarPreference;

public class FormattedSeekBarPreference extends SeekBarPreference {
    private TextView mSeekBarValueTextView;
    private int value;
    boolean mTrackingTouch;
    private int mSeekBarValue;

    public FormattedSeekBarPreference(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
    }

    public FormattedSeekBarPreference(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    public FormattedSeekBarPreference(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
    }

    public FormattedSeekBarPreference(@NonNull Context context) {
        super(context);
    }

    @Override
    public void onBindViewHolder(@NonNull PreferenceViewHolder holder) {
        super.onBindViewHolder(holder);
        this.mSeekBarValueTextView = (TextView) holder.findViewById(R.id.seekbar_value);
        SeekBar mSeekBar = (SeekBar) holder.findViewById(R.id.seekbar);
        mSeekBar.setOnSeekBarChangeListener(mSeekBarChangeListener);
        updateLabelValue(value);
    }

    private final SeekBar.OnSeekBarChangeListener mSeekBarChangeListener = new SeekBar.OnSeekBarChangeListener() {
        @Override
        public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
            if (fromUser && (FormattedSeekBarPreference.this.getUpdatesContinuously() || !mTrackingTouch)) {
                syncValueInternal(seekBar);
            } else {
                // We always want to update the text while the seekbar is being dragged
                updateLabelValue(progress + FormattedSeekBarPreference.this.getMin());
            }
        }

        @Override
        public void onStartTrackingTouch(SeekBar seekBar) {
            mTrackingTouch = true;
        }

        @Override
        public void onStopTrackingTouch(SeekBar seekBar) {
            mTrackingTouch = false;
            if (seekBar.getProgress() + FormattedSeekBarPreference.this.getMin() != mSeekBarValue) {
                syncValueInternal(seekBar);
            }
        }
    };

    void syncValueInternal(@NonNull SeekBar seekBar) {
        int seekBarValue = this.getMin() + seekBar.getProgress();
        if (seekBarValue != mSeekBarValue) {
            if (callChangeListener(seekBarValue)) {
                setValueInternal(seekBarValue, false);
            } else {
                seekBar.setProgress(mSeekBarValue - this.getMin());
                updateLabelValue(mSeekBarValue);
            }
        }
    }

    private void setValueInternal(int seekBarValue, boolean notifyChanged) {
        if (seekBarValue < this.getMin()) {
            seekBarValue = this.getMin();
        }
        if (seekBarValue > this.getMax()) {
            seekBarValue = this.getMax();
        }

        if (seekBarValue != mSeekBarValue) {
            mSeekBarValue = seekBarValue;
            updateLabelValue(mSeekBarValue);
            persistInt(seekBarValue);
            if (notifyChanged) {
                notifyChanged();
            }
        }
    }


    @Override
    public void setValue(int value) {
        super.setValue(value);
        this.value = value;
        updateLabelValue(value);
    }

    @SuppressLint("DefaultLocale")
    void updateLabelValue(int value) {
        if (mSeekBarValueTextView != null) {
            if (value == 0){
                mSeekBarValueTextView.setText(R.string.send_always);
            } else {
                mSeekBarValueTextView.setText(String.format("%.2f %%", Settings.minimumChangeIntToFloat(value)));
            }
        }
    }
}