package org.sensors2.osc.activities;

import android.annotation.SuppressLint;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.SharedPreferences;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Point;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.text.method.LinkMovementMethod;
import android.util.Log;
import android.view.Display;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.CompoundButton;
import android.widget.LinearLayout;
import android.widget.PopupWindow;
import android.widget.TextView;

import org.sensors2.common.dispatch.Measurement;
import org.sensors2.common.sensors.Parameters;
import org.sensors2.osc.R;
import org.sensors2.osc.dispatch.OscConfiguration;
import org.sensors2.osc.dispatch.OscDispatcher;
import org.sensors2.osc.dispatch.SensorConfiguration;
import org.sensors2.osc.dispatch.SensorService;
import org.sensors2.osc.fragments.bluetoothOverlay.ListFragment;
import org.sensors2.osc.fragments.sensorList.AbstractSensorFragment;
import org.sensors2.osc.fragments.MultiTouchFragment;
import org.sensors2.osc.fragments.StartupFragment;
import org.sensors2.osc.sensors.Settings;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.app.AppCompatDelegate;
import androidx.appcompat.widget.Toolbar;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

import static android.view.View.VISIBLE;
import static org.sensors2.osc.dispatch.SensorService.BT_REQUEST_CODE;
import static org.sensors2.osc.fragments.MultiTouchFragment.MAX_POINTER_COUNT;
import static org.sensors2.osc.sensors.Parameters.BT_SENSOR;

public class StartUpActivity extends AppCompatActivity implements CompoundButton.OnCheckedChangeListener, View.OnTouchListener {

    private static final String BT_FRAGMENT = "BLUETOOTH_FRAGMENT";
    private final List<AbstractSensorFragment> sensorFragments = new ArrayList<>();
    private Settings settings;
    private boolean active;
    private SensorService sensorService;
    private final ServiceConnection sensorServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            SensorService.OscBinder binder = (SensorService.OscBinder) service;
            StartUpActivity.this.sensorService = binder.getService();
            StartUpActivity.this.sensorService.setSettings(settings);
            StartUpActivity.this.active = StartUpActivity.this.sensorService.getIsSending();
            ((CompoundButton) findViewById(R.id.active)).setChecked(StartUpActivity.this.active);
            OscDispatcher dispatcher = (OscDispatcher) StartUpActivity.this.sensorService.getDispatcher();

            // Setup multitouch
            for (int i = 0; i < MAX_POINTER_COUNT; i++) {
                SensorConfiguration sensorConfiguration = new SensorConfiguration();
                sensorConfiguration.setSend(true);
                sensorConfiguration.setSensorType(Measurement.pointerIdToSensorType(i));
                sensorConfiguration.setOscParam("touch" + (i + 1));
                dispatcher.addSensorConfiguration(sensorConfiguration);
            }

            // Bluetooth
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2){
                SensorConfiguration bluetooth = new SensorConfiguration();
                bluetooth.setSendDuplicates(true);
                bluetooth.setSend(true);
                bluetooth.setSensorType(BT_SENSOR);
                bluetooth.setOscParam("bt");
                dispatcher.addSensorConfiguration(bluetooth);
            }

            // Setup location
            SensorConfiguration location = new SensorConfiguration();
            location.setSendDuplicates(true);
            location.setSend(true);
            location.setSensorType(0);
            location.setOscParam("location");
            dispatcher.addSensorConfiguration(location);

            // Hookup sensor activity buttons with service
            for (AbstractSensorFragment sensorFragment : StartUpActivity.this.sensorFragments) {
                sensorFragment.setSensorService(StartUpActivity.this.sensorService);
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {

        }
    };

    @Override
    @SuppressLint("NewApi")
    protected void onCreate(Bundle savedInstanceState) {
        this.settings = this.loadSettings();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            AppCompatDelegate.setDefaultNightMode(this.settings.getColorScheme());
        } else {
            AppCompatDelegate.setDefaultNightMode(AppCompatDelegate.MODE_NIGHT_YES);
        }

        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        FragmentManager fm = getSupportFragmentManager();
        FragmentTransaction transaction = fm.beginTransaction();
        StartupFragment startupFragment = (StartupFragment) fm.findFragmentByTag("sensorlist");
        if (startupFragment == null) {
            startupFragment = new StartupFragment();
            transaction.add(R.id.container, startupFragment, "sensorlist");
            transaction.commit();
        }
        startService(new Intent(this, SensorService.class));
        bindService(new Intent(this, SensorService.class), this.sensorServiceConnection, BIND_AUTO_CREATE);

        @SuppressLint({"MissingInflatedId", "LocalSuppress"}) Toolbar toolbar = findViewById(R.id.action_bar);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            toolbar.setNavigationIcon(R.drawable.sensors2osc);
        }
    }

    private void showFirstLaunch() {
        showPopup(R.string.popup_first_launch, R.string.action_settings, SettingsActivity.class);
    }

    @SuppressLint({"GestureBackNavigation", "ClickableViewAccessibility"})
    private void showPopup(int messageId, int buttonTextId, Class<?> buttonAction) {
        LayoutInflater inflater = (LayoutInflater)getSystemService(LAYOUT_INFLATER_SERVICE);
        @SuppressLint("InflateParams") View popupView = inflater.inflate(R.layout.popup, null);
        TextView messageView = popupView.findViewById(R.id.popup_text);
        messageView.setMovementMethod(LinkMovementMethod.getInstance());
        messageView.setText(messageId);
        boolean showButton = buttonTextId > 0;
        final PopupWindow popupWindow = new PopupWindow(popupView, LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT, true);
        if (showButton){
            Button actionButton = popupView.findViewById(R.id.popup_action);
            actionButton.setText(buttonTextId);
            actionButton.setOnClickListener(v -> {
                Intent intent = new Intent(StartUpActivity.this, buttonAction);
                startActivity(intent);
            });
            actionButton.setVisibility(VISIBLE);
        }
        popupWindow.showAtLocation(getWindow().getDecorView(), Gravity.CENTER, 0, 0);
        popupView.setOnTouchListener((v, event) -> {
            popupWindow.dismiss();
            return true;
        });
        popupWindow.getContentView().setOnKeyListener((v, keyCode, event) -> {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                popupWindow.dismiss();
                return true;
            }
            return false;
        });
        popupWindow.setOnDismissListener(() -> {
            SharedPreferences preferences = getSharedPreferences(getPackageName() + "_preferences", Context.MODE_PRIVATE);
            try {
                PackageInfo pInfo = getPackageManager().getPackageInfo(getPackageName(), 0);
                settings.setAndSaveLaunchVersion(pInfo.versionCode, preferences);
            } catch (PackageManager.NameNotFoundException e) {
                settings.setAndSaveLaunchVersion(-1, preferences);
                Log.e(StartUpActivity.class.getSimpleName(), "showFirstLaunch", e);
            }
        });

    }

    @Override
    public void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        setIntent(intent);
    }

    private Settings loadSettings() {
        SharedPreferences preferences = getSharedPreferences(getPackageName() + "_preferences", Context.MODE_PRIVATE);
        Settings settings = new Settings(preferences);
        OscConfiguration oscConfiguration = OscConfiguration.getInstance();
        oscConfiguration.setHost(settings.getHost());
        oscConfiguration.setPort(settings.getPort());
        oscConfiguration.setSendAsBundle(settings.getSetAsBundle());
        return settings;
    }

    @Override
    public boolean onCreateOptionsMenu(@NonNull Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.start_up, menu);
        return true;
    }

    @SuppressLint("NonConstantResourceId")
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        switch (id) {
            case R.id.action_settings: {
                Intent intent = new Intent(this, SettingsActivity.class);
                startActivity(intent);
                return true;
            }
            case R.id.action_guide: {
                Intent intent = new Intent(this, GuideActivity.class);
                startActivity(intent);
                return true;
            }
            case R.id.action_about: {
                Intent intent = new Intent(this, AboutActivity.class);
                startActivity(intent);
                return true;
            }
        }
        return super.onOptionsItemSelected(item);
    }


    @Override
    @SuppressLint("NewApi")
    protected void onResume() {
        super.onResume();
        if (sensorService != null){
            sensorService.setSettings(this.settings);
        }
        getWindow().getDecorView().post(() -> {
            if (settings.getLaunchVersion() == 0 && Objects.equals(settings.getHost(), Settings.DEFAULT_HOST)) {
                showFirstLaunch();
            }
        });
    }

    @Override
    public void onCheckedChanged(@NonNull CompoundButton compoundButton, boolean isChecked) {
        if (isChecked) {
            this.setRequestedOrientation(this.getCurrentOrientation());
            if (sensorService != null){
                sensorService.startSendingData();
            }
            if (this.settings.getKeepScreenAlive()){
                getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
            }
        } else {
            sensorService.stopSendingData();
            this.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED);
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        }
        active = isChecked;
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == SensorService.GEOLOCATION_PERMISSION_REQUEST) {
            boolean locationIsGranted = grantResults.length == 2 && (grantResults[0] == PackageManager.PERMISSION_GRANTED || grantResults[1] == PackageManager.PERMISSION_GRANTED);
            boolean fineLocationRequested = grantResults.length == 1;
            if (locationIsGranted || fineLocationRequested) {
                sensorService.rebindLocation();
            } else {
                // Geolocation permission is not granted
                for (AbstractSensorFragment fragment : this.sensorFragments){
                    if (fragment.getSensorType() == Parameters.GEOLOCATION){
                        fragment.deactivate();
                        break;
                    }
                }
            }
        } else if (requestCode == SensorService.BT_PERMISSION_REQUEST){
            boolean btIsGranted = grantResults.length == 1 && (grantResults[0] == PackageManager.PERMISSION_GRANTED);
            if (btIsGranted) {
                sensorService.rebindBluetooth();
            } else {
                Fragment btOverlay = getSupportFragmentManager().findFragmentByTag(BT_FRAGMENT);
                if (btOverlay != null) {
                    FragmentManager manager = getSupportFragmentManager();
                    manager.beginTransaction().remove(btOverlay).commit();
                    manager.popBackStack();
                }
                // Bluetooth permission is not granted
                for (AbstractSensorFragment fragment : this.sensorFragments){
                    if (fragment.getSensorType() == BT_SENSOR){
                        fragment.deactivate();
                        break;
                    }
                }
            }
        }
        else if(requestCode == SensorService.ACTIVITY_RECOGNITION_PERMISSION_REQUEST){
            boolean allowed = grantResults.length == 1 && (grantResults[0] == PackageManager.PERMISSION_GRANTED);
            if (!allowed){
                // Manifest.permission.ACTIVITY_RECOGNITION) needed for Q or up
                for (AbstractSensorFragment fragment : this.sensorFragments){
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N && Arrays.stream(SensorService.ACTIVITY_RECOGNITION_SENSORS).anyMatch(s -> s == fragment.getSensorType())) {
                        fragment.deactivate();
                    }
                }
            }
        }
    }
    @Override
    public void onActivityResult(final int requestCode, final int resultCode, final Intent resultData) {
        super.onActivityResult(requestCode, resultCode, resultData);
        if (requestCode == BT_REQUEST_CODE){
            if (resultCode != RESULT_OK)  {
                FragmentManager fm = getSupportFragmentManager();
                Fragment existing = fm.findFragmentByTag(BT_FRAGMENT);
                if (existing != null) {
                    FragmentTransaction transaction = fm.beginTransaction();
                    transaction.remove(existing);
                    transaction.commit();
                    fm.popBackStack();
                }
            }
        }
    }


    public void onStartMultiTouch(View view) {
        FragmentManager fm = getSupportFragmentManager();
        FragmentTransaction transaction = fm.beginTransaction();
        transaction.add(R.id.container, new MultiTouchFragment());
        transaction.addToBackStack(null);
        transaction.commit();
    }

    public void onStartBluetooth(View view) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
            FragmentManager fm = getSupportFragmentManager();
            Fragment existing = fm.findFragmentByTag(BT_FRAGMENT);
            if (existing == null) {
                FragmentTransaction transaction = fm.beginTransaction();
                ListFragment fragment = new ListFragment();
                transaction.add(R.id.container, fragment, BT_FRAGMENT);
                transaction.addToBackStack(null);
                transaction.commit();
                sensorService.checkForBluetoothPermission(this);
                sensorService.rebindBluetooth();
                fragment.setSensorService(sensorService);
            }
        }
    }

    public int getCurrentOrientation() {
        final Display display = this.getWindowManager().getDefaultDisplay();
        final int width, height;
        Point size = new Point();
        display.getSize(size);
        width = size.x;
        height = size.y;
        switch (display.getRotation()) {
            case Surface.ROTATION_90:
                if (width > height) {
                    return ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                }
            case Surface.ROTATION_180:
                if (height > width) {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                }
            case Surface.ROTATION_270:
                if (width > height) {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                }
            case Surface.ROTATION_0:
            default:
                if (height > width) {
                    return ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                }
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        unbindService(sensorServiceConnection);
    }

    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouch(View v, MotionEvent event) {
        if (this.sensorService.getIsSending()) {
            int width = v.getWidth();
            int height = v.getHeight();
            for (Measurement measurement : Measurement.measurements(event, width, height)) {
                this.sensorService.getDispatcher().dispatch(measurement);
            }
        }
        return false;
    }

    public void registerFragment(AbstractSensorFragment sensorFragment) {
        this.sensorFragments.add(sensorFragment);
    }
}
