package org.roto.data

import android.content.Context
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.booleanPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map

enum class ThemeOption { SYSTEM, LIGHT, DARK, FOREST, SUNSET, OCEAN, BLOSSOM, MIDNIGHT, SAND }
enum class ThemeMode { SYSTEM, LIGHT, DARK }

val Context.themePreferencesDataStore by preferencesDataStore(name = "theme_preferences")

class ThemePreferencesDataSource(private val context: Context) {

    private object Keys {
        val THEME_OPTION = stringPreferencesKey("theme_option")
        val THEME_MODE = stringPreferencesKey("theme_mode")
        val APPLY_SYSTEM_PADDING = booleanPreferencesKey("apply_system_padding")
    }

    val themeOptionFlow: Flow<ThemeOption> =
        context.themePreferencesDataStore.data.map { prefs ->
            prefs[Keys.THEME_OPTION]
                ?.let { runCatching { ThemeOption.valueOf(it) }.getOrNull() }
                ?: ThemeOption.SYSTEM
        }

    val themeModeFlow: Flow<ThemeMode> =
        context.themePreferencesDataStore.data.map { prefs ->
            prefs[Keys.THEME_MODE]
                ?.let { runCatching { ThemeMode.valueOf(it) }.getOrNull() }
                ?: ThemeMode.SYSTEM
        }

    val applySystemPaddingFlow: Flow<Boolean> =
        context.themePreferencesDataStore.data.map { prefs ->
            prefs[Keys.APPLY_SYSTEM_PADDING] ?: true
        }

    suspend fun saveTheme(option: ThemeOption) {
        context.themePreferencesDataStore.edit { prefs ->
            prefs[Keys.THEME_OPTION] = option.name
        }
    }

    suspend fun saveMode(mode: ThemeMode) {
        context.themePreferencesDataStore.edit { prefs ->
            prefs[Keys.THEME_MODE] = mode.name
        }
    }

    suspend fun saveApplySystemPadding(apply: Boolean) {
        context.themePreferencesDataStore.edit { prefs ->
            prefs[Keys.APPLY_SYSTEM_PADDING] = apply
        }
    }
}
