package org.residuum.alligator.pd;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.telephony.PhoneStateListener;
import android.telephony.TelephonyCallback;
import android.telephony.TelephonyManager;
import android.util.Log;

import org.puredata.android.io.AudioParameters;
import org.puredata.android.service.PdService;
import org.puredata.core.PdBase;
import org.puredata.core.PdReceiver;
import org.residuum.alligator.R;
import org.residuum.alligator.activities.IPdActivity;
import org.residuum.alligator.settings.AppSettings;
import org.residuum.alligator.settings.SampleInformation;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.core.content.ContextCompat;

import static android.telephony.TelephonyManager.INCLUDE_LOCATION_DATA_NONE;
import static org.residuum.alligator.pd.PdMessages.MSG_DELAYED;

public final class Binding {
    final IPdActivity activity;
    final Object lock = new Object();
    final Map<String, IPdReceiver> mReceivers = new HashMap<>();
    private final boolean mCanRecord;
    private final PdReceiver receiver = new PdReceiver() {

        @Override
        public void print(final String s) {
            Binding.this.activity.addDebugPrint(s);
        }

        @Override
        public void receiveBang(final String source) {
            for (final Map.Entry<String, IPdReceiver> entry : Binding.this.mReceivers.entrySet()) {
                if (source.equals(entry.getKey())) {
                    entry.getValue().addBang();
                }
            }
        }

        @Override
        public void receiveFloat(final String source, final float x) {
            for (final Map.Entry<String, IPdReceiver> entry : Binding.this.mReceivers.entrySet()) {
                if (source.equals(entry.getKey())) {
                    entry.getValue().addFloat(x);
                }
            }
        }

        @Override
        public void receiveSymbol(final String source, final String symbol) {
            if ("debug".equals(source)) {
                Log.d("Pd debug", symbol);
            }
            for (final Map.Entry<String, IPdReceiver> entry : Binding.this.mReceivers.entrySet()) {
                if (source.equals(entry.getKey())) {
                    entry.getValue().addSymbol(symbol);
                }
            }
        }

        @Override
        public void receiveList(final String source, final Object... args) {
            if ("debug".equals(source)) {
                Log.d("Pd debug", Arrays.stream(args).map(Object::toString)
                        .collect(Collectors.joining(", ")));
            }
            for (final Map.Entry<String, IPdReceiver> entry : Binding.this.mReceivers.entrySet()) {
                if (source.equals(entry.getKey())) {
                    entry.getValue().addList(args);
                }
            }
        }

        @Override
        public void receiveMessage(final String source, final String symbol, final Object... args) {
            if ("debug".equals(source)) {
                Log.d("Pd debug", symbol + ": " + Arrays.stream(args).map(Object::toString)
                        .collect(Collectors.joining(", ")));
            }
            for (final Map.Entry<String, IPdReceiver> entry : Binding.this.mReceivers.entrySet()) {
                if (source.equals(entry.getKey())) {
                    entry.getValue().addMessage(symbol, args);
                }
            }
        }
    };
    boolean isExternallyPaused;
    PdService pdService;
    @NonNull
    private final ServiceConnection pdConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(final ComponentName name, final IBinder service) {
            synchronized (Binding.this.lock) {
                Binding.this.pdService = ((PdService.PdBinder) service).getService();
                if (!Binding.this.pdService.isRunning()) {
                    final Thread thread = new Thread(() -> {
                        Looper.prepare();
                        Handler handler = new Handler();
                        handler.post(() ->
                                Binding.this.activity.runOnUiThread(Binding.this.activity::pdInitialized));
                        Looper.loop();
                    });
                    thread.setPriority(Thread.MIN_PRIORITY);
                    thread.start();
                }
            }
        }

        @Override
        public void onServiceDisconnected(final ComponentName name) {
            // this method will never be called?!
        }
    };
    private int patchHandle;

    public Binding(final IPdActivity activity, final boolean canRecord) {
        this.activity = activity;
        mCanRecord = canRecord;
        final Intent intent = new Intent(activity.getContext(), PdService.class);
        activity.bindService(intent, this.pdConnection, Context.BIND_AUTO_CREATE);
    }

    private boolean tryParseFloat(final String value) {
        try {
            Float.parseFloat(value);
            return true;
        } catch (final NumberFormatException e) {
            return false;
        }
    }

    public void loadPatch(final String pathPath) throws IOException {
        this.patchHandle = PdBase.openPatch(pathPath);
    }

    public void initPd() {
        // Configure the audio glue
        PdBase.setReceiver(this.receiver);
        PdBase.subscribe("debug");
    }

    public void clearPd() {
        this.isExternallyPaused = false;
        PdBase.closeAudio();
        PdBase.closePatch(this.patchHandle);
        this.activity.getContext().unbindService(this.pdConnection);
        this.activity.stopOtherListeners();
    }

    public void initTelephonyCallback(final boolean phoneStateCanBeRead) {
        if (Build.VERSION_CODES.S_V2 <= Build.VERSION.SDK_INT) {
            this.registerTelephonyCallback(phoneStateCanBeRead);
        } else {
            final TelephonyManager telephonyManager = (TelephonyManager) this.activity.getContext()
                    .getSystemService(Context.TELEPHONY_SERVICE);
            //noinspection deprecation
            telephonyManager.listen(new PhoneStateListener() {
                @Override
                public void onCallStateChanged(final int state, final String incomingNumber) {
                    if (null == pdService) {
                        return;
                    }
                    if (TelephonyManager.CALL_STATE_IDLE == state) {
                        if (Binding.this.isExternallyPaused) {
                            Binding.this.play(Binding.this.activity.getContext());
                        }
                        Binding.this.isExternallyPaused = false;
                    } else {
                        Binding.this.pause();
                        Binding.this.isExternallyPaused = true;
                    }
                }
            }, PhoneStateListener.LISTEN_CALL_STATE);
        }
    }

    private void registerTelephonyCallback(final boolean phoneStateCanBeRead) {
        if (!phoneStateCanBeRead) {
            return;
        }
        final TelephonyManager telephonyManager = (TelephonyManager) this.activity.getContext()
                .getSystemService(Context.TELEPHONY_SERVICE);
        final Executor mainExecutor = ContextCompat.getMainExecutor((Context) this.activity);
        if (Build.VERSION_CODES.TIRAMISU <= Build.VERSION.SDK_INT) {
            telephonyManager.registerTelephonyCallback(INCLUDE_LOCATION_DATA_NONE,
                    mainExecutor, new PhoneCallback());
        }
    }

    public void play(final Context context) {
        this.activity.startOtherServices();
        final Intent intent = new Intent(this.pdService, this.activity.getClass());
        intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
        this.pdService.startAudio(intent, R.drawable.ic_stat_name, context.getResources().getString(R.string.app_name),
                context.getResources().getString(R.string.app_is_running));
        this.sendToPd(PdMessages.METRO_ON, 1);
    }

    public void pause() {
        this.isExternallyPaused = false;
        this.pdService.stopAudio();
        this.activity.stopOtherListeners();
    }

    public void sendToPd(final String receiver, final Object value) {
        if (value instanceof Float) {
            PdBase.sendFloat(receiver, (float) value);
        } else if (value instanceof Integer) {
            PdBase.sendFloat(receiver, (1.0f * (Integer) value));
        } else if (value instanceof String) {
            PdBase.sendSymbol(receiver, (String) value);
        }
    }

    public void sendToPdOnNextBeat(final String receiver, final Object value) {
        PdBase.sendMessage(receiver, MSG_DELAYED, value);
    }

    private void sendBangToPd(final String receiver) {
        PdBase.sendBang(receiver);
    }

    public void addReceiver(final IPdReceiver receiver, final String receiverName) {
        this.mReceivers.put(receiverName, receiver);
        PdBase.subscribe(receiverName);
    }

    public void finishInit() throws IOException {
        PdBase.sendBang(PdMessages.LOAD_SAMPLES);
        AudioParameters.init(activity.getContext());
        final int sampleRate = AudioParameters.suggestSampleRate();
        this.pdService.initAudio(sampleRate, this.mCanRecord ? 1 : 0, 2, 10.0f);
    }

    public void loadSamples(@NonNull final List<? extends SampleInformation> sampleConfig) {
        this.sendToPd(PdMessages.RELATIVE_SPEED, 1);
        for (final SampleInformation sampleInformation : sampleConfig) {
            this.sendMessage(sampleInformation.getSampleGroup(), "load",
                    sampleInformation.getPosition(), sampleInformation.getPath(),
                    (float) AppSettings.DEFAULT_BPM / sampleInformation.getBpm());
        }
    }

    public void sendMessage(final String receiver, final String msg, final Object... value) {
        PdBase.sendMessage(receiver, msg, value);
    }

    public void sendMessageForNextBeat(final String receiver, final String msg, final Object... value) {
        final List<Object> dataToSend = new ArrayList<>();
        dataToSend.add(msg);
        if (Build.VERSION_CODES.UPSIDE_DOWN_CAKE <= Build.VERSION.SDK_INT) {
            dataToSend.addAll(Arrays.stream(value).toList());
        } else {
            Collections.addAll(dataToSend, value);
        }

        PdBase.sendMessage(receiver, MSG_DELAYED, dataToSend.toArray());
    }

    public void sendToPd(final String receiver, final Object... value) {
        PdBase.sendList(receiver, value);
    }

    public void resetSampleGroup(final String sendName) {
        this.sendMessage(sendName, PdMessages.RESET_SAMPLES, 1);
    }

    @RequiresApi(api = Build.VERSION_CODES.S)
    protected class PhoneCallback extends TelephonyCallback
            implements TelephonyCallback.CallStateListener {
        @Override
        public void onCallStateChanged(final int state) {
            if (null == Binding.this.pdService) {
                return;
            }
            if (TelephonyManager.CALL_STATE_IDLE == state) {
                if (Binding.this.isExternallyPaused) {
                    Binding.this.play(Binding.this.activity.getContext());
                }
                Binding.this.isExternallyPaused = false;
            } else {
                Binding.this.pause();
                Binding.this.isExternallyPaused = true;
            }
        }
    }
}