package org.residuum.alligator.fragments;

import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.SeekBar;
import android.widget.SpinnerAdapter;

import org.residuum.alligator.R;
import org.residuum.alligator.databinding.SamplegroupBinding;
import org.residuum.alligator.pd.Binding;
import org.residuum.alligator.pd.IPdBindable;
import org.residuum.alligator.pd.IPdReceiver;
import org.residuum.alligator.pd.PdMessages;
import org.residuum.alligator.settings.SampleInformation;
import org.residuum.alligator.utils.PitchAndPanning;

import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import static org.residuum.alligator.utils.Color.getColorFromAttr;

public class SampleGroupFragment extends Fragment implements IPdReceiver, IPdBindable {
    public static final String SEND_NAME = "sendName";
    final Handler resetBackgroundHandler = new Handler(Looper.getMainLooper());
    private final ArrayList<SampleInformation> mSampleInformations;
    private final PitchAndPanning mPitchAndPanning = new PitchAndPanning();
    protected String mSendName;
    protected Binding mPdBinding;
    protected SamplegroupBinding mViewBinding;
    private String mReceiverName;

    public SampleGroupFragment() {
        super(R.layout.samplegroup);
        this.mSampleInformations = new ArrayList<>();
    }

    public String getSendName() {
        return this.mSendName;
    }

    private void setSendName(final String sendName) {
        this.mSendName = sendName;
        this.mReceiverName = sendName + "-r";
        this.bindReceiver();
    }

    public void addSample(final SampleInformation information) {
        this.mSampleInformations.add(information);
    }

    public void clearSamples() {
        this.mSampleInformations.clear();
    }

    public void reloadSpinner(final Context context) {
        if (null == context || null == mViewBinding) {
            return;
        }
        setVisibility(!this.mSampleInformations.isEmpty());
        this.mSampleInformations.sort(new SampleInformation.SampleInformationComparator());
        final SpinnerAdapter adapter = new SampleInformationAdapter(context, android.R.layout.simple_spinner_dropdown_item, this.mSampleInformations);
        this.mViewBinding.sampleSelector.setAdapter(adapter);
    }

    private void setVisibility(boolean isVisible) {
        this.mViewBinding.container.setVisibility(isVisible ? View.VISIBLE : View.GONE);
    }

    @Override
    public View onCreateView(@NonNull final LayoutInflater inflater,
                             final ViewGroup container,
                             final Bundle savedInstanceState) {
        this.mViewBinding = SamplegroupBinding.inflate(inflater, container, false);
        final View view = this.mViewBinding.getRoot();
        final Bundle args = getArguments();
        assert null != args;
        setSendName(args.getString(SampleGroupFragment.SEND_NAME));
        this.mViewBinding.groupName.setText(String.format("%s %s", this.getString(R.string.sample_group), this.mSendName.substring(5)));
        this.mViewBinding.playButton.setOnCheckedChangeListener((buttonView, isChecked) -> {
            if (null != mPdBinding) {
                if (isChecked) {
                    mPdBinding.sendToPdOnNextBeat(mSendName, 1);
                    this.mViewBinding.container.setBackground(new ColorDrawable(getColorFromAttr(requireContext(), R.attr.topBackground)));
                } else {
                    mPdBinding.sendToPd(mSendName, 0);
                    this.mViewBinding.container.setBackground(new ColorDrawable(Color.TRANSPARENT));
                }
            }
        });
        this.mViewBinding.randomButton.setOnCheckedChangeListener((buttonView, isChecked) -> {
            if (null != mPdBinding) {
                this.mPdBinding.sendMessageForNextBeat(this.mSendName, PdMessages.RANDOM_SAMPLE, isChecked ? 1 : 0);
            }
            this.mViewBinding.sampleSelector.setEnabled(!isChecked);
        });
        this.mViewBinding.breakButton.setOnCheckedChangeListener((buttonView, isChecked) -> {
            if (null != mPdBinding) {
                this.mPdBinding.sendMessageForNextBeat(this.mSendName, PdMessages.BREAK, isChecked ? 1 : 0);
            }
        });
        if (null != mPdBinding) {
            this.mPdBinding.sendToPd(PdMessages.RANDOM_SEED, LocalTime.now().toSecondOfDay());
        }
        this.mPitchAndPanning.setPitchSlider(this.mViewBinding.pitchSlider);
        this.mViewBinding.pitchSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                final float pitch = (float) Math.pow(2, ((float) progress) / 1200);
                if (null != mPdBinding) {
                    SampleGroupFragment.this.mPdBinding.sendMessage(SampleGroupFragment.this.mSendName, PdMessages.PITCH, pitch);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mPitchAndPanning.setPanSlider(this.mViewBinding.panSlider);
        this.mViewBinding.panSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                if (null != mPdBinding) {
                    SampleGroupFragment.this.mPdBinding.sendMessage(SampleGroupFragment.this.mSendName, PdMessages.PANNING, ((float) progress) / 100);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mViewBinding.useOrientationPitch.setOnCheckedChangeListener((buttonView, isChecked) -> this.mPitchAndPanning.setUseOrientationForPitch(isChecked));
        this.mViewBinding.useOrientationPanning.setOnCheckedChangeListener((buttonView, isChecked) -> this.mPitchAndPanning.setUseOrientationForPanning(isChecked));
        this.mViewBinding.volumeSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                if (null != mPdBinding) {
                    SampleGroupFragment.this.mPdBinding.sendMessage(SampleGroupFragment.this.mSendName, PdMessages.VOLUME, progress);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mViewBinding.sampleSelector.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(final AdapterView<?> parent, final View viewInSelector, final int position, final long id) {
                if (null != mPdBinding && !SampleGroupFragment.this.mViewBinding.randomButton.isChecked()) {
                    SampleGroupFragment.this.mPdBinding.sendMessage(SampleGroupFragment.this.mSendName, PdMessages.SELECT_SAMPLE_INDEX, position);
                }
            }

            @Override
            public void onNothingSelected(final AdapterView<?> parent) {

            }
        });
        return view;
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        this.mViewBinding = null;
    }

    private void bindReceiver() {
        if (null != mReceiverName && null != mPdBinding) {
            this.mPdBinding.addReceiver(this, this.mReceiverName);
        }
    }

    public void setPdBinding(final Binding pdBinding) {
        this.mPdBinding = pdBinding;
        this.bindReceiver();
    }

    public void setOrientation(final float[] v, final int orientation) {
        this.mPitchAndPanning.setOrientation(v, orientation);
    }

    @Override
    public void setEnabled(final boolean enabled) {
        if (null == mViewBinding) {
            return;
        }
        this.mViewBinding.playButton.setEnabled(enabled);
    }

    public void addMessage(final String symbol, final Object[] args) {
        if (Objects.equals(symbol, PdMessages.RECEIVE_LOOP_START)) {
            this.flashLoop();
        }
    }

    private void flashLoop() {
        this.requireActivity().runOnUiThread(() -> {
            this.mViewBinding.container.setBackground(new ColorDrawable(Color.TRANSPARENT));
            resetBackgroundHandler.postDelayed(() -> {
                if (mViewBinding.playButton.isChecked()) {
                    mViewBinding.container.setBackground(new ColorDrawable(getColorFromAttr(this.requireContext(), R.attr.topBackground)));
                }
            }, 50);
        });
    }

    @Override
    public void addBang() {

    }

    @Override
    public void addFloat(final float x) {
        Objects.requireNonNull(this.mViewBinding.sampleSelector).setSelection((int) x);
    }

    @Override
    public void addList(final Object[] args) {
    }

    @Override
    public void addSymbol(final String symbol) {

    }

    private static class SampleInformationAdapter extends ArrayAdapter<SampleInformation> {
        public SampleInformationAdapter(@NonNull final Context context, final int resource, final List<SampleInformation> sampleInformations) {
            super(context, resource, sampleInformations);
        }
    }
}