package org.residuum.alligator.fragments;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CompoundButton;
import android.widget.SeekBar;

import org.puredata.android.io.AudioParameters;
import org.puredata.core.PdBase;
import org.residuum.alligator.R;
import org.residuum.alligator.activities.StartupActivity;
import org.residuum.alligator.databinding.RecordingBinding;
import org.residuum.alligator.pd.Binding;
import org.residuum.alligator.pd.IPdBindable;
import org.residuum.alligator.pd.IPdReceiver;
import org.residuum.alligator.pd.PdMessages;
import org.residuum.alligator.samplefiles.WaveFileExporter;
import org.residuum.alligator.utils.PitchAndPanning;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;

import static org.residuum.alligator.utils.Color.getColorFromAttr;

public class RecordingFragment extends Fragment implements IPdReceiver, IPdBindable {
    public static final String RECORDING_NO = "recording_number";
    final Handler resetBackgroundHandler = new Handler(Looper.getMainLooper());
    private final PitchAndPanning mPitchAndPanning = new PitchAndPanning();
    protected Binding mPdBinding;
    protected String mSendName;
    private RecordingBinding mViewBinding;
    private String mReceiverName;
    private int recordingNumber;

    public RecordingFragment() {
        super(R.layout.recording);
    }

    @Override
    public View onCreateView(@NonNull final LayoutInflater inflater,
                             final ViewGroup container,
                             final Bundle savedInstanceState) {
        this.mViewBinding = RecordingBinding.inflate(inflater, container, false);
        final Bundle args = getArguments();
        assert null != args;
        setRecordingNumber(args.getInt(RecordingFragment.RECORDING_NO));
        final View view = this.mViewBinding.getRoot();
        this.mViewBinding.recButton.setOnCheckedChangeListener(this.recButtonListener());
        this.mViewBinding.playButton.setOnCheckedChangeListener((buttonView, isChecked) -> {
            if (null != mPdBinding) {
                if (isChecked) {
                    this.mViewBinding.recButton.setChecked(false);
                    this.mPdBinding.sendToPdOnNextBeat(this.mSendName, 1);
                    this.mViewBinding.container.setBackground(new ColorDrawable(getColorFromAttr(requireContext(), R.attr.topBackground)));
                } else {
                    this.mPdBinding.sendToPd(this.mSendName, 0);
                    this.mViewBinding.container.setBackground(new ColorDrawable(Color.TRANSPARENT));
                }
            }
        });
        this.mViewBinding.breakButton.setOnCheckedChangeListener((buttonView, isChecked) -> {
            if (null != mPdBinding) {
                this.mPdBinding.sendMessageForNextBeat(this.mSendName, PdMessages.BREAK, isChecked ? 1 : 0);
            }
        });
        this.mPitchAndPanning.setPitchSlider(this.mViewBinding.pitchSlider);
        this.mViewBinding.pitchSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                final float pitch = (float) Math.pow(2, ((float) progress) / 1200);
                if (null != mPdBinding) {
                    RecordingFragment.this.mPdBinding.sendMessage(RecordingFragment.this.mSendName, PdMessages.PITCH, pitch);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mPitchAndPanning.setPanSlider(this.mViewBinding.panSlider);
        this.mViewBinding.panSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {

            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                if (null != mPdBinding) {
                    RecordingFragment.this.mPdBinding.sendMessage(RecordingFragment.this.mSendName, PdMessages.PANNING, ((float) progress) / 100);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mViewBinding.volumeSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                if (null != mPdBinding) {
                    RecordingFragment.this.mPdBinding.sendMessage(RecordingFragment.this.mSendName, PdMessages.VOLUME, progress);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });
        this.mViewBinding.useOrientationPitch.setOnCheckedChangeListener((buttonView, isChecked) -> this.mPitchAndPanning.setUseOrientationForPitch(isChecked));
        this.mViewBinding.useOrientationPanning.setOnCheckedChangeListener((buttonView, isChecked) -> this.mPitchAndPanning.setUseOrientationForPanning(isChecked));
        this.mViewBinding.exportButton.setOnClickListener(v -> saveWaveFile());
        return view;
    }

    private void setRecordingNumber(final int recordingNo) {
        this.mSendName = "record" + recordingNo;
        this.mReceiverName = this.mSendName + "-r";
        this.recordingNumber = recordingNo;
        mViewBinding.recordingName.setText(String.format("%s %d", getString(R.string.recording), recordingNo));
        this.bindReceiver();
    }

    private CompoundButton.OnCheckedChangeListener recButtonListener() {
        return (buttonView, isChecked) -> {
            if (null != mPdBinding) {
                if (isChecked) {
                    this.mViewBinding.playButton.setChecked(false);
                    this.mPdBinding.sendToPd(this.mSendName, 0);
                    this.mPdBinding.sendMessageForNextBeat(this.mSendName, PdMessages.RECORD, 1);
                } else {
                    this.mPdBinding.sendMessage(this.mSendName, PdMessages.RECORD, 0);
                }
            }
            this.setSampleControls(isChecked);
        };
    }

    private void saveWaveFile() {
        this.mPdBinding.sendMessage(this.mSendName, PdMessages.GET_SAMPLE_INFO);
    }

    private void bindReceiver() {
        if (null != mReceiverName && null != mPdBinding) {
            this.mPdBinding.addReceiver(this, this.mReceiverName);
        }
    }

    private void setSampleControls(final boolean isChecked) {
        this.mViewBinding.playButton.setEnabled(!isChecked);
        this.mViewBinding.exportButton.setEnabled(!isChecked);
        if (!isChecked) {
            this.mViewBinding.pitchSlider.setProgress(0);
            this.mViewBinding.panSlider.setProgress(50);
            this.mViewBinding.volumeSlider.setProgress(100);
            this.mViewBinding.playButton.setVisibility(View.VISIBLE);
            this.mViewBinding.panel1.setVisibility(View.VISIBLE);
        }
    }

    @Override
    public void addMessage(final String symbol, final Object[] args) {
        if (Objects.equals(symbol, PdMessages.RECEIVE_RECORDING_FORCE_STOP)) {
            // array is full, recording stopped
            this.mViewBinding.recButton.setOnCheckedChangeListener(null);
            this.mViewBinding.recButton.setChecked(false);
            this.mViewBinding.recButton.setOnCheckedChangeListener(this.recButtonListener());
            this.setSampleControls(false);
        }
        if (Objects.equals(symbol, PdMessages.RECEIVE_LOOP_START)) {
            this.flashLoop();
        }
    }

    @Override
    public void addBang() {

    }

    @Override
    public void addFloat(final float x) {
    }

    @Override
    public void addList(final Object[] args) {
        if (2 == args.length) {
            final float sampleLength = (Float) args[0];
            final float dollarZero = (Float) args[1];
            final String sampleArray = ((int) dollarZero) + PdMessages.ARRAY_SAMPLE_SUFFIX;
            final float[] audioData = new float[(int) sampleLength];
            if (0 == PdBase.readArray(audioData, 0, sampleArray, 0, (int) sampleLength)) {
                int suggestedSampleRate = PdBase.suggestSampleRate();
                if (0 > suggestedSampleRate) {
                    suggestedSampleRate = AudioParameters.suggestSampleRate();
                }
                final int sampleRate = suggestedSampleRate;
                final LocalDateTime now = LocalDateTime.now();
                ExecutorService executor = Executors.newSingleThreadExecutor();
                try {
                    final Handler handler = new Handler(Looper.getMainLooper());
                    executor.execute(() -> {
                        try {
                            @SuppressLint("UseRequireInsteadOfGet") String fileName = new WaveFileExporter(Objects.requireNonNull(getContext())).writeSampleRecording(audioData, sampleRate, recordingNumber, now);
                            handler.post(() -> fileExportSuccessful(fileName));
                        } catch (final IOException e) {
                            Log.e("Write", "recording.wav", e);
                            handler.post(this::fileExportFailed);
                        }
                    });
                } finally {
                    executor.shutdown();
                }
            }
        }
    }

    @Override
    public void addSymbol(final String symbol) {

    }

    private void fileExportSuccessful(String fileName) {
        Activity activity = getActivity();
        if (activity instanceof StartupActivity) {
            ((StartupActivity) activity).fileSaveSuccessful(fileName);
        }
    }

    private void fileExportFailed() {
        Activity activity = getActivity();
        if (activity instanceof StartupActivity) {
            ((StartupActivity) activity).fileSaveFailed();
        }

    }

    private void flashLoop() {
        this.requireActivity().runOnUiThread(() -> {
            this.mViewBinding.container.setBackground(new ColorDrawable(Color.TRANSPARENT));
            this.resetBackgroundHandler.postDelayed(() -> {
                if (this.mViewBinding.playButton.isChecked()) {
                    this.mViewBinding.container.setBackground(new ColorDrawable(getColorFromAttr(requireContext(), R.attr.topBackground)));
                }
            }, 50);
        });
    }

    @Override
    public void setPdBinding(final Binding pdBinding) {
        this.mPdBinding = pdBinding;
        this.bindReceiver();
    }

    @Override
    public void setOrientation(final float[] v, final int orientation) {
        this.mPitchAndPanning.setOrientation(v, orientation);
    }

    @Override
    public void setEnabled(final boolean enabled) {
        if (null == mViewBinding) {
            return;
        }
        this.mViewBinding.playButton.setEnabled(enabled);
        this.mViewBinding.recButton.setEnabled(enabled);
    }
}