package org.residuum.alligator.activities;

import android.annotation.SuppressLint;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ExpandableListAdapter;
import android.widget.ExpandableListView;
import android.widget.SpinnerAdapter;
import android.widget.TextView;

import com.google.android.material.snackbar.Snackbar;

import org.residuum.alligator.R;
import org.residuum.alligator.databinding.SettingsActivityBinding;
import org.residuum.alligator.samplefiles.FileContent;
import org.residuum.alligator.samplefiles.SampleContentCallback;
import org.residuum.alligator.samplefiles.SampleContentOperationResult;
import org.residuum.alligator.samplefiles.SampleContentOperator;
import org.residuum.alligator.samplefiles.SampleImporter;
import org.residuum.alligator.samplefiles.SamplePackExporter;
import org.residuum.alligator.samplefiles.SamplePackImporter;
import org.residuum.alligator.samplefiles.SamplePackResetter;
import org.residuum.alligator.settings.AppSettings;
import org.residuum.alligator.settings.SampleGroup;
import org.residuum.alligator.settings.SampleGroupAdapter;
import org.residuum.alligator.settings.SampleInformation;
import org.residuum.alligator.utils.FileUri;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatDelegate;
import androidx.core.app.NavUtils;

import static com.google.android.material.snackbar.BaseTransientBottomBar.LENGTH_LONG;
import static org.residuum.alligator.utils.Color.getColorFromAttr;


public class SettingsActivity extends ShowHideActivity {
    private static final int PICK_AUDIO_FILE = 2;
    private static final int PICK_SAMPLE_PACK = 3;
    protected AppSettings mAppSettings;
    private SettingsActivityBinding mViewBinding;
    private String groupName;
    private int position;
    private String newDirectory;
    private String newFilename;
    private Button deleteFile;
    private Button cancelFileEdit;
    private Button saveFile;
    private boolean fileOperationIsRunning;
    private AlertDialog fileSelector;
    private final View.OnClickListener loadSample = v -> {
        this.runOnUiThread(() -> this.disable(this.getString(R.string.picking_sample)));
        this.setDialogButtonState(false);
        Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.setType("audio/*");
        this.fileOperationIsRunning = true;
        this.startActivityForResult(intent, SettingsActivity.PICK_AUDIO_FILE);
    };
    private SampleGroupAdapter sampleGroupAdapter;

    @Override
    public void onActivityResult(final int requestCode, final int resultCode, final Intent resultData) {
        super.onActivityResult(requestCode, resultCode, resultData);
        if (RESULT_OK == resultCode && null != resultData) {
            ExecutorService executor = Executors.newSingleThreadExecutor();
            try {
                Handler handler = new Handler(Looper.getMainLooper());
                executor.execute(() -> {
                    SampleContentOperator importer = null;
                    switch (requestCode) {
                        case PICK_AUDIO_FILE:
                            importer = new SampleImporter(groupName, position, getApplicationContext(), new SampleContentCallback() {
                                @Override
                                public void onSuccess(SampleContentOperationResult result) {
                                    handler.post(() -> sampleAdded(result));
                                }

                                @Override
                                public void onProgress(String message) {
                                    handler.post(() -> progressMessage(message));
                                }

                                @Override
                                public void onError(String message) {
                                    handler.post(() -> addSampleFailed(message));
                                }
                            });
                            break;
                        case PICK_SAMPLE_PACK:
                            importer = new SamplePackImporter(getApplicationContext(), new SampleContentCallback() {
                                @Override
                                public void onSuccess(SampleContentOperationResult result) {
                                    handler.post(() -> fileOperationSuccessful(result));
                                }

                                @Override
                                public void onProgress(String message) {
                                    handler.post(() -> progressMessage(message));
                                }

                                @Override
                                public void onError(String message) {
                                    handler.post(() -> fileOperationFailed(message));
                                }
                            });
                            break;
                    }
                    if (importer != null) {
                        importer.operateOnData(new FileContent(resultData.getData(), getFilesDir(), getContentResolver(), getApplicationContext()));
                    }
                });
            } finally {
                executor.shutdown();
            }

        } else {
            setDialogButtonState(true);
        }
    }

    public void sampleAdded(SampleContentOperationResult result) {
        if (fileOperationIsRunning) {
            newDirectory = result.getDirectory();
            newFilename = result.getFileName();
            fileSelector.setTitle(result.getFileName());
        }
        setDialogButtonState(true);
        fileOperationIsRunning = false;
        runOnUiThread(this::enable);
    }

    private void progressMessage(String message) {
        mViewBinding.loadingText.setText(message);
    }

    public void addSampleFailed(String message) {
        showMessage(message);
        setDialogButtonState(true);
        fileOperationIsRunning = false;
        runOnUiThread(this::enable);
    }

    private void fileOperationSuccessful(SampleContentOperationResult result) {
        try {
            this.mAppSettings.loadSettingsAfterUpdate(this);
        } catch (IOException e) {
            Log.e("zip", "Load settings after unzip", e);
        }
        fileOperationIsRunning = false;
        mAppSettings.saveSettings(this);
        runOnUiThread(() -> {
            if (result.hasErrors()) {
                String message = getString(R.string.errored_files) + "\n\n" +
                        String.join("\n", result.getErroredFiles());
                showMessage(message);
                fileOperationIsRunning = false;
                runOnUiThread(this::enable);
            }
            sampleGroupAdapter.setSampleGroups(mAppSettings.getSampleGroups());
            setListHeight(this.mViewBinding.sampleList);
            this.enable();
        });
    }

    private void fileOperationFailed(String message) {
        showMessage(message);
        fileOperationIsRunning = false;
        runOnUiThread(this::enable);
    }

    private void setDialogButtonState(boolean state) {
        if (state) {
            fileSelector.show();
        } else {
            fileSelector.hide();
        }
        if (cancelFileEdit != null) {
            cancelFileEdit.setEnabled(state);
        }
        if (saveFile != null) {
            saveFile.setEnabled(state);
        }
        if (deleteFile != null) {
            deleteFile.setEnabled(state);
        }
    }

    private void enable() {
        mViewBinding.loadingScreen.setVisibility(View.GONE);
    }

    private void showMessage(String message) {
        AlertDialog dialog = new AlertDialog.Builder(this).setMessage(message)
                .setPositiveButton(R.string.ok, (d, i) -> d.cancel()).create();
        dialog.show();
        Button ok = dialog.getButton(DialogInterface.BUTTON_POSITIVE);
        if (null != ok) {
            ok.setTextColor(getColorFromAttr(this, R.attr.buttonPositive));
        }
    }

    private void setListHeight(ExpandableListView listView) {
        ExpandableListAdapter listAdapter = listView.getExpandableListAdapter();
        int totalHeight = 0;
        int desiredWidth = View.MeasureSpec.makeMeasureSpec(listView.getWidth(),
                View.MeasureSpec.UNSPECIFIED);
        for (int i = 0; i < listAdapter.getGroupCount(); i++) {
            View groupItem = listAdapter.getGroupView(i, true, null, listView);
            groupItem.measure(desiredWidth, View.MeasureSpec.UNSPECIFIED);
            totalHeight += groupItem.getMeasuredHeight();
            for (int j = 0; j < listAdapter.getChildrenCount(i); j++) {
                View listItem = listAdapter.getChildView(i, j, false, null,
                        listView);
                listItem.measure(desiredWidth, View.MeasureSpec.UNSPECIFIED);
                totalHeight += listItem.getMeasuredHeight();
            }
            totalHeight += listView.getDividerHeight() * listAdapter.getChildrenCount(i);
        }
        ViewGroup.LayoutParams params = listView.getLayoutParams();
        totalHeight += listView.getDividerHeight() * listAdapter.getGroupCount();
        if (totalHeight < 10)
            totalHeight = 200;
        params.height = totalHeight;
        listView.setLayoutParams(params);
        listView.requestLayout();
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        mViewBinding = SettingsActivityBinding.inflate(getLayoutInflater());
        this.mContentView = this.mViewBinding.mainLayout;
        setContentView(mViewBinding.getRoot());
        try {
            mAppSettings = AppSettings.loadSettings(this);
            AppCompatDelegate.setDefaultNightMode(mAppSettings.getColorSettings());
            bindColorSchemeSelector();
            bindSampleSelector();
            bindResetButton();
            bindSaveAndLoadButton();
            setListHeight(this.mViewBinding.sampleList);
            mViewBinding.currentExportFolder.setText(getString(R.string.download_folder, FileUri.getDirectoryPath("audio/wav"), FileUri.getDirectoryPath("application/zip")));
        } catch (IOException e) {
            runOnUiThread(this::enable);
            throw new RuntimeException(e);
        }
        createMenu();
    }

    private void bindColorSchemeSelector() {
        ArrayList<StringTranslation> colorSchemes = new ArrayList<>(Arrays.asList(new StringTranslation(AppSettings.COLOR_SCHEME_AUTO), new StringTranslation(AppSettings.COLOR_SCHEME_LIGHT), new StringTranslation(AppSettings.COLOR_SCHEME_DARK)));
        SpinnerAdapter adapter = new ColorSchemeAdapter(this, android.R.layout.simple_spinner_dropdown_item, colorSchemes);
        mViewBinding.colorSchemeSelector.setAdapter(adapter);
        mViewBinding.colorSchemeSelector.setSelection(colorSchemes.indexOf(new StringTranslation(mAppSettings.getColorScheme())));
        mViewBinding.colorSchemeSelector.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                if (null != SettingsActivity.this.mAppSettings) {
                    StringTranslation translation = colorSchemes.get(position);
                    mAppSettings.setColorScheme(translation.getValue());
                    mAppSettings.saveSettings(SettingsActivity.this);
                    AppCompatDelegate.setDefaultNightMode(mAppSettings.getColorSettings());
                }
            }

            @Override
            public void onNothingSelected(AdapterView<?> parent) {

            }
        });
    }

    private void bindSampleSelector() {
        ArrayList<SampleGroup> sampleGroups = mAppSettings.getSampleGroups();
        sampleGroupAdapter = new SampleGroupAdapter(this, sampleGroups);
        mViewBinding.sampleList.setAdapter(sampleGroupAdapter);
        for (int i = 0; sampleGroups.size() > i; i++) {
            mViewBinding.sampleList.expandGroup(i);
        }
        mViewBinding.sampleList.setOnGroupCollapseListener(groupPosition -> mViewBinding.sampleList.expandGroup(groupPosition));
        mViewBinding.sampleList.setOnChildClickListener((parent, view, groupPosition, childPosition, id) -> {
            SampleInformation sample = (SampleInformation) parent.getExpandableListAdapter().getChild(groupPosition, childPosition);
            this.position = sample.getPosition();
            groupName = sample.getSampleGroup();
            newFilename = null;
            newDirectory = null;

            View dialogView = buildDialogView(sample);
            EditText bpmEntry = dialogView.findViewById(R.id.bpm);
            AlertDialog.Builder builder = new AlertDialog.Builder(this);
            fileSelector = builder.setView(dialogView)
                    // Add action buttons
                    .setPositiveButton(R.string.save, (d, i) -> {
                        SampleInformation newSample;
                        if (null != this.newDirectory && null != this.newFilename) {
                            deleteFileIfOnlyUsage(sample);
                            newSample = new SampleInformation(groupName, position, newFilename, newDirectory,
                                    Integer.parseInt(String.valueOf(bpmEntry.getText())));
                        } else {
                            newSample = new SampleInformation(groupName, position, sample.getFileName(), sample.getDir(),
                                    Integer.parseInt(String.valueOf(bpmEntry.getText())));
                        }
                        mAppSettings.addOrUpdateSample(newSample);
                        mAppSettings.saveSettings(this);
                        sampleGroupAdapter.setSampleGroups(mAppSettings.getSampleGroups());
                        setListHeight(this.mViewBinding.sampleList);
                    })
                    .setNeutralButton(R.string.cancel, (d, i) -> {
                        if (null != this.newDirectory && null != this.newFilename) {
                            FileUtils.delete(newDirectory + File.separator + newFilename);
                        }
                    })
                    .setNegativeButton(R.string.delete, (d, i) -> {
                        if (null != this.newDirectory && null != this.newFilename) {
                            FileUtils.delete(newDirectory + File.separator + newFilename);
                        }
                        this.deleteFileIfOnlyUsage(sample);
                        mAppSettings.removeSample(groupName, position);
                        mAppSettings.saveSettings(this);
                        sampleGroupAdapter.setSampleGroups(mAppSettings.getSampleGroups());
                        setListHeight(this.mViewBinding.sampleList);
                    })
                    .setTitle(null == sample.getFileName() ? R.string.upload_file : R.string.change_file)
                    .create();
            fileSelector.show();
            deleteFile = fileSelector.getButton(DialogInterface.BUTTON_NEGATIVE);
            if (null != this.deleteFile) {
                deleteFile.setBackgroundColor(getColorFromAttr(this, R.attr.buttonWarning));
            }
            cancelFileEdit = fileSelector.getButton(DialogInterface.BUTTON_NEUTRAL);
            if (null != this.cancelFileEdit) {
                cancelFileEdit.setBackgroundColor(getColorFromAttr(this, R.attr.buttonNeutral));
            }
            saveFile = fileSelector.getButton(DialogInterface.BUTTON_POSITIVE);
            if (null != this.saveFile) {
                saveFile.setBackgroundColor(getColorFromAttr(this, R.attr.buttonPositive));
            }
            fileSelector.setOnCancelListener(dialog -> {
                fileOperationIsRunning = false;
                runOnUiThread(this::enable);
            });
            return false;
        });
        setListHeight(this.mViewBinding.sampleList);
    }

    private void deleteFileIfOnlyUsage(SampleInformation sample) {
        if (null != sample.getFileName() &&
                mAppSettings.getSamples().stream()
                        .filter(s -> s.getFileName().equals((sample.getFileName())))
                        .count() < 2) {
            FileUtils.delete(sample.getPath());
        }
    }

    @SuppressLint("ResourceType")
    private void bindResetButton() {
        mViewBinding.resetButton.setOnClickListener(v -> {
            AlertDialog confirmDialog = new AlertDialog.Builder(this)
                    .setTitle(R.string.settings_reset_alert)
                    .setMessage(R.string.settings_reset_alert_text)
                    .setPositiveButton(android.R.string.yes, (dialog, whichButton) -> {
                        dialog.dismiss();
                        SettingsActivity.this.resetSamples();
                    })
                    .setNegativeButton(android.R.string.no, null).show();
            Button ok = confirmDialog.getButton(DialogInterface.BUTTON_POSITIVE);
            if (null != ok) {
                ok.setTextColor(getColorFromAttr(this, R.attr.recordColor));
            }
            Button cancel = confirmDialog.getButton(DialogInterface.BUTTON_NEGATIVE);
            if (null != cancel) {
                cancel.setBackgroundColor(getColorFromAttr(this, R.attr.buttonNeutral));
            }
        });
    }

    private void resetSamples() {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        try {
            Handler handler = new Handler(Looper.getMainLooper());
            executor.execute(() -> {
                fileOperationIsRunning = true;
                runOnUiThread(() -> this.disable(this.getString(R.string.resetting_to_standard_samples)));
                SampleContentOperator resetter = new SamplePackResetter(getApplicationContext(),
                        this.mAppSettings, new SampleContentCallback() {
                    @Override
                    public void onSuccess(SampleContentOperationResult result) {
                        handler.post(() -> fileOperationSuccessful(result));
                    }

                    @Override
                    public void onProgress(String message) {
                        handler.post(() -> progressMessage(message));
                    }

                    @Override
                    public void onError(String message) {
                        handler.post(() -> fileOperationFailed(message));
                    }
                });
                resetter.operateOnData(new FileContent(null, getFilesDir(), getContentResolver(), getApplicationContext()));
            });
        } finally {
            executor.shutdown();
        }
    }

    private void bindSaveAndLoadButton() {
        mViewBinding.exportButton.setOnClickListener(v -> {
            this.fileOperationIsRunning = true;
            this.disable("Exporting sample pack.");
            ExecutorService executor = Executors.newSingleThreadExecutor();
            try {
                Handler handler = new Handler(Looper.getMainLooper());
                executor.execute(() -> {
                    SampleContentOperator exporter = new SamplePackExporter(getApplicationContext(),
                            this.mAppSettings.getSampleGroups(), new SampleContentCallback() {
                        @Override
                        public void onSuccess(SampleContentOperationResult result) {
                            handler.post(() -> samplePackExported(result));
                        }

                        @Override
                        public void onProgress(String message) {
                            handler.post(() -> progressMessage(message));
                        }

                        @Override
                        public void onError(String message) {
                            handler.post(() -> fileOperationFailed(message));
                        }
                    });
                    exporter.operateOnData(new FileContent(null, getFilesDir(), getContentResolver(), getApplicationContext()));
                });
            } finally {
                executor.shutdown();
            }
        });
        mViewBinding.importButton.setOnClickListener(v -> {
            this.runOnUiThread(() -> this.disable(this.getString(R.string.picking_sample_pack)));
            Intent intent = new Intent(Intent.ACTION_OPEN_DOCUMENT);
            intent.addCategory(Intent.CATEGORY_OPENABLE);
            intent.setType("application/zip");
            this.fileOperationIsRunning = true;
            this.startActivityForResult(intent, SettingsActivity.PICK_SAMPLE_PACK);
        });
    }

    @SuppressLint("NonConstantResourceId")
    private void createMenu() {
        mViewBinding.actionBar.inflateMenu(R.menu.main);
        Menu mainMenu = mViewBinding.actionBar.getMenu();
        mainMenu.removeItem(R.id.action_settings);

        mViewBinding.actionBar.setOnMenuItemClickListener(item -> {
            switch (item.getItemId()) {
//                case R.id.action_guide:
//                    startActivity(new Intent(SettingsActivity.this, HelpActivity.class));
//                    return true;
                case R.id.action_about:
                    startActivity(new Intent(this, AboutActivity.class));
                    return true;
                case R.id.action_bar:
                    NavUtils.navigateUpFromSameTask(this);
                    return true;
            }
            return true;
        });
    }

    @SuppressLint("SetTextI18n")
    private View buildDialogView(SampleInformation sample) {
        LayoutInflater inflater = getLayoutInflater();
        View dialogView = inflater.inflate(R.layout.sample_edit_dialog, null);
        if (null == sample.getFileName()) {
            ((TextView) dialogView.findViewById(R.id.sample_name)).setText(R.string.empty);
        } else {
            ((TextView) dialogView.findViewById(R.id.sample_name)).setText(sample.getFileName());
        }
        ((TextView) dialogView.findViewById(R.id.sample_group)).setText(sample.getSampleGroup());
        ((TextView) dialogView.findViewById(R.id.sample_position)).setText(Integer.toString(sample.getPosition()));
        ((TextView) dialogView.findViewById(R.id.bpm)).setText(Integer.toString(sample.getBpm()));
        dialogView.findViewById(R.id.file_select).setOnClickListener(loadSample);
        return dialogView;
    }

    private void disable(String message) {
        mViewBinding.loadingScreen.setVisibility(View.VISIBLE);
        mViewBinding.loadingText.setText(message);
    }

    private void samplePackExported(SampleContentOperationResult result) {
        fileOperationIsRunning = false;
        runOnUiThread(() -> {
            this.enable();
            Snackbar snackbar = Snackbar.make(this.mViewBinding.mainLayout, R.string.file_export_successful, LENGTH_LONG);
            TextView textView = snackbar.getView().findViewById(com.google.android.material.R.id.snackbar_text);
            textView.setText(String.format(textView.getText().toString(), result.getFileName()));
            snackbar.show();
        });
    }

    private static class ColorSchemeAdapter extends ArrayAdapter<StringTranslation> {

        public ColorSchemeAdapter(@NonNull Context context, int resource, List<StringTranslation> values) {
            super(context, resource, values);
        }
    }

    private class StringTranslation {

        private final String internalValue;
        private final String display;

        public StringTranslation(String value) {
            internalValue = value;
            if (Objects.equals(value, AppSettings.COLOR_SCHEME_DARK)) {
                display = getResources().getString(R.string.DARK);
            } else if (Objects.equals(value, AppSettings.COLOR_SCHEME_LIGHT)) {
                display = getResources().getString(R.string.LIGHT);
            } else {
                display = getResources().getString(R.string.AUTO);
            }
        }

        public String getValue() {
            return internalValue;
        }

        @Override
        public boolean equals(@Nullable Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof StringTranslation)) {
                return false;
            }
            return this.internalValue.equals(((StringTranslation) obj).internalValue);
        }

        @NonNull
        @Override
        public String toString() {
            return display;
        }
    }
}