package org.residuum.alligator.utils;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.ParcelFileDescriptor;
import android.provider.OpenableColumns;
import android.util.Log;

import org.puredata.core.utils.IoUtils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;


public enum FileUtils {
    ;
    public static final String TEMP_PREFIX = "temp_";

    public static File extractFullZipAndGetFile(final Context context, final int id, final String name,
                                                final boolean forceExtraction) throws IOException {
        final File dir = context.getFilesDir();
        if (!forceExtraction) {
            final File currentFile = new File(dir + File.separator + name);
            if (currentFile.exists()) {
                return currentFile;
            }
        }
        IoUtils.extractZipResource(context.getResources().openRawResource(id), dir, true);
        return new File(dir + File.separator + name);
    }

    public static void extractTempFilesFromZip(Uri zipUri, Context context, List<String> neededFiles){
        final File dir = context.getFilesDir();
        try(ParcelFileDescriptor descriptor = context.getContentResolver().openFileDescriptor(zipUri, "r")){
            try {
                assert descriptor != null;
                try (FileInputStream fileStream = new FileInputStream(descriptor.getFileDescriptor())) {
                    try (ZipInputStream zipStream = new ZipInputStream(fileStream)) {
                        ZipEntry zipEntry;
                        while ((zipEntry = zipStream.getNextEntry()) != null) {
                            if (neededFiles.contains(zipEntry.getName())) {
                                File file = new File(dir + File.separator + TEMP_PREFIX + zipEntry.getName());
                                try(FileOutputStream outputStream = new FileOutputStream(file)){
                                    int nRead;
                                    final byte[] data = new byte[16384];
                                    while (-1 != (nRead = zipStream.read(data, 0, data.length))) {
                                        outputStream.write(data, 0, nRead);
                                    }
                                    outputStream.flush();
                                }
                            }
                        }
                    }
                }
            } catch (IOException e) {
                Log.e("FileUtils", "zipInputStream", e);
            }
        } catch (IOException e) {
            Log.e("FileUtils", "fileStream", e);
        }
    }

    public static ZipContentAndSingleFile getSingleFileFromZip(Uri zipUri, Context context, String fileName){
        ArrayList<String> fileNames = new ArrayList<>();
        byte[] bytes = null;
        try(ParcelFileDescriptor descriptor = context.getContentResolver().openFileDescriptor(zipUri, "r")){
            try {
                assert descriptor != null;
                try (FileInputStream fileStream = new FileInputStream(descriptor.getFileDescriptor())) {
                    try (ZipInputStream zipStream = new ZipInputStream(fileStream)) {
                        ZipEntry zipEntry;
                        while ((zipEntry = zipStream.getNextEntry()) != null) {
                            fileNames.add(zipEntry.getName());
                            if (zipEntry.getName().equals(fileName)) {
                                try(ByteArrayOutputStream outputStream = new ByteArrayOutputStream(32)){
                                    int nRead;
                                    final byte[] data = new byte[16384];
                                    while (-1 != (nRead = zipStream.read(data, 0, data.length))) {
                                        outputStream.write(data, 0, nRead);
                                    }
                                    outputStream.flush();
                                    bytes = outputStream.toByteArray();
                                }
                                zipStream.read(bytes, 0, bytes.length);
                            }
                        }
                    }
                }
            } catch (IOException e) {
                Log.e("FileUtils", "zipInputStream", e);
                return null;
            }
        } catch (IOException e) {
            Log.e("FileUtils", "fileStream", e);
            return null;
        }
        return new ZipContentAndSingleFile(bytes, fileNames);
    }

    public static String getFileName(final Uri uri, final ContentResolver contentResolver) {
        try (final Cursor cursor = contentResolver.query(uri, null, null,
                null, null)) {
            assert null != cursor;
            final int nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME);
            cursor.moveToFirst();
            return cursor.getString(nameIndex);
        }
    }

    public static void delete(final String filePath) {
        final Path fileToDeletePath = Paths.get(filePath);
        try {
            Files.delete(fileToDeletePath);
        } catch (final IOException ignored) {
        }
    }

    public static List<File> getAllWavFiles(final Context context){
        final File dir = context.getFilesDir();
        List<File> wavFiles = new ArrayList<>();
        for (final File file : Objects.requireNonNull(dir.listFiles())) {
            if (file.isFile() && file.getName().endsWith(".wav")) {
                wavFiles.add(file);
            }
        }
        return wavFiles;
    }

    public static void deleteAllWavFiles(final Context context) {
        final File dir = context.getFilesDir();
        for (final File file : Objects.requireNonNull(dir.listFiles())) {
            if (file.isFile() && file.getName().endsWith(".wav")) {
                file.delete();
            }
        }
    }

    public static String readFile(String path, Charset encoding)
            throws IOException {
        byte[] encoded = Files.readAllBytes(Paths.get(path));
        return new String(encoded, encoding);
    }

    public static class ZipContentAndSingleFile{
        private final byte[] fileContent;
        private final List<String> filenames;

        public ZipContentAndSingleFile(byte[] fileContent, List<String> filenames) {
            this.fileContent = fileContent;
            this.filenames = filenames;
        }

        public List<String> getFilenames() {
            return filenames;
        }

        public byte[] getFileContent() {
            return fileContent;
        }
    }
}
