package org.residuum.alligator.settings;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.Expose;
import com.google.gson.reflect.TypeToken;

import org.jetbrains.annotations.Contract;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import androidx.annotation.NonNull;

public class SampleInformation {
    @Expose
    private final String sampleGroup;
    @Expose
    private final String fileName;
    private final String dir;
    @Expose
    private final int bpm;
    @Expose
    private int position;

    public SampleInformation(String sampleGroup, int position, String fileName, String dir, int bpm) {
        this.sampleGroup = sampleGroup;
        this.position = position;
        this.fileName = fileName;
        this.dir = dir;
        this.bpm = clampBpm(bpm);
    }

    private int clampBpm(int bpm) {
        if (bpm == 0) {
            return AppSettings.DEFAULT_BPM;
        }
        if (bpm < AppSettings.MIN_BPM) {
            return AppSettings.MIN_BPM;
        }
        return Math.min(bpm, AppSettings.MAX_BPM);
    }

    @NonNull
    @Contract(value = "_, _ -> new", pure = true)
    public static SampleInformation createEmptySample(String sampleGroup, int position) {
        return new SampleInformation(sampleGroup, position, null, null, AppSettings.DEFAULT_BPM);
    }

    public static List<SampleInformation> loadSampleInformation(File configFile) throws IOException {
        String dir = configFile.getAbsoluteFile().getParent();
        String configContent = FileUtils.readFile(configFile.getAbsolutePath(), StandardCharsets.UTF_8);
        return readConfigurationFromJson(configContent, dir);
    }

    public static List<SampleInformation> readConfigurationFromJson(String configContent, String dir) {
        Gson gson = new Gson();
        Type type = new TypeToken<SampleInformation[]>() {}.getType();
        final SampleInformation[] configValues = gson.fromJson(configContent, type);
        if (configValues == null){
            return null;
        }
        return Arrays.stream(configValues)
                .map(s -> new SampleInformation(s.sampleGroup, s.position, s.fileName, dir, s.bpm))
                .collect(Collectors.toList());
    }

    public String getSampleGroup() {
        return sampleGroup;
    }

    public int getPosition() {
        return position;
    }

    public void setPosition(int position) {
        this.position = position;
    }

    public int getBpm() {
        return bpm;
    }

    public String getFileName() {
        return fileName;
    }

    public String getPath() {
        return dir + File.separator + fileName;
    }

    public String getDir() {
        return dir;
    }

    @Override
    public int hashCode() {
        return Objects.hash(sampleGroup, position, fileName, dir, bpm);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (null == o || getClass() != o.getClass()) return false;
        SampleInformation that = (SampleInformation) o;
        return position == that.position && bpm == that.bpm && Objects.equals(sampleGroup, that.sampleGroup) && Objects.equals(fileName, that.fileName) && Objects.equals(dir, that.dir);
    }

    @NonNull
    @Override
    public String toString() {
        return fileName;
    }

    public static class SampleInformationComparator implements Comparator<SampleInformation>, Serializable {

        @Override
        public int compare(SampleInformation o1, SampleInformation o2) {
            return o1.getPosition() - o2.getPosition();
        }
    }

    public static byte[] SerializeSamplesForWriting(List<SampleInformation> samples) {
        SampleInformation[] filtered = samples.stream()
                .filter(s -> s.getFileName() != null && !s.getFileName().isEmpty())
                .toArray(SampleInformation[]::new);
        return new GsonBuilder()
                .excludeFieldsWithoutExposeAnnotation()
                .create().toJson(filtered)
                .getBytes(StandardCharsets.UTF_8);
    }
}
