package org.residuum.alligator.settings;

import android.content.Context;
import android.content.SharedPreferences;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import org.residuum.alligator.R;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import androidx.annotation.Nullable;
import androidx.preference.PreferenceManager;

import static androidx.appcompat.app.AppCompatDelegate.MODE_NIGHT_FOLLOW_SYSTEM;
import static androidx.appcompat.app.AppCompatDelegate.MODE_NIGHT_NO;
import static androidx.appcompat.app.AppCompatDelegate.MODE_NIGHT_YES;

public class AppSettings {
    public static final String COLOR_SCHEME_AUTO = "AUTO";
    public static final String COLOR_SCHEME_DARK = "NIGHT";
    public static final String COLOR_SCHEME_LIGHT = "DAY";
    public static final int MAX_SAMPLES_PER_GROUP = 10;
    public static final int DEFAULT_BPM = 120;
    public static final int MIN_BPM = 30;
    public static final int MAX_BPM = 300;
    public static final int GROUP_COUNT = 6;
    public static final String CONFIG_FILE = "config.json";
    private static final String BPM = "bpm";
    private static final String SAMPLES = "samples";
    private static final String COLOR_SCHEME = "color_scheme";
    private static final String GROUP_PREFIX = "group";
    private List<SampleInformation> samples;
    private float bpm;
    private String colorScheme;

    public static AppSettings loadSettings(final Context context) throws IOException {
        final SharedPreferences preferences =
                PreferenceManager.getDefaultSharedPreferences(context);
        final AppSettings appSettings = new AppSettings();
        appSettings.bpm = preferences.getFloat(AppSettings.BPM, AppSettings.DEFAULT_BPM);
        appSettings.colorScheme = preferences.getString(AppSettings.COLOR_SCHEME,
                AppSettings.COLOR_SCHEME_AUTO);
        List<SampleInformation> sampleConfig;
        final File sampleConfigFile = FileUtils.extractFullZipAndGetFile(context, R.raw.samples,
                CONFIG_FILE, false);
        if (!sampleConfigFile.exists()) {
            appSettings.reset(context);
            return appSettings;
        }
        sampleConfig = SampleInformation.loadSampleInformation(sampleConfigFile);
        final List<SampleInformation> storedSampleConfig =
                AppSettings.getSampleInformations(preferences);
        if (null != storedSampleConfig && !storedSampleConfig.isEmpty()) {
            sampleConfig = storedSampleConfig;
        }
        appSettings.samples = sampleConfig;
        return appSettings;
    }

    public void reset(final Context context) throws IOException {
        this.bpm = AppSettings.DEFAULT_BPM;
        final List<SampleInformation> sampleConfig;
        FileUtils.deleteAllWavFiles(context);
        final File sampleConfigFile = FileUtils.extractFullZipAndGetFile(context, R.raw.samples,
                CONFIG_FILE, true);
        sampleConfig = SampleInformation.loadSampleInformation(sampleConfigFile);
        this.samples = sampleConfig;
        saveSettings(context);
    }

    public void loadSettingsAfterUpdate(final Context context) throws IOException {
        final List<SampleInformation> sampleConfig;
        sampleConfig = SampleInformation.loadSampleInformation(
                new File(context.getFilesDir() + File.separator + CONFIG_FILE));
        this.samples = sampleConfig;
        saveSettings(context);
    }

    @Nullable
    private static List<SampleInformation> getSampleInformations(
            final SharedPreferences preferences) {
        final Gson gson = new Gson();
        final String json = preferences.getString(AppSettings.SAMPLES, null);
        final Type type = new TypeToken<ArrayList<SampleInformation>>() {
        }.getType();
        return gson.fromJson(json, type);
    }

    public void saveSettings(final Context context) {
        final SharedPreferences preferences =
                PreferenceManager.getDefaultSharedPreferences(context);
        final SharedPreferences.Editor editor = preferences.edit();
        editor.putFloat(AppSettings.BPM, this.bpm);
        editor.putString(AppSettings.COLOR_SCHEME, this.colorScheme);
        editor.putString(AppSettings.SAMPLES, new Gson().toJson(this.samples));
        editor.apply();
    }

    public float getBpm() {
        return this.bpm;
    }

    public void setBpm(final float value) {
        this.bpm = value;
    }

    public String getColorScheme() {
        return this.colorScheme;
    }

    public void setColorScheme(final String value) {
        this.colorScheme = value;
    }

    public List<SampleInformation> getSamples() {
        return samples;
    }

    public ArrayList<SampleGroup> getSampleGroups() {
        final ArrayList<SampleGroup> sampleGroups = new ArrayList<>();
        Map<String, List<SampleInformation>> groups = this.samples.stream()
                .collect(Collectors.groupingBy(SampleInformation::getSampleGroup));
        for (final Map.Entry<String, List<SampleInformation>> entry : groups.entrySet()) {
            final SampleGroup sampleGroup = new SampleGroup();
            sampleGroup.setName(entry.getKey());
            final List<SampleInformation> samplesInGroup = entry.getValue();
            samplesInGroup.sort(new SampleInformation.SampleInformationComparator());
            if (MAX_SAMPLES_PER_GROUP > samplesInGroup.size()) {
                samplesInGroup.add(SampleInformation.createEmptySample(entry.getKey(),
                        samplesInGroup.size()));
            }
            sampleGroup.setSamples(new ArrayList<>(samplesInGroup));
            sampleGroups.add(sampleGroup);
        }
        sampleGroups.sort(Comparator.comparing(SampleGroup::getName));
        if (sampleGroups.size() < GROUP_COUNT) {
            for (int i = 1; i <= GROUP_COUNT; i++) {
                String groupName = GROUP_PREFIX + i;

                if (sampleGroups.stream().noneMatch(g -> Objects.equals(g.getName(), groupName))) {
                    final SampleGroup sampleGroup = new SampleGroup();
                    sampleGroup.setName(groupName);
                    final List<SampleInformation> emptySamples = new ArrayList<>();
                    emptySamples.add(SampleInformation.createEmptySample(groupName,
                            emptySamples.size()));
                    sampleGroup.setSamples(new ArrayList<>(emptySamples));
                    sampleGroups.add(sampleGroup);
                }
            }
        }
        return sampleGroups;
    }

    public void removeSample(final String groupName, final int position) {
        final Optional<SampleInformation> existing = this.samples.stream()
                .filter(s -> Objects.equals(s.getSampleGroup(), groupName)
                        && s.getPosition() == position)
                .findFirst();
        if (existing.isPresent()) {
            this.samples.remove(existing.get());
            for (final SampleInformation other : this.samples) {
                if (Objects.equals(other.getSampleGroup(), groupName)
                        && other.getPosition() > position) {
                    other.setPosition(other.getPosition() - 1);
                }
            }
        }
    }

    public void addOrUpdateSample(final SampleInformation newSample) {
        final Optional<SampleInformation> existing = this.samples.stream()
                .filter(s -> Objects.equals(s.getSampleGroup(), newSample.getSampleGroup())
                        && s.getPosition() == newSample.getPosition())
                .findFirst();
        existing.ifPresent(this.samples::remove);
        this.samples.add(newSample);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.samples);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (null == o || this.getClass() != o.getClass()) return false;
        final AppSettings that = (AppSettings) o;
        return new HashSet<>(this.samples).containsAll(that.samples)
                && new HashSet<>(that.samples).containsAll(this.samples);
    }

    public int getColorSettings() {
        if (this.colorScheme.equals(AppSettings.COLOR_SCHEME_DARK)) {
            return MODE_NIGHT_YES;
        }
        if (this.colorScheme.equals(AppSettings.COLOR_SCHEME_LIGHT)) {
            return MODE_NIGHT_NO;
        }
        return MODE_NIGHT_FOLLOW_SYSTEM;
    }
}
