package org.residuum.alligator.pd;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.os.Binder;
import android.os.IBinder;

import org.puredata.android.io.AudioParameters;
import org.puredata.android.io.PdAudio;
import org.puredata.core.PdBase;

import java.io.IOException;

import androidx.core.app.NotificationCompat;
import androidx.preference.PreferenceManager;

public class PdService extends Service {

    private static final String TAG = "Alligator Service";
    private static final int NOTIFICATION_ID = 1312;
    private static final boolean abstractionsInstalled = false;
    private final PdBinder binder = new PdBinder();
    private boolean hasForeground;
    private volatile int sampleRate;
    private volatile int inputChannels;
    private volatile int outputChannels;
    private volatile float bufferSizeMillis;

    /**
     * @return the current audio buffer size in milliseconds (approximate value;
     * the exact value is a multiple of the Pure Data tick size (64 samples))
     */
    public float getBufferSizeMillis() {
        return this.bufferSizeMillis;
    }

    /**
     * @return number of input channels
     */
    public int getInputChannels() {
        return this.inputChannels;
    }

    /**
     * @return number of output channels
     */
    public int getOutputChannels() {
        return this.outputChannels;
    }

    /**
     * @return current sample rate
     */
    public int getSampleRate() {
        return this.sampleRate;
    }

    /**
     * Initialize Pure Data and audio thread
     *
     * @param samplerate sample rate
     * @param nic        number of input channels
     * @param noc        number of output channels
     * @param millis     audio buffer size in milliseconds; for Java audio only (Android 2.2 or earlier),
     *                   will be ignored by OpenSL components
     * @throws IOException if the audio parameters are not supported by the device
     */
    public synchronized void initAudio(int samplerate, int nic, int noc, float millis)
            throws IOException {
        this.stopForeground();
        final Resources res = this.getResources();
        final SharedPreferences prefs = PreferenceManager.getDefaultSharedPreferences(
                this.getApplicationContext());
        if (0 > samplerate) {
            samplerate = PdBase.suggestSampleRate();
            if (0 > samplerate) {
                samplerate = AudioParameters.suggestSampleRate();
            }
        }
        if (0 > nic) {
            nic = PdBase.suggestInputChannels();
            if (0 > nic) {
                nic = AudioParameters.suggestInputChannels();
            }
        }
        if (0 > noc) {
            noc = PdBase.suggestOutputChannels();
            if (0 > noc) {
                noc = AudioParameters.suggestOutputChannels();
            }
        }
        if (0 > millis) {
            millis = 50.0f;  // conservative choice
        }
        final int tpb = (int) (0.001f * millis * samplerate / PdBase.blockSize()) + 1;
        PdAudio.initAudio(samplerate, nic, noc, tpb, true);
        this.sampleRate = samplerate;
        this.inputChannels = nic;
        this.outputChannels = noc;
        this.bufferSizeMillis = millis;
    }

    private void stopForeground() {
        if (this.hasForeground) {
            this.stopForeground(true);
            this.hasForeground = false;
        }
    }

    /**
     * Start the audio thread without foreground privileges
     */
    public synchronized void startAudio() {
        PdAudio.startAudio(this);
    }

    /**
     * Start the audio thread with foreground privileges
     *
     * @param intent      intent to be triggered when the user selects notification of the service
     * @param icon        icon representing the notification
     * @param title       title of the notification
     * @param description description of the notification
     */
    public synchronized void startAudio(final Intent intent, final int icon, final String title,
                                        final String description) {
        this.startAudio(this.makeNotification(intent, icon, title, description));
    }

    /**
     * Start the audio thread with foreground privileges
     *
     * @param notification notification to display
     */
    public synchronized void startAudio(final Notification notification) {
        this.startForeground(notification);
        PdAudio.startAudio(this);
    }

    private Notification makeNotification(final Intent intent, final int icon, final String title,
                                          final String description) {
        this.initNotificationChannel();
        final PendingIntent pi = PendingIntent.getActivity(this,
                (int) System.currentTimeMillis(), intent,
                PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);
        return new NotificationCompat.Builder(this, PdService.TAG)
                .setSmallIcon(icon)
                .setContentTitle(title)
                .setTicker(title)
                .setContentText(description)
                .setOngoing(true)
                .setContentIntent(pi)
                .setWhen(System.currentTimeMillis())
                .setUsesChronometer(true)
                .build();
    }

    private void startForeground(final Notification notification) {
        this.stopForeground();
        this.startForeground(PdService.NOTIFICATION_ID, notification);
        this.hasForeground = true;
    }

    public void initNotificationChannel() {
        this.getApplicationContext();
        final NotificationManager notificationManager = (NotificationManager)
                this.getSystemService(NOTIFICATION_SERVICE);
        final NotificationChannel channel = new NotificationChannel(PdService.TAG, PdService.TAG,
                NotificationManager.IMPORTANCE_LOW);
        if (null != notificationManager) {
            notificationManager.createNotificationChannel(channel);
        }
    }

    /**
     * @return true if and only if the audio thread is running
     */
    public synchronized boolean isRunning() {
        return PdAudio.isRunning();
    }

    @Override
    public void onCreate() {
        super.onCreate();
        AudioParameters.init(this);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        this.release();
    }

    @Override
    public IBinder onBind(final Intent intent) {
        return this.binder;
    }

    @Override
    public boolean onUnbind(final Intent intent) {
        this.release();
        return false;
    }

    /**
     * Releases all resources
     */
    public synchronized void release() {
        this.stopAudio();
        PdAudio.release();
        PdBase.release();
    }

    /**
     * Stop the audio thread
     */
    public synchronized void stopAudio() {
        PdAudio.stopAudio();
        this.stopForeground();
    }

    public class PdBinder extends Binder {
        public PdService getService() {
            return PdService.this;
        }
    }
}
