package org.residuum.alligator.samplefiles;

import android.content.Context;
import android.net.Uri;
import android.os.Build;
import android.util.Log;

import com.linkedin.android.litr.TransformationListener;
import com.linkedin.android.litr.analytics.TrackTransformationInfo;
import com.linkedin.android.litr.exception.MediaSourceException;

import org.residuum.alligator.R;
import org.residuum.alligator.settings.AppSettings;
import org.residuum.alligator.settings.SampleGroup;
import org.residuum.alligator.settings.SampleInformation;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Stream;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static java.util.stream.Collectors.groupingBy;

public class SamplePackImporter implements SampleContentOperator {
    private final SampleContentCallback callback;
    private final Context context;
    private final TransformationListener transformerListener = new TransformationListener() {
        @Override
        public void onStarted(@NonNull String id) {
            SamplePackImporter.this.callback.onProgress(SamplePackImporter.this.context.getString(R.string.converting_file_starting));
        }

        @Override
        public void onProgress(@NonNull String id, float progress) {
            SamplePackImporter.this.callback.onProgress(String.format(SamplePackImporter.this.context.getString(R.string.converting_file_progress), (int)progress*100));
        }

        @Override
        public void onCompleted(@NonNull String id, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            final String tempFilename = id.substring(id.indexOf('/'));
            cleanupAfterTranscode(tempFilename);
        }

        @Override
        public void onCancelled(@NonNull String id, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            final String tempFilename = id.substring(id.indexOf('/'));
            missingFiles.add(tempFilename.substring(FileUtils.TEMP_PREFIX.length()));
            SamplePackImporter.this.callback.onError(SamplePackImporter.this.context.getString(R.string.copying_audio_file_aborted));
            cleanupAfterTranscode(tempFilename);
        }

        @Override
        public void onError(@NonNull String id, @Nullable Throwable cause, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            final String tempFilename = id.substring(id.indexOf('/'));
            missingFiles.add(tempFilename.substring(FileUtils.TEMP_PREFIX.length()));
            SamplePackImporter.this.callback.onProgress(SamplePackImporter.this.context.getString(R.string.copying_audio_file_failed));
            cleanupAfterTranscode(tempFilename);
        }
    };

    private static boolean filterForCorrectFormat(SampleInformation sample) {
        String sampleGroup = sample.getSampleGroup();
        try {
            final Integer groupNumber = Integer.valueOf(sampleGroup.substring(SampleGroup.GROUP_PREFIX.length()));
            return sampleGroup.startsWith(SampleGroup.GROUP_PREFIX) && groupNumber >= 1 && groupNumber <= SampleGroup.NO_OF_GROUPS;
        } catch (NumberFormatException ignored){
            return false;
        }
    }

    private void cleanupAfterTranscode(String tempFilename) {
        transcodedFiles++;
        final File dir = context.getFilesDir();
        final File tempFile =new File(dir + File.separator + tempFilename);
        tempFile.delete();
        if (transcodedFiles >= neededFiles.size()){
            finalizeSampleConfiguration(dir);
        }
    }

    private void finalizeSampleConfiguration(File dir) {
        try {
            writeSampleconfiguration(informations, missingFiles, dir);
            this.callback.onSuccess(new SampleContentOperationResult(missingFiles));
        } catch (IOException e) {
            this.callback.onError(context.getString(R.string.copying_audio_file_failed));
        }
    }

    private ArrayList<String> neededFiles;
    private ArrayList<String> missingFiles;
    private int transcodedFiles = 0;
    List<SampleInformation> informations;

    public SamplePackImporter(Context context, SampleContentCallback callback) {
        this.context = context;
        this.callback = callback;
    }

    @Override
    public void operateOnData(@NonNull FileContent content) {
        try {
            if (!Objects.equals(content.getContentResolver().getType(content.getUri()), "application/zip")) {
                this.callback.onError(this.context.getString(R.string.sample_import_nozip));
                return;
            }
            this.callback.onProgress(this.context.getString(R.string.searching_configuration));
            FileUtils.ZipContentAndSingleFile sampleConfigFile = FileUtils.getSingleFileFromZip(
                    content.getUri(), content.getContext(), AppSettings.CONFIG_FILE);
            assert sampleConfigFile != null;
            informations = extractSampleConfiguration(sampleConfigFile.getFileContent());
            if (informations == null) {
                this.callback.onError(this.context.getString(R.string.sample_import_noconfig));
                return;
            }
            List<String> filesInZip = sampleConfigFile.getFilenames();
            FileUtils.deleteAllWavFiles(content.getContext());
            missingFiles = new ArrayList<>();
            neededFiles = new ArrayList<>();
            for (SampleInformation sampleInformation : informations) {
                if (filesInZip.stream()
                        .noneMatch(s -> s.equals(sampleInformation.getFileName()))){
                    missingFiles.add(sampleInformation.getFileName());
                } else {
                    neededFiles.add(sampleInformation.getFileName());
                }
            }
            final File dir = context.getFilesDir();
            this.callback.onProgress(this.context.getString(R.string.extracting_zip));
            FileUtils.extractTempFilesFromZip(content.getUri(), content.getContext(), neededFiles);
            SampleTranscoder transcoder = new SampleTranscoder(transformerListener);
            for (String fileName : neededFiles) {
                final String outputFilename = getWaveFilename(fileName);
                File input = new File(dir + File.separator + FileUtils.TEMP_PREFIX + fileName);
                File output = new File(dir + File.separator + outputFilename);
                transcodeFile(transcoder, input, content.getContext(), output);
            }
        } catch (final Exception e) {
            Log.e("zip", "unzip", e);
            this.callback.onError(this.context.getString(R.string.sample_import_noconfig));
        }
    }

    @NonNull
    private static String getWaveFilename(String fileName) {
        final String namePortion = fileName.substring(0, fileName.lastIndexOf('.'));
        final String finalExtension = ".wav";
        return namePortion + finalExtension;
    }

    private static void writeSampleconfiguration(List<SampleInformation> informations, List<String> missingFiles, File dir) throws IOException {
        informations = cleanupConfiguration(informations, missingFiles);
        File file = new File(dir + File.separator + AppSettings.CONFIG_FILE);
        try(FileOutputStream outputStream = new FileOutputStream(file)){
            final byte[] bytes = SampleInformation.SerializeSamplesForWriting(informations);
            outputStream.write(bytes, 0, bytes.length);
            outputStream.flush();
        }
    }

    private void transcodeFile(SampleTranscoder transcoder, File input, Context context, File output) throws MediaSourceException {
        transcoder.transcodeFile(context, Uri.fromFile(input), input.getName(), output);
    }

    private static List<SampleInformation> cleanupConfiguration(List<SampleInformation> informations, List<String> missingFiles) {
        List<SampleInformation> cleanedSamples = new ArrayList<>();
        Map<String, List<SampleInformation>> groupedSamples = informations.stream().filter(s -> !missingFiles.contains(s.getFileName())).collect(groupingBy(SampleInformation::getSampleGroup));
        for (List<SampleInformation> sampleGroup : groupedSamples.values()) {
            int position = 0;
            for (SampleInformation information: sampleGroup) {
                cleanedSamples.add(new SampleInformation(information.getSampleGroup(), position, getWaveFilename(information.getFileName()), null, information.getBpm()));
                position++;
            }
        }
        return cleanedSamples;
    }

    private List<SampleInformation> extractSampleConfiguration(byte[] sampleConfigFile) {
        if (sampleConfigFile != null){
            String jsonContent = new String(sampleConfigFile, StandardCharsets.UTF_8);
            List<SampleInformation> samples = SampleInformation.readConfigurationFromJson(jsonContent, "");
            if (samples == null){
                return null;
            }
            final Stream<SampleInformation> filteredStream = samples.stream().filter(SamplePackImporter::filterForCorrectFormat);
            List<SampleInformation> filtered;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
                filtered = filteredStream.toList();
            } else {
                filtered  = Arrays.asList(filteredStream.toArray(SampleInformation[]::new));
            }
            if (filtered.isEmpty()){
                return null;
            }
            return filtered;
        }
        return null;
    }
}
