package org.residuum.alligator.samplefiles;

import android.content.Context;
import android.net.Uri;

import org.residuum.alligator.R;
import org.residuum.alligator.settings.AppSettings;
import org.residuum.alligator.settings.SampleGroup;
import org.residuum.alligator.settings.SampleInformation;
import org.residuum.alligator.utils.FileUri;
import org.residuum.alligator.utils.FileUtils;
import java.io.File;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import androidx.annotation.NonNull;

public class SamplePackExporter implements SampleContentOperator {
    private static final String SAMPLE_PACK_NAME = "Alligator_Bytes_sample_pack_%s.zip";
    private final DateTimeFormatter dateTimeFormat = DateTimeFormatter.ofPattern("yyyy-MM-dd--HH-mm-ss");
    private final Context context;
    private final List<SampleInformation> samples;
    private final SampleContentCallback callback;

    public SamplePackExporter(Context context, ArrayList<SampleGroup> sampleGroups, SampleContentCallback callback) {
        this.context = context;
        this.samples = sampleGroups.stream().flatMap(group -> group.getSamples().stream()).collect(Collectors.toList());
        this.callback = callback;
    }

    @Override
    public void operateOnData(@NonNull FileContent content) {
        final LocalDateTime exportDateTime = LocalDateTime.now();
        final String zipFileName = String.format(SAMPLE_PACK_NAME, exportDateTime.format(this.dateTimeFormat));
        final Uri audioCollection = content.getExportFolder();
        Uri fileUri = FileUri.getUriForFile(audioCollection, zipFileName, this.context, "application/zip");
        if (fileUri == null){
            this.callback.onError(context.getString(R.string.error_writing_zip));
        }
        try {
            try(OutputStream fileStream = content.getContentResolver().openOutputStream(fileUri)){
                try(ZipOutputStream zipStream = new ZipOutputStream(fileStream)) {
                    for (File wavFile: FileUtils.getAllWavFiles(content.getContext())) {
                        this.callback.onProgress(String.format(context.getString(R.string.writing_to_zip), wavFile.getName()));
                        ZipEntry zipEntry = new ZipEntry(wavFile.getName());
                        zipStream.putNextEntry(zipEntry);
                        byte[] bytes = Files.readAllBytes(Paths.get(wavFile.getAbsolutePath()));
                        zipStream.write(bytes, 0, bytes.length);
                        zipStream.closeEntry();
                    }
                    this.callback.onProgress(String.format(context.getString(R.string.writing_to_zip), AppSettings.CONFIG_FILE));
                    ZipEntry zipEntry = new ZipEntry(AppSettings.CONFIG_FILE);
                    zipStream.putNextEntry(zipEntry);
                    final byte[] bytes = SampleInformation.SerializeSamplesForWriting(this.samples);
                    zipStream.write(bytes, 0, bytes.length);
                    zipStream.closeEntry();
                    this.callback.onSuccess(new SampleContentOperationResult(null, zipFileName, null, 0));
                } catch (IOException e) {
                    this.callback.onError(context.getString(R.string.error_writing_zip));
                }
            }
        } catch (IOException e) {
            this.callback.onError(context.getString(R.string.error_writing_zip));
        }
    }


}
