package org.residuum.alligator.samplefiles;

import android.content.Context;

import com.linkedin.android.litr.TransformationListener;
import com.linkedin.android.litr.analytics.TrackTransformationInfo;
import com.linkedin.android.litr.exception.MediaSourceException;

import org.residuum.alligator.R;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

public class SampleImporter implements SampleContentOperator {
    private final String groupName;
    private final int position;
    private final Context context;
    private final SampleContentCallback callback;
    private String dir;
    private String fileName;
    private final TransformationListener transformerListener = new TransformationListener() {
        @Override
        public void onStarted(@NonNull String id) {
            SampleImporter.this.callback.onProgress(SampleImporter.this.context.getString(R.string.converting_file_starting));
        }

        @Override
        public void onProgress(@NonNull String id, float progress) {
            SampleImporter.this.callback.onProgress(String.format(SampleImporter.this.context.getString(R.string.converting_file_progress), (int)progress*100));
        }

        @Override
        public void onCompleted(@NonNull String id, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            SampleImporter.this.callback.onSuccess(new SampleContentOperationResult(SampleImporter.this.dir, SampleImporter.this.fileName,
                    SampleImporter.this.groupName, SampleImporter.this.position));
        }

        @Override
        public void onCancelled(@NonNull String id, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            SampleImporter.this.callback.onError(SampleImporter.this.context.getString(R.string.copying_audio_file_aborted));
        }

        @Override
        public void onError(@NonNull String id, @Nullable Throwable cause, @Nullable List<TrackTransformationInfo> trackTransformationInfos) {
            SampleImporter.this.callback.onError(SampleImporter.this.context.getString(R.string.copying_audio_file_failed));
        }
    };

    public SampleImporter(final @NonNull String groupName, final int position, Context context, final @NonNull SampleContentCallback callback) {
        this.groupName = groupName;
        this.context = context;
        this.position = position;
        this.callback = callback;
    }

    public void operateOnData(@NonNull final FileContent content) {
        try {
            callback.onProgress(context.getString(R.string.loading_file));
            fileName = FileUtils.getFileName(content.getUri(), content.getContentResolver());
            final String namePortion = fileName.substring(0, fileName.lastIndexOf('.'));
            final String finalExtension = ".wav";
            dir = content.getFilesDir().getAbsoluteFile().getPath();
            fileName = namePortion + finalExtension;
            File outputFile = new File(content.getFilesDir(), namePortion + finalExtension);
            if (outputFile.exists()) {
                int counter = 0;
                while (outputFile.exists()) {
                    fileName = namePortion + "_" + counter + finalExtension;
                    outputFile = new File(content.getFilesDir(), fileName);
                    counter++;
                }
            }
            transcodeFile(content, outputFile);
        } catch (final Exception e) {
            SampleImporter.this.callback.onError(SampleImporter.this.context.getString(R.string.copying_audio_file_failed));
        }
    }

    private void transcodeFile(@NonNull FileContent content, File outputFile) throws MediaSourceException {
        new SampleTranscoder(transformerListener).transcodeFile(content.getContext(), content.getUri(), "", outputFile);
    }
}