package org.residuum.alligator.activities;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.graphics.Point;
import android.graphics.drawable.ColorDrawable;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.provider.MediaStore;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.Display;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.Surface;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.CompoundButton;
import android.widget.SeekBar;
import android.widget.TextView;

import com.google.android.material.snackbar.Snackbar;

import org.jetbrains.annotations.Contract;
import org.puredata.android.io.AudioParameters;
import org.puredata.core.PdBase;
import org.residuum.alligator.R;
import org.residuum.alligator.databinding.StartupActivityBinding;
import org.residuum.alligator.fragments.RecordingFragment;
import org.residuum.alligator.fragments.SampleGroupFragment;
import org.residuum.alligator.pd.Binding;
import org.residuum.alligator.pd.IPdReceiver;
import org.residuum.alligator.pd.PdMessages;
import org.residuum.alligator.samplefiles.WaveFileExporter;
import org.residuum.alligator.settings.AppSettings;
import org.residuum.alligator.settings.SampleGroup;
import org.residuum.alligator.settings.SampleInformation;
import org.residuum.alligator.utils.FileUtils;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatDelegate;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;

import static android.view.MotionEvent.ACTION_UP;
import static com.google.android.material.snackbar.BaseTransientBottomBar.LENGTH_LONG;
import static com.google.android.material.snackbar.BaseTransientBottomBar.LENGTH_SHORT;
import static org.residuum.alligator.utils.Color.getColorFromAttr;

/**
 * An example full-screen activity that shows and hides the system UI (i.e.
 * status bar and navigation/system bar) with user interaction.
 */
public class StartupActivity extends ShowHideActivity implements IPdActivity, IPdReceiver, SensorEventListener {
    /**
     * Some older devices needs a small delay between UI widget updates
     * and a change of the status and navigation bar.
     */
    private static final String RECORDING_FRAGMENT_PREFIX = "recording-";
    private static final String SAMPLEGROUP_FRAGMENT_PREFIX = "samplegroup-";
    private static final String GENERAL_RECEIVER = "general-r";
    protected final Collection<SampleGroupFragment> mSampleFragments = new ArrayList<>(6);
    protected final Collection<RecordingFragment> mRecordingFragments = new ArrayList<>(4);
    private final Handler resetBackgroundHandler = new Handler(Looper.getMainLooper());
    private final ArrayList<Long> mBeatLengthInMs = new ArrayList<>(4);

    protected boolean ongoingRecording;
    protected StartupActivityBinding mViewBinding;
    protected Binding mPdBinding;
    final ActivityResultLauncher<String[]> permissionLauncher = this.registerForActivityResult(new ActivityResultContracts.RequestMultiplePermissions(),
            isGranted -> this.startAndBindPd(Boolean.TRUE.equals(isGranted.get(Manifest.permission.RECORD_AUDIO))));
    protected LocalDateTime recordDateTime;
    private final CompoundButton.OnCheckedChangeListener mSessionRecordClicked = (compoundButton, checked) -> {
        StartupActivity.this.ongoingRecording = checked;
        if (checked) {
            StartupActivity.this.mPdBinding.sendToPd(PdMessages.RECORD_SESSION, 1);
            StartupActivity.this.recordDateTime = LocalDateTime.now();
            StartupActivity.this.mPdBinding.play(StartupActivity.this);
        } else {
            StartupActivity.this.mPdBinding.sendToPd(PdMessages.RECORD_SESSION, 0);
        }
    };
    private int mOrientation = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
    private AppSettings mAppSettings;
    private boolean isScrolled;
    private SensorManager mSensorManager;
    private Sensor mMagneticFieldSensor;
    private Sensor mGravitySensor;
    private float[] mMagneticFieldValues;
    private float[] mGravityValues;
    private Sensor mOrientationSensor;
    private long mLatestStep;

    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        this.mViewBinding = StartupActivityBinding.inflate(this.getLayoutInflater());
        this.mContentView = this.mViewBinding.fullscreenContent;
        this.mLatestStep = System.currentTimeMillis();
        this.setContentView(this.mViewBinding.getRoot());
        this.createSampleGroupFragments();
        this.createRecordingFragments();
        this.createMenu();
        try {
            this.mAppSettings = AppSettings.loadSettings(this);
            AppCompatDelegate.setDefaultNightMode(this.mAppSettings.getColorSettings());
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }

        final boolean hasNecessaryPermissions = this.tryGettingPermissions();
        if (hasNecessaryPermissions) {
            this.startAndBindPd(true);
        }
        this.initBpm(this.mAppSettings.getBpm());
    }

    private void createSampleGroupFragments() {
        final FragmentManager manager = this.getSupportFragmentManager();
        final Fragment exSg1 = manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 1);
        if (null != exSg1) {
            manager.beginTransaction()
                    .remove(exSg1)
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 2)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 3)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 4)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 5)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 6)))
                    .commitNow();
        }
        final SampleGroupFragment sampleGroup1 = StartupActivity.getSampleGroupFragment("group1");
        final SampleGroupFragment sampleGroup2 = StartupActivity.getSampleGroupFragment("group2");
        final SampleGroupFragment sampleGroup3 = StartupActivity.getSampleGroupFragment("group3");
        final SampleGroupFragment sampleGroup4 = StartupActivity.getSampleGroupFragment("group4");
        final SampleGroupFragment sampleGroup5 = StartupActivity.getSampleGroupFragment("group5");
        final SampleGroupFragment sampleGroup6 = StartupActivity.getSampleGroupFragment("group6");
        this.mSampleFragments.add(sampleGroup1);
        this.mSampleFragments.add(sampleGroup2);
        this.mSampleFragments.add(sampleGroup3);
        this.mSampleFragments.add(sampleGroup4);
        this.mSampleFragments.add(sampleGroup5);
        this.mSampleFragments.add(sampleGroup6);
        manager.beginTransaction()
                .add(R.id.group_fragments, sampleGroup1, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 1)
                .add(R.id.group_fragments, sampleGroup2, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 2)
                .add(R.id.group_fragments, sampleGroup3, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 3)
                .add(R.id.group_fragments, sampleGroup4, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 4)
                .add(R.id.group_fragments, sampleGroup5, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 5)
                .add(R.id.group_fragments, sampleGroup6, StartupActivity.SAMPLEGROUP_FRAGMENT_PREFIX + 6)
                .setReorderingAllowed(true)
                .commit();
    }

    private void createRecordingFragments() {
        final FragmentManager manager = this.getSupportFragmentManager();
        final Fragment exRec1 = manager.findFragmentByTag(StartupActivity.RECORDING_FRAGMENT_PREFIX + 1);
        if (null != exRec1) {
            manager.beginTransaction()
                    .remove(exRec1)
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.RECORDING_FRAGMENT_PREFIX + 2)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.RECORDING_FRAGMENT_PREFIX + 3)))
                    .remove(Objects.requireNonNull(manager.findFragmentByTag(StartupActivity.RECORDING_FRAGMENT_PREFIX + 4)))
                    .commitNow();
        }
        final RecordingFragment rec1 = StartupActivity.getRecordingFragment(1);
        final RecordingFragment rec2 = StartupActivity.getRecordingFragment(2);
        final RecordingFragment rec3 = StartupActivity.getRecordingFragment(3);
        final RecordingFragment rec4 = StartupActivity.getRecordingFragment(4);
        this.mRecordingFragments.add(rec1);
        this.mRecordingFragments.add(rec2);
        this.mRecordingFragments.add(rec3);
        this.mRecordingFragments.add(rec4);
        manager.beginTransaction()
                .add(R.id.recording_fragments, rec1, StartupActivity.RECORDING_FRAGMENT_PREFIX + 1)
                .add(R.id.recording_fragments, rec2, StartupActivity.RECORDING_FRAGMENT_PREFIX + 2)
                .add(R.id.recording_fragments, rec3, StartupActivity.RECORDING_FRAGMENT_PREFIX + 3)
                .add(R.id.recording_fragments, rec4, StartupActivity.RECORDING_FRAGMENT_PREFIX + 4)
                .setReorderingAllowed(true)
                .commit();
        this.mViewBinding.recordingFragments.setVisibility(View.GONE);
    }

    @SuppressLint("NonConstantResourceId")
    private void createMenu() {
        this.mViewBinding.actionBar.inflateMenu(R.menu.main);
        final Menu mainMenu = this.mViewBinding.actionBar.getMenu();
        mainMenu.removeItem(R.id.action_bar);
        this.mViewBinding.actionBar.setOnMenuItemClickListener(item -> {
            switch (item.getItemId()) {
                case R.id.action_settings:
                    this.startActivity(new Intent(this, SettingsActivity.class));
                    break;
//                case R.id.action_guide:
//                    startActivity(new Intent(StartupActivity.this, HelpActivity.class));
//                    return true;
                case R.id.action_about:
                    this.startActivity(new Intent(this, AboutActivity.class));
                    return true;
            }
            return true;
        });
    }

    private boolean tryGettingPermissions() {
        final ArrayList<String> necessaryPermissions = new ArrayList<>();
        if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(getContext(), Manifest.permission.RECORD_AUDIO)) {
            necessaryPermissions.add(Manifest.permission.RECORD_AUDIO);
        }
        if (Build.VERSION_CODES.TIRAMISU <= Build.VERSION.SDK_INT) {
            if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(getContext(), Manifest.permission.POST_NOTIFICATIONS)) {
                necessaryPermissions.add(Manifest.permission.POST_NOTIFICATIONS);
            }
        }
        if (Build.VERSION_CODES.S <= Build.VERSION.SDK_INT) {
            if (PackageManager.PERMISSION_GRANTED != ContextCompat.checkSelfPermission(getContext(), Manifest.permission.READ_PHONE_STATE)) {
                necessaryPermissions.add(Manifest.permission.READ_PHONE_STATE);
            }
        }
        if (!necessaryPermissions.isEmpty()) {
            this.permissionLauncher.launch(necessaryPermissions.toArray(new String[0]));
        }
        return !necessaryPermissions.contains(Manifest.permission.RECORD_AUDIO);
    }

    private void startAndBindPd(final boolean canRecord) {
        if (canRecord) {
            this.mViewBinding.recordingFragments.setVisibility(View.VISIBLE);
        } else {
            this.mViewBinding.recordingFragments.setVisibility(View.GONE);
        }
        this.bindToPd(canRecord);
        if (Build.VERSION_CODES.S_V2 <= Build.VERSION.SDK_INT) {
            final boolean hasPermission = PackageManager.PERMISSION_GRANTED == ContextCompat.checkSelfPermission(getContext(), Manifest.permission.READ_PHONE_STATE);
            if (hasPermission) {
                this.mPdBinding.initTelephonyCallback(true);
            }
        } else {
            this.mPdBinding.initTelephonyCallback(true);
        }
    }


    @NonNull
    private static SampleGroupFragment getSampleGroupFragment(final String groupName) {
        final SampleGroupFragment fragment = new SampleGroupFragment();
        final Bundle bundle = new Bundle();
        bundle.putString(SampleGroupFragment.SEND_NAME, groupName);
        fragment.setArguments(bundle);
        return fragment;
    }

    @NonNull
    private static RecordingFragment getRecordingFragment(final int recordingNo) {
        final RecordingFragment fragment = new RecordingFragment();
        final Bundle bundle = new Bundle();
        bundle.putInt(RecordingFragment.RECORDING_NO, recordingNo);
        fragment.setArguments(bundle);
        return fragment;
    }

    private void bindToPd(final boolean canRecord) {
        this.mPdBinding = new Binding(this, canRecord);
    }

    @Override
    protected void onResume() {
        super.onResume();

        final AppSettings currentSettings = this.mAppSettings;

        try {
            this.mAppSettings = AppSettings.loadSettings(this);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }
        if (!this.mAppSettings.equals(currentSettings)) {
            this.mViewBinding.startStopButton.setOnCheckedChangeListener(null);
            this.mViewBinding.startStopButton.setChecked(false);
            this.mViewBinding.startStopButton.setOnCheckedChangeListener(this.mStartStopClicked);
            if (null != mViewBinding.startStopButton2) {
                this.mViewBinding.startStopButton2.setChecked(false);
                this.mViewBinding.startStopButton2.setOnCheckedChangeListener(null);
                this.mViewBinding.startStopButton2.setOnCheckedChangeListener(this.mStartStopClicked);
            }
            this.mViewBinding.sessionRecordButton.setOnCheckedChangeListener(null);
            this.mViewBinding.sessionRecordButton.setChecked(false);
            this.mViewBinding.sessionRecordButton.setOnCheckedChangeListener(this.mSessionRecordClicked);
            this.mPdBinding.pause();
            this.updateSampleGroups(currentSettings.getSampleGroups(), this.mAppSettings.getSampleGroups());
        }
    }

    private void updateSampleGroups(final Collection<? extends SampleGroup> oldSampleGroups, final Collection<? extends SampleGroup> newSampleGroups) {
        final Thread thread = new Thread(() -> {
            Looper.prepare();

            Handler handler = new Handler();
            handler.post(() -> {
                this.runOnUiThread(() -> this.disable(this.getString(R.string.samples_loading)));
                for (final SampleGroupFragment groupFragment : this.mSampleFragments) {
                    Collection<SampleInformation> oldSamples = oldSampleGroups.stream().filter(sg -> sg.getName()
                            .equals(groupFragment.getSendName())).findFirst().get().getSamples();
                    Collection<SampleInformation> newSamples = newSampleGroups.stream().filter(sg -> sg.getName()
                            .equals(groupFragment.getSendName())).findFirst().get().getSamples();
                    if (oldSamples.containsAll(newSamples) && newSamples.containsAll(oldSamples)) {
                        continue;
                    }
                    final List<SampleInformation> samplesToLoad = newSamples.stream().filter(s -> null != s.getFileName()).collect(Collectors.toList());
                    this.mPdBinding.resetSampleGroup(groupFragment.getSendName());
                    this.mPdBinding.loadSamples(samplesToLoad);
                    this.runOnUiThread(() -> {
                        groupFragment.clearSamples();
                        samplesToLoad.forEach(groupFragment::addSample);
                        groupFragment.reloadSpinner(getContext());
                    });
                }
                this.runOnUiThread(this::enable);
            });
            Looper.loop();
        });
        thread.setPriority(Thread.MIN_PRIORITY);
        thread.start();

    }

    public void disable(final String message) {
        this.mViewBinding.loadingScreen.setVisibility(View.VISIBLE);
        this.mViewBinding.loadingText.setText(message);
    }

    public void enable() {
        this.mViewBinding.loadingScreen.setVisibility(View.GONE);
    }

    @Override
    protected void onStart() {
        super.onStart();
        bindUserInteractions();
        bindSensors();
        if (null != mViewBinding.startStopButton2) {
            makeHeaderShrinkAndExpandable();
        }
    }

    @Override
    protected void onStop() {
        this.mAppSettings.saveSettings(this);
        super.onStop();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        this.mPdBinding.clearPd();
        if (this.mSensorManager != null) {
            this.mSensorManager.unregisterListener(this);
            this.mSensorManager = null;
        }
    }

    private void bindUserInteractions() {
        this.mViewBinding.startStopButton.setOnCheckedChangeListener(this.mStartStopClicked);
        if (null != mViewBinding.startStopButton2) {
            this.mViewBinding.startStopButton2.setOnCheckedChangeListener(this.mStartStopClicked);
        }
        this.mViewBinding.sessionRecordButton.setOnCheckedChangeListener(this.mSessionRecordClicked);
        this.mViewBinding.bpmSlider.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress, final boolean fromUser) {
                String bpmAsString = String.format(
                        StartupActivity.this.getResources().getConfiguration().getLocales().get(0),
                        "%03d", (int) (progress / 100.0f));
                StartupActivity.this.mViewBinding.speedTapButton.setText(bpmAsString);
                StartupActivity.this.mViewBinding.speedEntry.setText(bpmAsString);
                StartupActivity.this.sendBpm(progress / 100.0f);
                if (fromUser) {
                    StartupActivity.this.initBpm(progress / 100.0f);
                }
            }

            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {
            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {
            }
        });

        this.mViewBinding.speedTapButton.setOnClickListener(v -> this.tapSpeed());
        this.mViewBinding.speedTapButton.setOnLongClickListener(v -> this.showSpeedEditor());
        this.mViewBinding.speedEntry.setOnEditorActionListener(speedEditorListener());
        this.mViewBinding.speedEntry.addTextChangedListener(new TextWatcher() {
            final Handler handler = new Handler(Looper.getMainLooper() /*UI thread*/);
            final Runnable switchToButton = StartupActivity.this::hideSpeedEntry;

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                handler.removeCallbacks(switchToButton);
            }

            @Override
            public void afterTextChanged(Editable s) {
                handler.postDelayed(switchToButton, 30000);
            }
        });
    }

    private void bindSensors() {
        if (this.mSensorManager == null) {
            this.mSensorManager = (SensorManager) this.getSystemService(SENSOR_SERVICE);
        }
        //noinspection deprecation
        if (null != this.mSensorManager && null != mSensorManager.getDefaultSensor(Sensor.TYPE_ORIENTATION)) {
            //noinspection deprecation
            this.mOrientationSensor = this.mSensorManager.getDefaultSensor(Sensor.TYPE_ORIENTATION);
        }
        if (null != this.mSensorManager && null != mSensorManager.getDefaultSensor(Sensor.TYPE_GRAVITY)
                && null != mSensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD)) {
            this.mGravitySensor = this.mSensorManager.getDefaultSensor(Sensor.TYPE_GRAVITY);
            this.mMagneticFieldSensor = this.mSensorManager.getDefaultSensor(Sensor.TYPE_MAGNETIC_FIELD);
        }
    }

    @SuppressLint("ClickableViewAccessibility")
    private void makeHeaderShrinkAndExpandable() {
        this.mViewBinding.scrollView.getViewTreeObserver().addOnScrollChangedListener(() -> {
            final int scrollY = this.mViewBinding.scrollView.getScrollY();
            if (!this.isScrolled && 420 < scrollY) {
                this.mViewBinding.top.setVisibility(View.GONE);
                assert this.mViewBinding.startStopButton2 != null;
                this.mViewBinding.startStopButton2.setVisibility(View.VISIBLE);
                this.isScrolled = true;
            }
        });
        this.mViewBinding.scrollView.setOnTouchListener((v, event) -> {
            if (event.getAction() == ACTION_UP && StartupActivity.this.isScrolled) {
                final int scrollY = this.mViewBinding.scrollView.getScrollY();
                if (1 > scrollY) {
                    StartupActivity.this.mViewBinding.top.setVisibility(View.VISIBLE);
                    assert StartupActivity.this.mViewBinding.startStopButton2 != null;
                    StartupActivity.this.mViewBinding.startStopButton2.setVisibility(View.GONE);
                    StartupActivity.this.mViewBinding.scrollView.setEnabled(false);
                    StartupActivity.this.mViewBinding.scrollView.setScrollY(0);
                    StartupActivity.this.mViewBinding.scrollView.setEnabled(true);
                    StartupActivity.this.isScrolled = false;
                }
            }
            return false;
        });
    }

    protected void sendBpm(final float bpmValue) {
        this.mAppSettings.setBpm(bpmValue);
        this.mPdBinding.sendToPd(PdMessages.BPM, bpmValue);
    }

    private void tapSpeed() {
        final long now = System.currentTimeMillis();
        final long elapsed = now - this.mLatestStep;
        this.mLatestStep = now;
        if (100 > elapsed || 2000 < elapsed) {
            //too short or too long
            return;
        }
        final double weightedLength =
                (elapsed + (this.mBeatLengthInMs.get(0) * 0.75) + (this.mBeatLengthInMs.get(1) * 0.5) + (this.mBeatLengthInMs.get(2) * 0.25))
                        / (1 + 0.75 + 0.5 + 0.25);
        this.mBeatLengthInMs.remove(3);
        this.mBeatLengthInMs.add(0, elapsed);
        int bpmValueInflatedBy100 = (int) ((float) Math.round(60000.0f / (float) weightedLength * 100.0f));
        bpmValueInflatedBy100 = Math.min(bpmValueInflatedBy100, AppSettings.MAX_BPM * 100);
        bpmValueInflatedBy100 = Math.max(bpmValueInflatedBy100, AppSettings.MIN_BPM * 100);
        this.mViewBinding.bpmSlider.setProgress(bpmValueInflatedBy100);
    }

    private boolean showSpeedEditor() {
        this.mViewBinding.speedTapButton.setVisibility(View.GONE);
        this.mViewBinding.speedEntry.setVisibility(View.VISIBLE);
        if (this.mViewBinding.speedEntry.requestFocus()) {
            InputMethodManager imm = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
            imm.showSoftInput(this.mViewBinding.speedEntry, InputMethodManager.SHOW_IMPLICIT);
            return true;
        }
        return false;
    }

    @NonNull
    @Contract(pure = true)
    private TextView.OnEditorActionListener speedEditorListener() {
        return (v, actionId, event) -> {
            if (actionId == EditorInfo.IME_ACTION_DONE || (event != null && event.getKeyCode() == KeyEvent.KEYCODE_ENTER)) {
                hideSpeedEntry();
                try {
                    int speed = Integer.parseUnsignedInt(String.valueOf(this.mViewBinding.speedEntry.getText()), 10);
                    speed = Math.min(speed, AppSettings.MAX_BPM);
                    speed = Math.max(speed, AppSettings.MIN_BPM);
                    this.initBpm(speed);
                    this.mViewBinding.bpmSlider.setProgress(speed * 100);
                } catch (NumberFormatException ignored) {
                }
                return false;
            }
            return false;
        };
    }

    private void hideSpeedEntry() {
        this.mViewBinding.speedEntry.setVisibility(View.GONE);
        this.mViewBinding.speedTapButton.setVisibility(View.VISIBLE);
    }

    @Override
    public void startOtherServices() {
        this.mOrientation = this.getCurrentOrientation();
        this.setRequestedOrientation(this.getCurrentOrientation());
        if (this.mSensorManager == null) {
            this.mSensorManager = (SensorManager) this.getSystemService(SENSOR_SERVICE);
        }
        if (null != this.mSensorManager && null != mOrientationSensor) {
            this.mSensorManager.registerListener(this, this.mOrientationSensor, SensorManager.SENSOR_DELAY_GAME);
        } else if (null != this.mSensorManager && null != mGravitySensor && null != mMagneticFieldSensor) {
            Log.d("Sensor registration", "registering sensors");
            this.mSensorManager.registerListener(this, this.mGravitySensor, SensorManager.SENSOR_DELAY_GAME);
            this.mSensorManager.registerListener(this, this.mMagneticFieldSensor, SensorManager.SENSOR_DELAY_GAME);
        }
    }

    private int getCurrentOrientation() {
        Display display = getWindowManager().getDefaultDisplay();
        int width, height;
        final Point size = new Point();
        display.getSize(size);
        width = size.x;
        height = size.y;
        switch (display.getRotation()) {
            case Surface.ROTATION_90:
                if (width > height) {
                    return ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                }
            case Surface.ROTATION_180:
                if (height > width) {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                }
            case Surface.ROTATION_270:
                if (width > height) {
                    return ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                }
            case Surface.ROTATION_0:
            default:
                if (height > width) {
                    return ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
                } else {
                    return ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE;
                }
        }
    }

    @Override
    public void initBpm(final float value) {
        this.mLatestStep = System.currentTimeMillis();
        final long meanDifference = (long) (60000 / value);
        this.mBeatLengthInMs.clear();
        this.mBeatLengthInMs.add(0, meanDifference);
        this.mBeatLengthInMs.add(1, meanDifference);
        this.mBeatLengthInMs.add(2, meanDifference);
        this.mBeatLengthInMs.add(3, meanDifference);
    }

    @Override
    public void addDebugPrint(final String print) {
        Log.d("Alligator", print);
    }

    @Override
    public Context getContext() {
        return this;
    }

    @Override
    public void stopOtherListeners() {
        this.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED);
        if (this.mSensorManager != null) {
            this.mSensorManager.unregisterListener(this);
            this.mSensorManager = null;
        }
    }

    @Override
    public void pdInitialized() {
        final Thread thread = new Thread(() -> {
            Looper.prepare();

            Handler handler = new Handler();
            handler.post(() -> {
                this.runOnUiThread(() -> this.disable(this.getString(R.string.audio_engine_init)));
                this.mPdBinding.initPd();
                try {
                    final File patch = FileUtils.extractFullZipAndGetFile(this.getContext(), R.raw.pdlogic, "background.pd", true);
                    this.mPdBinding.loadPatch(patch.getAbsolutePath());
                } catch (final Exception e) {
                    Log.e("Alligator", Objects.requireNonNull(e.getMessage()));
                }
                this.runOnUiThread(() -> this.disable(this.getString(R.string.samples_loading)));
                final List<SampleInformation> sampleConfig = this.mAppSettings.getSamples();
                this.mPdBinding.loadSamples(sampleConfig);

                this.runOnUiThread(() -> {
                    for (final SampleGroupFragment sampleFragment : this.mSampleFragments) {
                        sampleFragment.setPdBinding(this.mPdBinding);
                        sampleFragment.setEnabled(false);
                    }
                    for (final RecordingFragment sampleFragment : this.mRecordingFragments) {
                        sampleFragment.setPdBinding(this.mPdBinding);
                        sampleFragment.setEnabled(false);
                    }
                    for (final SampleGroupFragment groupFragment : this.mSampleFragments) {
                        sampleConfig.stream().filter(c -> c.getSampleGroup().equals(groupFragment.getSendName())).forEach(groupFragment::addSample);
                        groupFragment.reloadSpinner(getContext());
                    }
                });
                this.runOnUiThread(() -> this.disable(this.getString(R.string.audio_engine_init)));
                try {
                    this.mPdBinding.finishInit();
                } catch (final IOException e) {
                    Log.e("Alligator", Objects.requireNonNull(e.getMessage()));
                }
                this.bindReceiver();
                this.runOnUiThread(() -> {
                    for (final SampleGroupFragment sampleFragment : this.mSampleFragments) {
                        sampleFragment.setEnabled(false);
                    }
                    for (final RecordingFragment sampleFragment : this.mRecordingFragments) {
                        sampleFragment.setEnabled(false);
                    }
                    this.enable();
                });
            });
            Looper.loop();
        });
        thread.setPriority(Thread.MIN_PRIORITY);
        thread.start();
    }

    private void bindReceiver() {
        if (null != mPdBinding) {
            this.mPdBinding.addReceiver(this, StartupActivity.GENERAL_RECEIVER);
        }
    }

    @Override
    public void onSensorChanged(final SensorEvent event) {
        switch (event.sensor.getType()) {
            case Sensor.TYPE_GRAVITY:
                this.mGravityValues = event.values;
                this.updateRotation();
                break;
            case Sensor.TYPE_MAGNETIC_FIELD:
                this.mMagneticFieldValues = event.values;
                this.updateRotation();
                break;
            //noinspection deprecation
            case Sensor.TYPE_ORIENTATION:
                // TYPE_ORIENTATION returns degrees, while getRotationMatrix() returns radians, so convert the values to radians
                final float[] orientationInRadians = new float[3];
                orientationInRadians[0] = (float) (event.values[0] * Math.PI / 180);
                orientationInRadians[1] = (float) (event.values[1] * Math.PI / 180);
                orientationInRadians[2] = (float) (event.values[2] * Math.PI / 180);
                this.sendOrientation(orientationInRadians);
                break;
        }
    }

    private void updateRotation() {
        if (null == mOrientationSensor && null != mMagneticFieldValues && null != mGravityValues) {
            final float[] rotationMatrix = new float[16];
            final float[] inclinationMatrix = new float[16];
            if (SensorManager.getRotationMatrix(rotationMatrix, inclinationMatrix, this.mGravityValues, this.mMagneticFieldValues)) {
                float[] orientation = new float[3];
                orientation = SensorManager.getOrientation(rotationMatrix, orientation);
                this.sendOrientation(orientation);
            }
        }
    }

    private void sendOrientation(final float[] orientation) {
        for (final SampleGroupFragment sampleFragment : this.mSampleFragments) {
            sampleFragment.setOrientation(orientation, this.mOrientation);
        }
        for (final RecordingFragment recordingFragment : this.mRecordingFragments) {
            recordingFragment.setOrientation(orientation, this.mOrientation);
        }
    }

    @Override
    public void onAccuracyChanged(final Sensor sensor, final int accuracy) {
    }

    @Override
    public void addMessage(final String symbol, final Object[] args) {
        if (Objects.equals(symbol, PdMessages.RECEIVE_METROBANG)) {
            this.flashBeat();
        } else if (Objects.equals(symbol, PdMessages.RECEIVE_RECORDING_CHUNK)) {
            if (3 == args.length) {
                final int sampleLength = Math.round((Float) args[0]);
                final int snippetNumber = Math.round((Float) args[1]);
                final int bufferUsed = Math.round((Float) args[2]);
                this.saveSessionToWav(sampleLength, bufferUsed);
            }
        }
    }

    private void flashBeat() {
        this.runOnUiThread(() -> {
            this.mViewBinding.speedTapButton.setBackground(new ColorDrawable(getColorFromAttr(this, R.attr.topBackground)));
            this.resetBackgroundHandler.postDelayed(() ->
                    StartupActivity.this.mViewBinding.speedTapButton.setBackground(
                            new ColorDrawable(getColorFromAttr(StartupActivity.this, R.attr.fullscreenBackgroundColor))),
                    50);
        });
    }

    private void saveSessionToWav(final int sampleLength, final int buffer) {
        final float[] leftAudio = new float[sampleLength];
        final float[] rightAudio = new float[sampleLength];
        if (0 == PdBase.readArray(leftAudio, 0, PdMessages.ARRAY_SESSION_L_PREFIX + buffer, 0, sampleLength)
                && 0 == PdBase.readArray(rightAudio, 0, PdMessages.ARRAY_SESSION_R_PREFIX + buffer, 0, sampleLength)) {
            int suggestedSampleRate = PdBase.suggestSampleRate();
            if (0 > suggestedSampleRate) {
                suggestedSampleRate = AudioParameters.suggestSampleRate();
            }
            int sampleRate = suggestedSampleRate;
            ExecutorService executor = Executors.newSingleThreadExecutor();
            try {
                final Handler handler = new Handler(Looper.getMainLooper());
                executor.execute(() -> {
                    final float[] audioData = new float[sampleLength * 2];
                    for (int i = 0; i < sampleLength; i++) {
                        audioData[i * 2] = leftAudio[i];
                        audioData[i * 2 + 1] = rightAudio[i];
                    }
                    try {
                        final WaveFileExporter exporter = new WaveFileExporter(this.getContext());
                        exporter.writeSessionRecording(audioData, this.recordDateTime);
                        if (!this.ongoingRecording) {
                            String fileName = exporter.finalizeSessionRecording(this.recordDateTime, sampleRate);
                            handler.post(() -> fileSaveSuccessful(fileName));
                        }
                    } catch (final IOException e) {
                        Log.e("Write", "session.wav", e);
                        handler.post(this::fileSaveFailed);
                    } catch (final SecurityException e){
                        Log.e("Write", "session.wav", e);
                        handler.post(this::fileSaveFailed);
                    }
                });
            } finally {
                executor.shutdown();
            }
        }
    }

    public void fileSaveSuccessful(String fileName) {
        Snackbar snackbar = Snackbar.make(this.mViewBinding.mainLayout, R.string.file_export_successful, LENGTH_SHORT);
        TextView textView = snackbar.getView().findViewById(com.google.android.material.R.id.snackbar_text);
        textView.setText(String.format(textView.getText().toString(), fileName));
        snackbar.setAction(R.string.open_folder, (v) -> {
            Uri uri = mAppSettings.getExportFolder();
            final Intent intent = new Intent(Intent.ACTION_VIEW);
            intent.setDataAndType(uri, "*/*");
            try {
                startActivity(intent);
            } catch (ActivityNotFoundException e) {
                Log.d("Open Folder", "*/*", e);
                //Giving up
            }
        });
        snackbar.show();
    }

    public void fileSaveFailed() {
        Snackbar snackbar = Snackbar.make(this.mViewBinding.mainLayout, R.string.file_export_failed, LENGTH_LONG);
        snackbar.show();
    }

    @Override
    public void addBang() {

    }

    @Override
    public void addFloat(final float x) {

    }

    @Override
    public void addList(final Object[] args) {

    }

    @Override
    public void addSymbol(final String symbol) {

    }

    protected final CompoundButton.OnCheckedChangeListener mStartStopClicked = new CompoundButton.OnCheckedChangeListener() {
        @Override
        public void onCheckedChanged(@NonNull final CompoundButton compoundButton, final boolean checked) {
            StartupActivity.this.mViewBinding.startStopButton.setOnCheckedChangeListener(null);
            StartupActivity.this.mViewBinding.startStopButton.setChecked(checked);
            StartupActivity.this.mViewBinding.startStopButton.setOnCheckedChangeListener(StartupActivity.this.mStartStopClicked);
            if (null != mViewBinding.startStopButton2) {
                StartupActivity.this.mViewBinding.startStopButton2.setOnCheckedChangeListener(null);
                StartupActivity.this.mViewBinding.startStopButton2.setChecked(checked);
                StartupActivity.this.mViewBinding.startStopButton2.setOnCheckedChangeListener(StartupActivity.this.mStartStopClicked);
            }
            if (checked) {
                StartupActivity.this.mPdBinding.play(StartupActivity.this);
            } else {
                StartupActivity.this.mViewBinding.sessionRecordButton.setChecked(false);
                StartupActivity.this.mPdBinding.pause();
            }
            StartupActivity.this.mViewBinding.sessionRecordButton.setEnabled(checked);
            for (final SampleGroupFragment sampleFragment : StartupActivity.this.mSampleFragments) {
                sampleFragment.setEnabled(checked);
            }
            for (final RecordingFragment sampleFragment : StartupActivity.this.mRecordingFragments) {
                sampleFragment.setEnabled(checked);
            }
        }
    };
}