extends CanvasLayer

signal start_game
signal select_difficulty(difficulty: Dictionary)
signal select_card_set(name: Dictionary)
signal set_player_count(count: int)


func _ready() -> void:
	set_difficulty_and_card_set_labels()

	# Set version number
	%BrainTitleContainer/VersionNumber.text = Helpers.get_version_number_str()

	focus()


func get_default_player_count() -> int:
	return Config.get_param("default_player_count", 1)


func get_default_difficulty_index() -> int:
	return Config.get_param("default_difficulty_index", 2)


func get_default_card_set_index() -> int:
	return Config.get_param("default_card_set_index", 0)


func get_difficulty_levels() -> Dictionary:
	return {
		0: {
			"label": tr("Very Easy"),
			"siblings_cards": 2,
			"different_images": 3
		},
		1: {
			"label": tr("Easy"),
			"siblings_cards": 2,
			"different_images": 6
		},
		2: {
			"label": tr("Normal"),
			"siblings_cards": 2,
			"different_images": 12
		},
		3: {
			"label": tr("Hard"),
			"siblings_cards": 2,
			"different_images": 18
		},
		4: {
			"label": tr("Very Hard"),
			"siblings_cards": 3,
			"different_images": 12
		},
	}


func get_card_sets() -> Dictionary:
	return {
		"animals": tr("Animals"),
		"cats": tr("Cats"),
		"mythical creatures": tr("Mythical Creatures"),
	}


func set_difficulty_and_card_set_labels() -> void:
	var difficulty_levels = get_difficulty_levels()
	var card_sets = get_card_sets()
	
	# Clean current labels
	%DifficultyPresetSelect/OptionButton.clear()
	%CardSetSelect/OptionButton.clear()
	
	# Add the difficulty options
	for i in difficulty_levels:
		#%DifficultyPresetSelect/OptionButton.remove_item(i)
		%DifficultyPresetSelect/OptionButton.add_item(difficulty_levels[i].label, i)

	# Add card sets options and metadata (which is the name of the folder containing the cards)
	var i = 0
	for c in card_sets:
		%CardSetSelect/OptionButton.add_item(card_sets[c], i)
		%CardSetSelect/OptionButton.set_item_metadata(i, c)
		i += 1

	# Set default player count
	var default_player_count: int = get_default_player_count()
	%PlayerCountSelect/HSlider.value = default_player_count
	# NOTE BUG: Setting the `value` should trigger the `value_changed` event,
	# but as Godot 4.5.1 it does not work for some reason.
	# It might be a bug, check again later if it's possible to remove the manual call below.
	_on_player_count_changed(default_player_count)

	# Select default card set in the list
	var default_card_set_index: int = get_default_card_set_index()
	%CardSetSelect/OptionButton.select(default_card_set_index)
	select_card_set.emit(%CardSetSelect/OptionButton.get_item_metadata(default_card_set_index))

	# Select the default difficulty and set its details
	var default_difficulty_index: int = get_default_difficulty_index()
	%DifficultyPresetSelect/OptionButton.select(default_difficulty_index)
	_on_difficulty_selected(default_difficulty_index)


func show_main_menu() -> void:
	Helpers.show_nodes_group("hidable_main_menu")
	focus()


func _on_start_button_pressed() -> void:
	Helpers.hide_nodes_group("hidable_main_menu")
	start_game.emit()


func _on_message_timer_timeout() -> void:
	%BrainTitleContainer.hide()


func _on_player_count_changed(count: int) -> void:
	set_player_count.emit(count)
	%PlayerCountSelect/Label.text =\
		tr(
			"Number of Players: %s",
			"The number of players who will be playing in this round"
		) % count
	Config.store("default_player_count", count)


func _on_difficulty_selected(index: int) -> void:
	var difficulty = get_difficulty_levels()[index]
	select_difficulty.emit(difficulty)
	set_difficulty_details(difficulty)
	Config.store("default_difficulty_index", index)


func _on_card_set_selected(index: int) -> void:
	var card_set_name: String = %CardSetSelect/OptionButton.get_item_metadata(index)
	select_card_set.emit(card_set_name)
	Config.store("default_card_set_index", index)


func set_difficulty_details(difficulty: Dictionary) -> void:
	var cards_count: int = difficulty.different_images * difficulty.siblings_cards
	%CardSetDetailsContainer/CardsNumberContainer/Label.text =\
		tr("%s cards", "The number of cards of the chosen difficulty") % cards_count
	%CardSetDetailsContainer/SiblingsNumberContainer/Label.text =\
		tr(
			"%s siblings cards",
			"The number of siblings cards for each image. Memory games usually only have 2 siblings per image, but this game offer a higher difficulty level with 3 siblings per image."
		) % difficulty.siblings_cards


func focus() -> void:
	%StartButton.grab_focus()
