extends Node

@onready var cards_container := $AspectRatioContainer/FlowContainer
var card_set_name: String
var card_scene: PackedScene = preload("res://card.tscn")
var card_size_limit := 300
var cards: Array[Control]
var player_count: int
var number_of_siblings_cards: int
var number_of_different_images: int
var flipped_cards_attempt: Array[Node]
var number_of_images_found := 0
const SUCCESS_SOUNDS := [
	preload("res://assets/sounds/success_flips/success1.ogg"),
	preload("res://assets/sounds/success_flips/success2.ogg"),
	preload("res://assets/sounds/success_flips/success3.ogg"),
	preload("res://assets/sounds/success_flips/success4.ogg"),
	preload("res://assets/sounds/success_flips/success5.ogg")
]
var game_in_progress := false


func _ready() -> void:
	$CardFlipbackTimer.wait_time = $HUD/MenuContainer/MenuItems/CardsFlipBackSpeedContainer/CardsFlipBackSpeedRange.value / 1000


func game_win() -> void:
	$HUD.game_end()
	$HUD.show_game_win()
	$WinSound.play()


func clear_game() -> void:
	$CardFlipbackTimer.stop()
	$CardFlipbackTimer.timeout.emit()
	game_in_progress = false
	flipped_cards_attempt.clear()
	number_of_images_found = 0
	cards.clear()
	Player.reset()
	for n in cards_container.get_children(true):
		n.queue_free()


func start_game() -> void:
	game_in_progress = true

	for _i in player_count:
		Player.new()

	update_scores()
	$HUD.game_start(player_count > 1)
	
	var card_set: CardSet = CardSet.new(card_set_name)
	var image_names: Array = card_set.image_names
	# Shuffle to have different images of the card set
	image_names.shuffle()
	for image_name in card_set.image_names.slice(0, number_of_different_images):
		var i = 0
		while i < number_of_siblings_cards:
			var card := card_scene.instantiate()
			card.card_name = image_name.get_basename()
			card.front_texture = load("res://card_sets/"+card_set_name+"/"+image_name)
			card.connect("card_flipped", card_flipped)
			cards.append(card)
			i += 1

	# Shuffle to spread siblings
	cards.shuffle()
	add_cards()

	# Focus the first card to be able to play with the keyboard
	focus_first_card()


func add_cards() -> void:
	var max_card_size := max_card_size_in_container(
		$AspectRatioContainer.size.x,
		$AspectRatioContainer.size.y,
		cards.size()
	)
	var card_size: int = min(max_card_size, card_size_limit)
	var particles_scale: float = card_size / 250.0
	
	for c in cards:
		c.custom_minimum_size = Vector2(card_size, card_size)
		c.set_particles_scale(Vector2(particles_scale, particles_scale))
		cards_container.add_child(c)


func max_card_size_in_container(width: int, height: int, cards_nbr: int) -> int:
	var card_width: int
	var card_height: int

	var pw: float = ceil(sqrt(cards_nbr * width / height))
	if (floor(pw * height / width ) * pw < cards_nbr):
		card_width = height / ceil(pw * height / width)
	else:
		card_width = width / pw

	var ph: float = ceil(sqrt(cards_nbr * height / width))
	if floor(ph * width / height) * ph < cards_nbr:
		card_height = width / ceil(width * ph / height)
	else:
		card_height = height / ph;
	
	return max(card_width, card_height) - 20


func card_flipped(card: Node) -> void:
	set_all_back_cards_disabled(true)
	flipped_cards_attempt.append(card)
	card.disable_card(true)
	# if any card in flipped_cards_attempt is different than the first one
	if flipped_cards_attempt.any(func(c): return c.card_name != flipped_cards_attempt[0].card_name):
		$CardFlipbackTimer.start()
		await $CardFlipbackTimer.timeout
		$FlipCardsBackSound.play()
		for c in flipped_cards_attempt:
			c.disable_card(false)
			c.flip_card_back()
		flipped_cards_attempt.clear()
		Player.next_player()
		update_scores()
	elif len(flipped_cards_attempt) == number_of_siblings_cards:
		# Wait for the last card animation to finish
		await flipped_cards_attempt.back().find_child("AnimationPlayer").animation_finished

		for c in flipped_cards_attempt:
			c.emit_particle()
		flipped_cards_attempt.clear()
		number_of_images_found += 1
		Player.current_player().score += 1
		update_scores()
		if number_of_images_found == number_of_different_images:
			game_win()
		else:
			play_success_flip()
			update_scores()

	set_all_back_cards_disabled(false)


func set_all_back_cards_disabled(disabled: bool) -> void:
	for card in cards.filter(func(c): return not c.find_child("Button").button_pressed):
		card.disable_card(disabled)


func play_success_flip() -> void:
	$SuccessFlipSound.stream = SUCCESS_SOUNDS.pick_random()
	$SuccessFlipSound.play()


func _on_main_menu_set_player_count(count: int) -> void:
	player_count = count


func _on_main_menu_select_difficulty(difficulty: Dictionary) -> void:
	number_of_siblings_cards = difficulty["siblings_cards"]
	number_of_different_images = difficulty["different_images"]


func _on_main_menu_select_card_set(selected_card_set_name: String) -> void:
	card_set_name = selected_card_set_name.to_lower()


func update_scores() -> void:
	if player_count > 1:
		$HUD.update_score_multiplayer()
	else:
		$HUD.update_score(number_of_images_found, number_of_different_images)


func _on_hud_surrender() -> void:
	$HUD.game_end()
	clear_game()
	$MainMenu.show_main_menu()


func _on_hud_menu_closed():
	if game_in_progress:
		focus_first_card()
	else:
		$MainMenu.focus()


func focus_first_card() -> void:
	cards_container.get_child(0).find_child("Button").grab_focus()


func _on_hud_language_selected():
	# This is done to update the difficulty labels with the new language
	$MainMenu.set_difficulty_and_card_set_labels()


func _on_hud_win_menu_closed(play_again: bool) -> void:
	clear_game()
	if play_again:
		start_game()
	else:
		$MainMenu.show_main_menu()


func _on_hud_change_cards_flip_back_speed(value: float) -> void:
	$CardFlipbackTimer.wait_time = value / 1000
