extends CanvasLayer

signal surrender
signal menu_closed
signal language_selected
signal win_menu_closed(play_again: bool)
signal change_cards_flip_back_speed(duration: float)

const VOLUME_UP := preload("res://assets/volume-up-fill.webp")
const VOLUME_MUTED := preload("res://assets/volume-mute-fill.webp")
# Set locales in a var because we sort them at runtime since there is no way to do it at compile time yet
var locales := [
	{"code": "en", "name": "English"},
	{"code": "fr", "name": "français"},
	{"code": "pt_BR", "name": "Portugues do Brasil"},
	{"code": "es", "name": "Español"},
	{"code": "tr", "name": "Türkçe"},
	{"code": "nl", "name": "Nederlands"},
	{"code": "pl", "name": "Polski"},
	{"code": "eu", "name": "Euskara"},
	{"code": "de", "name": "Deutsch"},
	{"code": "it", "name": "Italiano"},
	{"code": "cs", "name": "Čeština"},
	{"code": "zh_Hans", "name": "简体中文"},
	{"code": "zh_Hant", "name": "繁體中文"},
]
var master_bus_index := AudioServer.get_bus_index("Master")


func _ready() -> void:
	# There is no way to get the language names in their own locale for now
	# https://github.com/godotengine/godot-proposals/issues/2378
	# locales.append_array(TranslationServer.get_loaded_locales())
	var locale = Config.get_param("locale", get_default_locale())
	locales.sort_custom(func(a, b): return a.name.naturalnocasecmp_to(b.name) < 0)
	var saved_locale_index :int
	for i in locales.size():
		$MenuContainer/MenuItems/LanguageOptions.add_item(locales[i].name, i)
		$MenuContainer/MenuItems/LanguageOptions.set_item_metadata(i, locales[i].code)
		if locale == locales[i].code:
			saved_locale_index = i

	if locale != null:
		$MenuContainer/MenuItems/LanguageOptions.select(saved_locale_index)
		TranslationServer.set_locale(locale)

	# Set mute sound state
	var mute_sounds: bool = Config.get_param("mute_sounds", false)
	$MenuContainer/MenuItems/MuteSounds.button_pressed = mute_sounds
	_on_mute_sounds_toggled(mute_sounds)
	
	# Set cards flip back speed state
	var initial_flip_back_speed = %CardsFlipBackSpeedRange.value
	var flip_back_speed = Config.get_param("cards_flip_back_speed", initial_flip_back_speed)
	%CardsFlipBackSpeedRange.value = flip_back_speed

	# %CardsFlipBackSpeedRange.set_value_no_signal(flip_back_speed)
	
	# Set version number
	$MenuContainer/AboutContainer/HBoxContainer/Version.text = Helpers.get_version_number_str()
	
	# Make rich text label links clickable
	if OS.get_name() != "HTML5":
		$MenuContainer/AboutContainer/RichTextLabel\
		.connect("meta_clicked", _on_RichTextLabel_meta_clicked)


# This function is needed because sometime the
# Translation server has a default locale set by the OS
# which can be different of the codes used here.
# Without this, the language is correctly set to the default of the OS,
# but the select in the menu is set to the default "en".
# e.g. "fr_FR" should match "fr"
func get_default_locale() -> String:
	var current_locale := TranslationServer.get_locale()

	# Iterate through locales and find a matching language code
	for i in locales.size():
		if current_locale.find(locales[i].code) == 0:
			return locales[i].code

	# If no match is found, use the default locale
	return locales[0].code


func _on_menu_button_pressed() -> void:
	if $MenuContainer.visible:
		$MenuContainer.hide()
		menu_closed.emit()
	else:
		$MenuContainer.show()
		$MenuContainer/MenuItems.get_child(0).grab_focus()


func _on_return_pressed() -> void:
	_on_menu_button_pressed()


func _on_surrender_pressed() -> void:
	surrender.emit()
	$MenuContainer.hide()


func _on_language_options_item_selected(index) -> void:
	var locale: String = $MenuContainer/MenuItems/LanguageOptions.get_item_metadata(index)
	TranslationServer.set_locale(locale)
	Config.store("locale", locale)
	language_selected.emit()


func update_score(number_of_images_found: int, number_of_different_images: int) -> void:
	%ScoreContainer/Score.text = "%s/%s" % [number_of_images_found, number_of_different_images]


func update_score_multiplayer() -> void:
	%MultiplayerScoreContainer/Label.text = tr("Player {number}", "Player ID, e.g. 'Player 1'")\
			.format({number = Player.current_player().index})

	%MultiplayerScoreContainer/Score.text = tr("Score: {number}", "Number of image found used as the player score")\
			.format({number = Player.current_player().score})


func game_start(is_multiplayer: bool = false) -> void:
	if is_multiplayer:
		%MultiplayerScoreContainer.show()
	else:
		%ScoreContainer.show()
	%SurrenderButton.show()


func game_end() -> void:
	%ScoreContainer.hide()
	%MultiplayerScoreContainer.hide()
	%SurrenderButton.hide()


func show_game_win() -> void:
	$TopContainer.hide()

	if (Player.players.size() > 1):
		show_multiplayer_game_win_nodes()
		Player.sort_players_by_score()

		if (Player.players[0].score == Player.players[1].score):
			$WinMessage/WinMessageContainer/MultiplayerWinLabel.text  = tr("Draw", "Describes when there is no winner because top players scores are equal")
		else :
			$WinMessage/WinMessageContainer/MultiplayerWinLabel.text = tr("Player %s wins!") % Player.players[0].index

		$WinMessage/WinMessageContainer/Scores.text = Player.format_scores()
		$WinMessage/WinMessageContainer/MarginContainer/HBoxContainer/ReplayButton.grab_focus()
	else:
		show_single_player_game_win_nodes()
		
	$WinMessage.show()


func show_multiplayer_game_win_nodes() -> void:
	$WinMessage/WinMessageContainer/MultiplayerWinLabel.show()
	$WinMessage/WinMessageContainer/SinglePlayerWinLabel.hide()
	$WinMessage/WinMessageContainer/Scores.show()


func show_single_player_game_win_nodes() -> void:
	$WinMessage/WinMessageContainer/MultiplayerWinLabel.hide()
	$WinMessage/WinMessageContainer/SinglePlayerWinLabel.show()
	$WinMessage/WinMessageContainer/Scores.hide()


func _on_about_pressed() -> void:
	$MenuContainer/MenuItems.hide()
	$MenuContainer/AboutContainer.show()
	$MenuContainer/AboutContainer/AboutClose.grab_focus()


func _on_about_close_pressed() -> void:
	$MenuContainer/AboutContainer.hide()
	$MenuContainer/MenuItems.show()
	$MenuContainer/MenuItems/About.grab_focus()


func _on_mute_sounds_toggled(button_pressed: bool) -> void:
	$MenuContainer/MenuItems/MuteSounds.icon = VOLUME_MUTED if button_pressed else VOLUME_UP
	AudioServer.set_bus_mute(master_bus_index, button_pressed)
	Config.store("mute_sounds", button_pressed)


func _on_RichTextLabel_meta_clicked(meta) -> void:
	OS.shell_open(meta)


func _on_cards_flip_back_speed_range_value_changed(value: float):
	change_cards_flip_back_speed.emit(value)
	Config.store("cards_flip_back_speed", value)


func _on_replay_button_pressed() -> void:
	$TopContainer.show()
	$WinMessage.hide()
	win_menu_closed.emit(true)


func _on_new_button_pressed() -> void:
	$TopContainer.show()
	$WinMessage.hide()
	win_menu_closed.emit(false)
