// Copyright (C) 2021-2024 Yaroslav Pronin <proninyaroslav@mail.ru>
// Copyright (C) 2021 Insurgo Inc. <insurgo@riseup.net>
//
// This file is part of LibreTrack.
//
// LibreTrack is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// LibreTrack is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with LibreTrack.  If not, see <http://www.gnu.org/licenses/>.

import 'package:collection/collection.dart';
import 'package:enum_to_string/enum_to_string.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter_secure_storage/flutter_secure_storage.dart';
import 'package:injectable/injectable.dart';
import 'package:libretrack/core/entity/entity.dart';
import 'package:libretrack/core/platform_info.dart';
import 'package:libretrack/core/storage/database.dart';

abstract class ServiceAuthStorage {
  Future<void> insert({
    required TrackingServiceType type,
    required AuthData authData,
  });

  Future<void> deleteByServiceType(TrackingServiceType type);

  Future<AuthData?> getByServiceType(TrackingServiceType type);

  bool get isSecureStorageSupported;
}

@Injectable(as: ServiceAuthStorage)
class ServiceAuthStorageImpl implements ServiceAuthStorage {
  final AppDatabase _db;
  final FlutterSecureStorage _secureStorage;
  final PlatformInfo _platformInfo;

  ServiceAuthStorageImpl(
    this._db,
    this._secureStorage,
    this._platformInfo,
  );

  @override
  Future<void> deleteByServiceType(TrackingServiceType type) async {
    if (isSecureStorageSupported) {
      await _secureStorageDelete(_secureStorage, type);
    } else {
      await _dbDelete(_db, type);
    }
  }

  @override
  Future<AuthData?> getByServiceType(TrackingServiceType type) async {
    if (isSecureStorageSupported) {
      return _secureStorageGet(_secureStorage, type);
    } else {
      return _dbGet(_db, type);
    }
  }

  @override
  Future<void> insert({
    required TrackingServiceType type,
    required AuthData authData,
  }) async {
    if (isSecureStorageSupported) {
      await _secureStorageInsert(_secureStorage, type, authData);
    } else {
      await _dbInsert(_db, type, authData);
    }
  }

  // TODO: Windows/macOS/Web support
  @override
  bool get isSecureStorageSupported =>
      _platformInfo.isAndroid || _platformInfo.isIOS || _platformInfo.isLinux;
}

Future<void> _dbInsert(
  AppDatabase db,
  TrackingServiceType type,
  AuthData authData,
) async {
  await db.serviceAuthDao.insertData(authData.entries
      .map(
        (entry) => AuthDataField(
          key: entry.key,
          value: entry.value,
          serviceType: type,
        ),
      )
      .toList());
}

Future<void> _dbDelete(
  AppDatabase db,
  TrackingServiceType type,
) async {
  await db.serviceAuthDao.deleteDataByServiceId(type);
}

Future<AuthData?> _dbGet(
  AppDatabase db,
  TrackingServiceType type,
) async {
  final data = await db.serviceAuthDao.getDataByServiceType(type);

  return data.isEmpty
      ? null
      : AuthData({
          for (var e in data) e.key: e.value,
        });
}

@immutable
class _SecureKey {
  final TrackingServiceType serviceType;
  final String baseKey;

  const _SecureKey({
    required this.serviceType,
    required this.baseKey,
  });

  // ignore: prefer_constructors_over_static_methods
  static _SecureKey? fromKey(String? key) {
    if (key == null) {
      return null;
    }
    final delimeter = key.indexOf(':');
    if (delimeter < 0) {
      return null;
    }
    final serviceType = key.substring(0, delimeter);
    final baseKey = key.substring(delimeter + 1);
    if (serviceType.isEmpty || baseKey.isEmpty) {
      return null;
    } else {
      return _SecureKey(
        serviceType: EnumToString.fromString(
          TrackingServiceType.values,
          serviceType,
        )!,
        baseKey: baseKey,
      );
    }
  }

  static bool isServiceKey(TrackingServiceType type, String key) =>
      key.startsWith('${EnumToString.convertToString(type)}:');

  @override
  String toString() => '${EnumToString.convertToString(serviceType)}:$baseKey';
}

Future<void> _secureStorageInsert(
  FlutterSecureStorage secureStorage,
  TrackingServiceType type,
  AuthData authData,
) async {
  for (final entry in authData.entries) {
    await secureStorage.write(
      key: _SecureKey(
        serviceType: type,
        baseKey: entry.key,
      ).toString(),
      value: entry.value,
    );
  }
}

Future<void> _secureStorageDelete(
  FlutterSecureStorage secureStorage,
  TrackingServiceType type,
) async {
  final keyValueMap = await secureStorage.readAll();
  for (final key in keyValueMap.keys) {
    if (_SecureKey.isServiceKey(type, key)) {
      await secureStorage.delete(key: key);
    }
  }
}

Future<AuthData?> _secureStorageGet(
  FlutterSecureStorage secureStorage,
  TrackingServiceType type,
) async {
  final keyValueMap = await secureStorage.readAll();
  final entries = keyValueMap.entries;
  final map = <String, String>{};
  for (final e in entries) {
    final key = _SecureKey.fromKey(e.key);
    if (key == null || key.serviceType != type) {
      continue;
    }
    map[key.baseKey] = e.value;
  }
  final authData = AuthData(map);

  return authData.isEmpty ? null : authData;
}

@visibleForTesting
class TestFlutterSecureStorage implements FlutterSecureStorage {
  final Map<String?, String?> _keyValueMap = {};
  final Map<String, List<ValueChanged<String?>>> _listeners = {};

  TestFlutterSecureStorage();

  @override
  AndroidOptions get aOptions => AndroidOptions.defaultOptions;

  @override
  Future<bool> containsKey(
      {required String key,
      IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    return _keyValueMap.containsKey(key);
  }

  @override
  Future<void> delete(
      {required String key,
      IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    _keyValueMap.remove(key);
  }

  @override
  Future<void> deleteAll(
      {IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    _keyValueMap.clear();
  }

  @override
  IOSOptions get iOptions => throw UnimplementedError();

  @override
  Future<bool?> isCupertinoProtectedDataAvailable() async {
    return false;
  }

  @override
  LinuxOptions get lOptions => LinuxOptions.defaultOptions;

  @override
  MacOsOptions get mOptions => MacOsOptions.defaultOptions;

  @override
  Stream<bool>? get onCupertinoProtectedDataAvailabilityChanged =>
      Stream.value(false);

  @override
  Future<String?> read(
      {required String key,
      IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    return _keyValueMap[key];
  }

  @override
  Future<Map<String, String>> readAll(
      {IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    return Map.from(_keyValueMap);
  }

  @override
  void registerListener(
      {required String key, required ValueChanged<String?> listener}) {
    var list = _listeners.putIfAbsent(key, () => []);
    list.add(listener);
    _listeners[key] = list;
  }

  @override
  void unregisterAllListeners() {
    _listeners.clear();
  }

  @override
  void unregisterAllListenersForKey({required String key}) {
    _listeners.remove(key);
  }

  @override
  void unregisterListener(
      {required String key, required ValueChanged<String?> listener}) {
    var list = _listeners[key];
    var newList = list?.whereNot((value) => value == listener).toList();
    if (newList != null) {
      _listeners[key] = newList;
    }
  }

  @override
  WindowsOptions get wOptions => WindowsOptions.defaultOptions;

  @override
  WebOptions get webOptions => WebOptions.defaultOptions;

  @override
  Future<void> write(
      {required String key,
      required String? value,
      IOSOptions? iOptions,
      AndroidOptions? aOptions,
      LinuxOptions? lOptions,
      WebOptions? webOptions,
      MacOsOptions? mOptions,
      WindowsOptions? wOptions}) async {
    _keyValueMap[key] = value;
  }
}
