/*
 * Copyright (C) 2016-2025 Yaroslav Pronin <proninyaroslav@mail.ru>
 *
 * This file is part of LibreTorrent.
 *
 * LibreTorrent is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * LibreTorrent is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with LibreTorrent.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.proninyaroslav.libretorrent.ui.settings.pages;

import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.text.InputFilter;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.navigation.fragment.NavHostFragment;
import androidx.preference.EditTextPreference;
import androidx.preference.ListPreference;
import androidx.preference.Preference;
import androidx.preference.SwitchPreferenceCompat;

import com.google.android.material.snackbar.Snackbar;

import org.proninyaroslav.libretorrent.MainActivity;
import org.proninyaroslav.libretorrent.R;
import org.proninyaroslav.libretorrent.core.InputFilterRange;
import org.proninyaroslav.libretorrent.core.RepositoryHelper;
import org.proninyaroslav.libretorrent.core.exception.UnknownUriException;
import org.proninyaroslav.libretorrent.core.settings.SessionSettings;
import org.proninyaroslav.libretorrent.core.settings.SettingsRepository;
import org.proninyaroslav.libretorrent.core.system.FileSystemFacade;
import org.proninyaroslav.libretorrent.core.system.SystemFacadeHelper;
import org.proninyaroslav.libretorrent.ui.NavBarFragment;
import org.proninyaroslav.libretorrent.ui.NavBarFragmentDirections;
import org.proninyaroslav.libretorrent.ui.filemanager.FileManagerConfig;
import org.proninyaroslav.libretorrent.ui.filemanager.FileManagerFragment;
import org.proninyaroslav.libretorrent.ui.settings.CustomPreferenceFragment;

import java.util.ArrayList;
import java.util.List;

/*
 * TODO: add PeX enable/disable feature
 */

public class NetworkSettingsFragment extends CustomPreferenceFragment
        implements Preference.OnPreferenceChangeListener {
    private static final String TAG = NetworkSettingsFragment.class.getSimpleName();

    private static final String KEY_CHOOSE_IP_FILTER_FILE_DIALOG_REQUEST = TAG + "_choose_ip_filter_file_dialog";
    private static final String KEY_ANONYMOUS_MODE_SETTINGS_REQUEST = TAG + "_anonymous_mode_settings";

    private MainActivity activity;
    private SettingsRepository pref;
    private FileSystemFacade fs;

    @Override
    public void onAttach(@NonNull Context context) {
        super.onAttach(context);

        if (context instanceof MainActivity a) {
            activity = a;
        }
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        if (activity == null) {
            activity = (MainActivity) requireActivity();
        }

        binding.appBar.setTitle(R.string.pref_header_network);
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        Context context = activity.getApplicationContext();
        pref = RepositoryHelper.getSettingsRepository(context);
        fs = SystemFacadeHelper.getFileSystemFacade(context);

        var proxy = findPreference(getString(R.string.pref_key_network_proxy_settings));
        if (proxy != null) {
            proxy.setOnPreferenceClickListener((p) -> {
                var action = NetworkSettingsFragmentDirections.actionProxySettings();
                NavHostFragment.findNavController(this).navigate(action);
                return true;
            });
        }

        String keyEnableDht = getString(R.string.pref_key_enable_dht);
        SwitchPreferenceCompat enableDht = findPreference(keyEnableDht);
        if (enableDht != null) {
            enableDht.setChecked(pref.enableDht());
            bindOnPreferenceChangeListener(enableDht);
        }

        String keyEnableLsd = getString(R.string.pref_key_enable_lsd);
        SwitchPreferenceCompat enableLsd = findPreference(keyEnableLsd);
        if (enableLsd != null) {
            enableLsd.setChecked(pref.enableLsd());
            bindOnPreferenceChangeListener(enableLsd);
        }

        String keyEnableUtp = getString(R.string.pref_key_enable_utp);
        SwitchPreferenceCompat enableUtp = findPreference(keyEnableUtp);
        if (enableUtp != null) {
            enableUtp.setChecked(pref.enableUtp());
            bindOnPreferenceChangeListener(enableUtp);
        }

        String keyEnableUpnp = getString(R.string.pref_key_enable_upnp);
        SwitchPreferenceCompat enableUpnp = findPreference(keyEnableUpnp);
        if (enableUpnp != null) {
            enableUpnp.setChecked(pref.enableUpnp());
            bindOnPreferenceChangeListener(enableUpnp);
        }

        String keyEnableNatpmp = getString(R.string.pref_key_enable_natpmp);
        SwitchPreferenceCompat enableNatpmp = findPreference(keyEnableNatpmp);
        if (enableNatpmp != null) {
            enableNatpmp.setChecked(pref.enableNatPmp());
            bindOnPreferenceChangeListener(enableNatpmp);
        }

        String keyRandomPort = getString(R.string.pref_key_use_random_port);
        SwitchPreferenceCompat randomPort = findPreference(keyRandomPort);
        if (randomPort != null) {
            randomPort.setSummary(getString(R.string.pref_use_random_port_summarty,
                    SessionSettings.DEFAULT_PORT_RANGE_FIRST,
                    SessionSettings.DEFAULT_PORT_RANGE_SECOND - 10));
            randomPort.setDisableDependentsState(true);
            randomPort.setChecked(pref.useRandomPort());
            bindOnPreferenceChangeListener(randomPort);
        }

        InputFilter[] portFilter = new InputFilter[]{InputFilterRange.PORT_FILTER};

        String keyPortStart = getString(R.string.pref_key_port_range_first);
        EditTextPreference portStart = findPreference(keyPortStart);
        if (portStart != null) {
            String value = Integer.toString(pref.portRangeFirst());
            portStart.setOnBindEditTextListener((editText) -> editText.setFilters(portFilter));
            portStart.setSummary(value);
            portStart.setText(value);
            bindOnPreferenceChangeListener(portStart);
        }

        String keyPortEnd = getString(R.string.pref_key_port_range_second);
        EditTextPreference portEnd = findPreference(keyPortEnd);
        if (portEnd != null) {
            String value = Integer.toString(pref.portRangeSecond());
            portEnd.setOnBindEditTextListener((editText) -> editText.setFilters(portFilter));
            portEnd.setSummary(value);
            portEnd.setText(value);
            bindOnPreferenceChangeListener(portEnd);
        }

        String keyEncryptInConnectionsMode = getString(R.string.pref_key_enc_in_connections_mode);
        ListPreference encryptInConnectionsMode = findPreference(keyEncryptInConnectionsMode);
        if (encryptInConnectionsMode != null) {
            var type = pref.encryptInConnectionsMode();
            encryptInConnectionsMode.setValueIndex(type);
            bindOnPreferenceChangeListener(encryptInConnectionsMode);
        }

        String keyEncryptOutConnectionsMode = getString(R.string.pref_key_enc_out_connections_mode);
        ListPreference encryptOutConnectionsMode = findPreference(keyEncryptOutConnectionsMode);
        if (encryptOutConnectionsMode != null) {
            var type = pref.encryptOutConnectionsMode();
            encryptOutConnectionsMode.setValueIndex(type);
            bindOnPreferenceChangeListener(encryptOutConnectionsMode);
        }

        String keyIpFilter = getString(R.string.pref_key_enable_ip_filtering);
        SwitchPreferenceCompat ipFilter = findPreference(keyIpFilter);
        if (ipFilter != null) {
            ipFilter.setChecked(pref.enableIpFiltering());
            bindOnPreferenceChangeListener(ipFilter);
        }

        String keyIpFilterFile = getString(R.string.pref_key_ip_filtering_file);
        Preference ipFilterFile = findPreference(keyIpFilterFile);
        if (ipFilterFile != null) {
            String path = pref.ipFilteringFile();
            if (path != null) {
                try {
                    ipFilterFile.setSummary(fs.getFilePath(Uri.parse(path)));
                } catch (UnknownUriException e) {
                    Log.e(TAG, Log.getStackTraceString(e));
                }
            }
            ipFilterFile.setOnPreferenceClickListener((Preference preference) -> {
                fileChooseDialog();

                return true;
            });
        }

        String keyShowNatErrors = getString(R.string.pref_key_show_nat_errors);
        SwitchPreferenceCompat showNatErrors = findPreference(keyShowNatErrors);
        if (showNatErrors != null) {
            showNatErrors.setChecked(pref.showNatErrors());
            bindOnPreferenceChangeListener(showNatErrors);
        }

        String keyAnonymousMode = getString(R.string.pref_key_anonymous_mode);
        Preference anonymousMode = findPreference(keyAnonymousMode);
        if (anonymousMode != null) {
            anonymousMode.setSummary(pref.anonymousMode() ? R.string.switch_on : R.string.switch_off);
            anonymousMode.setOnPreferenceClickListener((p) -> {
                var action = NetworkSettingsFragmentDirections.actionAnonymouModeSettings(
                        KEY_ANONYMOUS_MODE_SETTINGS_REQUEST
                );
                NavHostFragment.findNavController(this).navigate(action);
                return true;
            });
        }

        String keySeedingOutgoingConn = getString(R.string.pref_key_seeding_outgoing_connections);
        SwitchPreferenceCompat seedingOutgoingConn = findPreference(keySeedingOutgoingConn);
        if (seedingOutgoingConn != null) {
            seedingOutgoingConn.setChecked(pref.seedingOutgoingConnections());
            bindOnPreferenceChangeListener(seedingOutgoingConn);
        }

        String keyValidateHttpsTrackers = getString(R.string.pref_key_validate_https_trackers);
        SwitchPreferenceCompat validateHttpsTrackers = findPreference(keyValidateHttpsTrackers);
        if (validateHttpsTrackers != null) {
            validateHttpsTrackers.setChecked(pref.validateHttpsTrackers());
            bindOnPreferenceChangeListener(validateHttpsTrackers);
        }

        var navBarFragment = activity.findNavBarFragment(this);
        if (navBarFragment != null) {
            setOpenIpFilterFileDialogListener(navBarFragment);
        }
        setAnonymousModeSettingsListener();
    }

    private void setOpenIpFilterFileDialogListener(@NonNull NavBarFragment navBarFragment) {
        navBarFragment.getParentFragmentManager().setFragmentResultListener(
                KEY_CHOOSE_IP_FILTER_FILE_DIALOG_REQUEST,
                this,
                (requestKey, result) -> {
                    FileManagerFragment.Result resultValue = result.getParcelable(FileManagerFragment.KEY_RESULT);
                    if (resultValue == null || resultValue.config().showMode != FileManagerConfig.Mode.FILE_CHOOSER) {
                        return;
                    }
                    var uri = resultValue.uri();
                    if (uri == null) {
                        Snackbar.make(
                                activity,
                                binding.coordinatorLayout,
                                getString(R.string.error_open_ip_filter_file),
                                Snackbar.LENGTH_SHORT
                        ).show();
                    } else {
                        pref.ipFilteringFile(uri.toString());
                        var keyIpFilterFile = getString(R.string.pref_key_ip_filtering_file);
                        var ipFilterFile = findPreference(keyIpFilterFile);
                        if (ipFilterFile != null) {
                            try {
                                ipFilterFile.setSummary(fs.getFilePath(uri));
                            } catch (UnknownUriException e) {
                                Log.e(TAG, Log.getStackTraceString(e));
                            }
                        }
                    }
                }
        );
    }

    private void setAnonymousModeSettingsListener() {
        getParentFragmentManager().setFragmentResultListener(
                KEY_ANONYMOUS_MODE_SETTINGS_REQUEST,
                this,
                (requestKey, result) -> {
                    boolean enabled = result.getBoolean(AnonymousModeSettingsFragment.KEY_RESULT);
                    var anonymousMode = findPreference(getString(R.string.pref_key_anonymous_mode));
                    if (anonymousMode != null) {
                        anonymousMode.setSummary(enabled ? R.string.switch_on : R.string.switch_off);
                    }
                }
        );
    }

    @Override
    public void onCreatePreferences(Bundle savedInstanceState, String rootKey) {
        setPreferencesFromResource(R.xml.pref_network, rootKey);
    }

    private void fileChooseDialog() {
        var config = new FileManagerConfig(
                null,
                null,
                FileManagerConfig.Mode.FILE_CHOOSER);
        List<String> fileTypes = new ArrayList<>();
        fileTypes.add("dat");
        fileTypes.add("p2p");
        config.highlightFileTypes = fileTypes;

        var action = NavBarFragmentDirections.actionOpenFileDialog(
                config,
                KEY_CHOOSE_IP_FILTER_FILE_DIALOG_REQUEST);
        activity.getRootNavController().navigate(action);
    }

    private void bindOnPreferenceChangeListener(Preference preference) {
        preference.setOnPreferenceChangeListener(this);
    }

    @Override
    public boolean onPreferenceChange(Preference preference, Object newValue) {
        if (preference.getKey().equals(getString(R.string.pref_key_port_range_first))) {
            int value = SessionSettings.DEFAULT_PORT_RANGE_FIRST;
            if (!TextUtils.isEmpty((String) newValue))
                value = Integer.parseInt((String) newValue);
            pref.portRangeFirst(value);
            preference.setSummary(Integer.toString(value));

        } else if (preference.getKey().equals(getString(R.string.pref_key_port_range_second))) {
            int value = SessionSettings.DEFAULT_PORT_RANGE_SECOND;
            if (!TextUtils.isEmpty((String) newValue))
                value = Integer.parseInt((String) newValue);
            pref.portRangeSecond(value);
            preference.setSummary(Integer.toString(value));

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_dht))) {
            pref.enableDht((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_lsd))) {
            pref.enableLsd((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_utp))) {
            pref.enableUtp((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_upnp))) {
            pref.enableUpnp((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_natpmp))) {
            pref.enableNatPmp((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_show_nat_errors))) {
            pref.showNatErrors((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_use_random_port))) {
            pref.useRandomPort((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enc_in_connections_mode))) {
            int type = Integer.parseInt((String) newValue);
            pref.encryptInConnectionsMode(type);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enc_out_connections_mode))) {
            int type = Integer.parseInt((String) newValue);
            pref.encryptOutConnectionsMode(type);

        } else if (preference.getKey().equals(getString(R.string.pref_key_enable_ip_filtering))) {
            pref.enableIpFiltering((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_seeding_outgoing_connections))) {
            pref.seedingOutgoingConnections((boolean) newValue);

        } else if (preference.getKey().equals(getString(R.string.pref_key_validate_https_trackers))) {
            pref.validateHttpsTrackers((boolean) newValue);
        }

        return true;
    }
}
