import 'package:flutter/foundation.dart';
import 'package:sodium/sodium.dart' show Sodium;
import 'package:synchronized/synchronized.dart';

import 'sodium_platform.dart';
import 'version_check.dart';

/// Static class to obtain a [Sodium] instance.
///
/// This is a static wrapper around [SodiumPlatform], which simplifies loading
/// the [Sodium] instance and makes sure, the current platform plugin has
/// been correctly loaded. Use [init] to obtain a [Sodium] instance.
abstract class SodiumInit {
  static final _instanceLock = Lock();
  static Sodium? _instance;

  const SodiumInit._(); // coverage:ignore-line

  /// Creates a new [Sodium] instance and initializes it
  ///
  /// Internally, this method ensures the correct [SodiumPlatform] is available
  /// and then uses [SodiumPlatform.loadSodium] to create an instance.
  ///
  /// In addition, when not running in release mode, it also performs a version
  /// check on the library to ensure you are using the correct native binary on
  /// platforms, where the binary is fetched dynamically.
  ///
  /// **Note:** Calling this method multiple times will always return the same
  /// instance.
  static Future<Sodium> init() => _instanceLock.synchronized(() async {
        if (_instance != null) {
          return _instance!;
        }

        _instance = await SodiumPlatform.instance.loadSodium();

        if (kDebugMode) {
          VersionCheck.check(SodiumPlatform.instance, _instance!);
        }

        return _instance!;
      });
}
