package org.primftpd.util;

import android.content.Context;
import android.widget.Toast;

import org.primftpd.R;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

public class IpAddressProvider {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    public static String extractIp(String remoteAddress) {
        String ip = remoteAddress;

        // in some cases we get ip-address with leading /, just cut it off
        if (remoteAddress.charAt(0) == '/') {
            ip = remoteAddress.substring(1);
        }

        // if square brackets are present we can reliably say it is IPv6
        if (ip.charAt(0) == '[') {
            int index = ip.indexOf(']');
            ip = ip.substring(0, index + 1);
            return ip;
        }

        // if length is more than IPv4 dotted quad notation, we guess it would be IPv6
        if (ip.length() > (4*3+3)) {
            return ip;
        }

        // remove port for IPv4
        int indexOfColon = ip.indexOf(':');
        if (indexOfColon > 0) {
            ip = ip.substring(0, indexOfColon);
        }
        return ip;
    }

    public List<String> ipAddressTexts(Context ctxt, boolean verbose, boolean isLeftToRight) {
        List<String> result = new ArrayList<>();
        try {
            Enumeration<NetworkInterface> ifaces = NetworkInterface.getNetworkInterfaces();
            while (ifaces.hasMoreElements()) {
                NetworkInterface iface = ifaces.nextElement();
                String ifaceDispName = iface.getDisplayName();
                String ifaceName = iface.getName();
                Enumeration<InetAddress> inetAddrs = iface.getInetAddresses();

                while (inetAddrs.hasMoreElements()) {
                    InetAddress inetAddr = inetAddrs.nextElement();
                    String hostAddr = inetAddr.getHostAddress();
                    if (hostAddr == null) {
                        hostAddr = "";
                    }

                    logger.debug("addr: '{}', iface name: '{}', disp name: '{}', loopback: '{}'",
                            new Object[]{
                                    inetAddr,
                                    ifaceName,
                                    ifaceDispName,
                                    inetAddr.isLoopbackAddress()});

                    if (inetAddr.isLoopbackAddress()) {
                        continue;
                    }
                    if (inetAddr.isAnyLocalAddress()) {
                        continue;
                    }

                    if (hostAddr.contains("::")) {
                        // Don't include the raw encoded names. Just the raw IP addresses.
                        logger.debug("Skipping IPv6 address '{}'", hostAddr);
                        continue;
                    }

                    if (!verbose && !ifaceDispName.startsWith("wlan")) {
                        continue;
                    }

                    String displayText = hostAddr;
                    if (verbose) {
                        String verboseText =  "(" + ifaceDispName + ")";
                        if (isLeftToRight) {
                            displayText += " " + verboseText;
                        } else {
                            displayText = verboseText + " " + displayText;
                        }
                    }

                    result.add(displayText);
                }
            }
        } catch (SocketException e) {
            logger.info("exception while iterating network interfaces", e);

            String msg = ctxt.getText(R.string.ifacesError) + e.getLocalizedMessage();
            Toast.makeText(ctxt, msg, Toast.LENGTH_SHORT).show();
        }
        return result;
    }

    public boolean isIpAvail(String ip) {
        if (ip == null) {
            return true;
        }
        try {
            Enumeration<NetworkInterface> ifaces = NetworkInterface.getNetworkInterfaces();
            while (ifaces.hasMoreElements()) {
                NetworkInterface iface = ifaces.nextElement();
                Enumeration<InetAddress> inetAddrs = iface.getInetAddresses();

                while (inetAddrs.hasMoreElements()) {
                    InetAddress inetAddr = inetAddrs.nextElement();
                    String hostAddr = inetAddr.getHostAddress();
                    if (ip.equals(hostAddr)) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            logger.info("could not check ip", e);
        }
        return false;
    }

    public boolean isIpv6(String address) {
        return address.contains(":");
    }
}
