/*
 * Copyright (C) 2017 Moez Bhatti <moez.bhatti@gmail.com>
 *
 * This file is part of QKSMS.
 *
 * QKSMS is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QKSMS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QKSMS.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.prauga.messages.feature.compose.part

import android.content.Context
import android.content.res.ColorStateList
import android.media.AudioAttributes
import android.media.MediaMetadataRetriever
import android.view.View
import android.widget.SeekBar
import com.moez.QKSMS.common.QkMediaPlayer
import org.prauga.messages.R
import org.prauga.messages.common.Navigator
import org.prauga.messages.common.base.QkViewHolder
import org.prauga.messages.common.util.Colors
import org.prauga.messages.common.util.extensions.resolveThemeColor
import org.prauga.messages.common.util.extensions.setBackgroundTint
import org.prauga.messages.common.util.extensions.setTint
import org.prauga.messages.common.util.extensions.withAlpha
import org.prauga.messages.common.widget.BubbleImageView
import org.prauga.messages.databinding.MmsAudioPreviewListItemBinding
import org.prauga.messages.extensions.isAudio
import org.prauga.messages.extensions.resourceExists
import org.prauga.messages.feature.compose.MessagesAdapter
import org.prauga.messages.model.Message
import org.prauga.messages.model.MmsPart
import org.prauga.messages.util.GlideApp
import io.reactivex.Observable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.schedulers.Schedulers
import java.util.concurrent.TimeUnit
import javax.inject.Inject


class AudioBinder @Inject constructor(colors: Colors, private val context: Context) :
    PartBinder() {

    companion object {
        const val DEFAULT_SHARE_FILENAME = "quik-audio-attachment.mp3"
    }

    @Inject lateinit var navigator: Navigator

    override val partLayout = R.layout.mms_audio_preview_list_item
    override var theme = colors.theme()

    override fun canBindPart(part: MmsPart) = part.isAudio()

    var audioState = MessagesAdapter.AudioState(-1, QkMediaPlayer.PlayingState.Stopped)

    private fun startSeekBarUpdateTimer() {
        audioState.apply {
            seekBarUpdater?.dispose()
            seekBarUpdater = Observable.interval(500, TimeUnit.MILLISECONDS)
                .subscribeOn(Schedulers.single())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnNext {
                    viewHolder?.containerView
                        ?.findViewById<SeekBar>(R.id.seekBar)
                        ?.progress = QkMediaPlayer.currentPosition
                }
                .subscribe()
        }
    }

    private fun uiToPlaying(viewHolder: QkViewHolder) {
        val binding = MmsAudioPreviewListItemBinding.bind(viewHolder.containerView)
        binding.seekBar.max = QkMediaPlayer.duration
        binding.seekBar.isEnabled = true
        binding.seekBar.progress = QkMediaPlayer.currentPosition
        binding.playPause.setImageResource(R.drawable.exo_icon_pause)
        binding.playPause.tag = QkMediaPlayer.PlayingState.Playing
        binding.metadataTitle.isSelected = true     // start marquee
    }

    private fun uiToPaused(viewHolder: QkViewHolder) {
        val binding = MmsAudioPreviewListItemBinding.bind(viewHolder.containerView)
        binding.playPause.setImageResource(R.drawable.exo_icon_play)
        binding.playPause.tag = QkMediaPlayer.PlayingState.Paused
    }

    private fun uiToStopped(viewHolder: QkViewHolder) {
        val binding = MmsAudioPreviewListItemBinding.bind(viewHolder.containerView)
        binding.seekBar.progress = 0
        binding.seekBar.max = 0
        binding.seekBar.isEnabled = false
        binding.playPause.setImageResource(R.drawable.exo_icon_play)
        binding.playPause.tag = QkMediaPlayer.PlayingState.Stopped
        binding.metadataTitle.isSelected = false   // stop marquee
    }

    override fun bindPart(
        holder: QkViewHolder,
        part: MmsPart,
        message: Message,
        canGroupWithPrevious: Boolean,
        canGroupWithNext: Boolean,
    ) {
        val binding = MmsAudioPreviewListItemBinding.bind(holder.containerView)

        // click on background - passes back to compose view model
        holder.containerView.setOnClickListener { clicks.onNext(part.id) }

        // play/pause button click handling
        binding.playPause.setOnClickListener {
            when (binding.playPause.tag) {
                QkMediaPlayer.PlayingState.Playing -> {
                    if (audioState.partId == part.id) {
                        QkMediaPlayer.pause()
                        uiToPaused(holder)
                        audioState.state = QkMediaPlayer.PlayingState.Paused

                        // stop progress bar update timer
                        audioState.seekBarUpdater?.dispose()
                    }
                }
                QkMediaPlayer.PlayingState.Paused -> {
                    if (audioState.partId == part.id) {
                        QkMediaPlayer.start()
                        uiToPlaying(holder)
                        audioState.state = QkMediaPlayer.PlayingState.Playing

                        // start progress bar update timer
                        startSeekBarUpdateTimer()
                    }
                }
                else -> {
                    if (part.getUri().resourceExists(context)) {
                        QkMediaPlayer.reset() // make sure reset before trying to (re-)use

                        QkMediaPlayer.setOnPreparedListener {
                            // start media playing
                            QkMediaPlayer.start()

                            uiToPlaying(holder)

                            // set current view holder and part as active
                            audioState.apply {
                                audioState.state = QkMediaPlayer.PlayingState.Playing
                                partId = part.id
                                viewHolder = holder
                            }

                            // start progress bar update timer
                            startSeekBarUpdateTimer()
                        }

                        QkMediaPlayer.setOnCompletionListener {   // also called on error because we don't have an onerrorlistener
                            audioState.apply {
                                // if this part is currently active, set it to stopped and inactive
                                if ((partId == part.id) && (viewHolder != null))
                                    uiToStopped(viewHolder!!)

                                state = QkMediaPlayer.PlayingState.Stopped
                                partId = -1
                                viewHolder = null
                            }
                        }

                        // start the media player play sequence
                        QkMediaPlayer.setAudioAttributes(
                            AudioAttributes.Builder()
                                .setContentType(AudioAttributes.CONTENT_TYPE_MUSIC)     // music, maybe?? could be voice. don't want to use CONTENT_TYPE_UNKNOWN though
                                .setUsage(AudioAttributes.USAGE_MEDIA)
                                .build()
                        )

                        QkMediaPlayer.setDataSource(context, part.getUri())

                        QkMediaPlayer.prepareAsync()
                    }
                }
            }
        }

        // if this item is the active active audio item update the active view holder
        if (audioState.partId == part.id)
            audioState.viewHolder = holder
        // else, this is not the active item so ensure the stored view holder is not this one
        else if (audioState.viewHolder == holder)
            audioState.viewHolder = null

        // tint colours
        val secondaryColor =
            if (!message.isMe())
                theme.theme
            else
                holder.containerView.context.resolveThemeColor(R.attr.bubbleColor)
        val primaryColor =
            if (!message.isMe())
                theme.textPrimary
            else
                holder.containerView.context.resolveThemeColor(android.R.attr.textColorPrimary)

        // sound wave
        binding.soundWave.setTint(primaryColor)

        // seek bar
        binding.seekBar.apply {
            setTint(secondaryColor)
            thumbTintList = ColorStateList.valueOf(secondaryColor)

            setOnSeekBarChangeListener(object : SeekBar.OnSeekBarChangeListener {
                override fun onProgressChanged(p0: SeekBar?, progress: Int, fromUser: Boolean) {
                    // if seek was initiated by the user and this part is currently playing
                    if (fromUser)
                        QkMediaPlayer.seekTo(progress)
                }
                override fun onStartTrackingTouch(p0: SeekBar?) { /* nothing */ }
                override fun onStopTrackingTouch(p0: SeekBar?) { /* nothing */ }
            })
        }

        // playPause button
        binding.playPause. apply {
            if ((audioState.partId == part.id) &&
                (audioState.state == QkMediaPlayer.PlayingState.Playing))
                uiToPlaying(holder)
            else if ((audioState.partId == part.id) &&
                (audioState.state == QkMediaPlayer.PlayingState.Paused))
                uiToPaused(holder)
            else
                uiToStopped(holder)

            setTint(secondaryColor)
            setBackgroundTint(primaryColor)
        }

        MediaMetadataRetriever().apply {
            if (part.getUri().resourceExists(context))
                setDataSource(context, part.getUri())

            // metadata title
            binding.metadataTitle.apply {
                text = extractMetadata(MediaMetadataRetriever.METADATA_KEY_TITLE)

                if (text.isEmpty())
                    visibility = View.GONE
                else {
                    visibility = View.VISIBLE
                    setTextColor(primaryColor)
                    setBackgroundTint(secondaryColor.withAlpha(0xcc))    // hex value is alpha
                }
            }

            // bubble / embedded image
            binding.thumbnail.apply {
                bubbleStyle = when {
                    !canGroupWithPrevious && canGroupWithNext ->
                        if (message.isMe()) BubbleImageView.Style.OUT_FIRST else BubbleImageView.Style.IN_FIRST
                    canGroupWithPrevious && canGroupWithNext ->
                        if (message.isMe()) BubbleImageView.Style.OUT_MIDDLE else BubbleImageView.Style.IN_MIDDLE
                    canGroupWithPrevious && !canGroupWithNext ->
                        if (message.isMe()) BubbleImageView.Style.OUT_LAST else BubbleImageView.Style.IN_LAST
                    else -> BubbleImageView.Style.ONLY
                }

                val embeddedPicture = embeddedPicture
                if (embeddedPicture == null) {
                    binding.frame.layoutParams.height = (binding.frame.layoutParams.width / 2)
                    setTint(secondaryColor)
                    setImageResource(R.drawable.rectangle)
                } else {
                    binding.frame.layoutParams.height = binding.frame.layoutParams.width
                    setTint(null)
                    GlideApp.with(context)
                        .asBitmap()
                        .load(embeddedPicture)
                        .override(
                            binding.frame.layoutParams.width,
                            binding.frame.layoutParams.height
                        )
                        .into(this)
                }
            }

            release()
        }
    }
}
