/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package androidx.media3.exoplayer.e2etest;

import static com.google.common.truth.Truth.assertThat;
import static org.robolectric.annotation.GraphicsMode.Mode.NATIVE;

import android.content.Context;
import android.graphics.SurfaceTexture;
import android.net.Uri;
import android.view.Surface;
import androidx.media3.common.C;
import androidx.media3.common.MediaItem;
import androidx.media3.common.MimeTypes;
import androidx.media3.common.Player;
import androidx.media3.common.util.Clock;
import androidx.media3.exoplayer.ExoPlayer;
import androidx.media3.test.utils.CapturingRenderersFactory;
import androidx.media3.test.utils.DumpFileAsserts;
import androidx.media3.test.utils.FakeClock;
import androidx.media3.test.utils.robolectric.PlaybackOutput;
import androidx.media3.test.utils.robolectric.ShadowMediaCodecConfig;
import androidx.media3.test.utils.robolectric.TestPlayerRunHelper;
import androidx.test.core.app.ApplicationProvider;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import com.google.common.collect.ImmutableList;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.robolectric.annotation.GraphicsMode;

/** End-to-end tests for playlists. */
@RunWith(AndroidJUnit4.class)
@GraphicsMode(value = NATIVE)
public final class PlaylistPlaybackTest {

  @Rule
  public ShadowMediaCodecConfig mediaCodecConfig =
      ShadowMediaCodecConfig.forAllSupportedMimeTypes();

  @Test
  public void test_bypassOnThenOff() throws Exception {
    Context applicationContext = ApplicationProvider.getApplicationContext();
    CapturingRenderersFactory capturingRenderersFactory =
        new CapturingRenderersFactory(applicationContext);
    ExoPlayer player =
        new ExoPlayer.Builder(applicationContext, capturingRenderersFactory)
            .setClock(new FakeClock(/* isAutoAdvancing= */ true))
            .build();
    PlaybackOutput playbackOutput = PlaybackOutput.register(player, capturingRenderersFactory);

    player.addMediaItem(MediaItem.fromUri("asset:///media/wav/sample.wav"));
    player.addMediaItem(MediaItem.fromUri("asset:///media/mka/bear-opus.mka"));
    player.prepare();
    player.play();
    TestPlayerRunHelper.runUntilPlaybackState(player, Player.STATE_ENDED);
    player.release();

    DumpFileAsserts.assertOutput(
        applicationContext, playbackOutput, "playbackdumps/playlists/bypass-on-then-off.dump");
  }

  @Test
  public void test_bypassOffThenOn() throws Exception {
    Context applicationContext = ApplicationProvider.getApplicationContext();
    CapturingRenderersFactory capturingRenderersFactory =
        new CapturingRenderersFactory(applicationContext);
    ExoPlayer player =
        new ExoPlayer.Builder(applicationContext, capturingRenderersFactory)
            .setClock(new FakeClock(/* isAutoAdvancing= */ true))
            .build();
    PlaybackOutput playbackOutput = PlaybackOutput.register(player, capturingRenderersFactory);

    player.addMediaItem(MediaItem.fromUri("asset:///media/mka/bear-opus.mka"));
    player.prepare();
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ true);
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ false);
    player.addMediaItem(MediaItem.fromUri("asset:///media/wav/sample.wav"));
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ true);
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ false);
    // Wait until second period has fully loaded to start the playback.
    player.play();
    TestPlayerRunHelper.runUntilPlaybackState(player, Player.STATE_ENDED);
    player.release();

    DumpFileAsserts.assertOutput(
        applicationContext, playbackOutput, "playbackdumps/playlists/bypass-off-then-on.dump");
  }

  @Test
  public void test_subtitle() throws Exception {
    Context applicationContext = ApplicationProvider.getApplicationContext();
    CapturingRenderersFactory capturingRenderersFactory =
        new CapturingRenderersFactory(applicationContext);
    ExoPlayer player =
        new ExoPlayer.Builder(applicationContext, capturingRenderersFactory)
            .setClock(new FakeClock(/* isAutoAdvancing= */ true))
            .build();
    Surface surface = new Surface(new SurfaceTexture(/* texName= */ 1));
    player.setVideoSurface(surface);
    PlaybackOutput playbackOutput = PlaybackOutput.register(player, capturingRenderersFactory);

    player.addMediaItem(MediaItem.fromUri("asset:///media/mp4/preroll-5s.mp4"));
    player.prepare();
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ true);
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ false);
    MediaItem mediaItemWithSubtitle =
        new MediaItem.Builder()
            .setUri("asset:///media/mp4/preroll-5s.mp4")
            .setSubtitleConfigurations(
                ImmutableList.of(
                    new MediaItem.SubtitleConfiguration.Builder(
                            Uri.parse("asset:///media/webvtt/typical"))
                        .setMimeType(MimeTypes.TEXT_VTT)
                        .setLanguage("en")
                        .setSelectionFlags(C.SELECTION_FLAG_DEFAULT)
                        .build()))
            .build();
    player.addMediaItem(mediaItemWithSubtitle);
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ true);
    TestPlayerRunHelper.runUntilIsLoading(player, /* expectedIsLoading= */ false);
    // Wait until second period has fully loaded to start the playback.
    player.play();
    TestPlayerRunHelper.runUntilPlaybackState(player, Player.STATE_ENDED);
    player.release();
    surface.release();

    DumpFileAsserts.assertOutput(
        applicationContext, playbackOutput, "playbackdumps/playlists/playlist_with_subtitles.dump");
  }

  @Test
  public void testPlaylist_withImageAndAudioVideoItems_rendersExpectedContent() throws Exception {
    Context applicationContext = ApplicationProvider.getApplicationContext();
    CapturingRenderersFactory renderersFactory = new CapturingRenderersFactory(applicationContext);
    Clock clock = new FakeClock(/* isAutoAdvancing= */ true);
    ExoPlayer player =
        new ExoPlayer.Builder(applicationContext, renderersFactory).setClock(clock).build();
    PlaybackOutput playbackOutput = PlaybackOutput.register(player, renderersFactory);
    long durationMs = 5 * C.MILLIS_PER_SECOND;
    player.setMediaItems(
        ImmutableList.of(
            new MediaItem.Builder()
                .setUri("asset:///media/png/media3test.png")
                .setImageDurationMs(durationMs)
                .build(),
            new MediaItem.Builder()
                .setUri("asset:///media/png/media3test.png")
                .setImageDurationMs(durationMs)
                .build(),
            MediaItem.fromUri("asset:///media/mp4/sample.mp4")));
    player.prepare();

    TestPlayerRunHelper.runUntilPlaybackState(player, Player.STATE_READY);
    long playerStartedMs = clock.elapsedRealtime();
    player.play();
    TestPlayerRunHelper.runUntilPlaybackState(player, Player.STATE_ENDED);
    long playbackDurationMs = clock.elapsedRealtime() - playerStartedMs;
    player.release();

    // Playback duration should be greater than the sum of the image item durations.
    assertThat(playbackDurationMs).isGreaterThan(durationMs * 2);
    DumpFileAsserts.assertOutput(
        applicationContext, playbackOutput, "playbackdumps/playlists/image_av_playlist.dump");
  }
}
