/*
 * Copyright 2023 The Android Open Source Project
 * Copyright 2009 Sonic Network Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package androidx.media3.decoder.midi;

import static androidx.media3.common.util.Assertions.checkArgument;
import static androidx.media3.common.util.Assertions.checkStateNotNull;
import static java.lang.Math.max;

import android.content.Context;
import com.google.common.io.BaseEncoding;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

/** General MIDI Instrument library extracted from Sonivox library in Android file "wt_22khz.c". */
/* package */ final class SonivoxWaveData {

  public static class WavetableRegion {
    private static final int REGION_FLAG_IS_LOOPED = 0x01;
    private static final int REGION_FLAG_USE_NOISE_GENERATOR = 0x02;
    private static final int REGION_FLAG_IS_LAST = 0x8000;

    public final int loopStart;
    public final int loopEnd;
    public final int waveIndex;
    public final int artIndex;
    public final int keyGroupAndFlags;
    public final byte rangeLow;
    public final byte rangeHigh;
    public final short tuning;
    public final short gain;

    public WavetableRegion(
        int loopStart,
        int loopEnd,
        int waveIndex,
        int artIndex,
        int keyGroupAndFlags,
        byte rangeLow,
        byte rangeHigh,
        short tuning,
        short gain) {
      this.loopStart = loopStart;
      this.loopEnd = loopEnd;
      this.waveIndex = waveIndex;
      this.artIndex = artIndex;
      this.keyGroupAndFlags = keyGroupAndFlags;
      this.rangeLow = rangeLow;
      this.rangeHigh = rangeHigh;
      this.tuning = tuning;
      this.gain = gain;
    }

    public boolean useNoise() {
      return (keyGroupAndFlags & REGION_FLAG_USE_NOISE_GENERATOR) != 0;
    }

    public boolean isLooped() {
      return (keyGroupAndFlags & REGION_FLAG_IS_LOOPED) != 0;
    }

    public boolean isLast() {
      return (keyGroupAndFlags & REGION_FLAG_IS_LAST) != 0;
    }

    public boolean isNoteInRange(int noteNumber) {
      return rangeLow <= noteNumber && rangeHigh >= noteNumber;
    }
  }

  private static final int REGION_NUM_VALUES = 9;
  private static final int ARTICULATION_NUM_VALUES = 17;
  private static final int SONIVOX_SAMPLE_RATE = 22050;

  /**
   * Loads the wave table data from file {@code sonivox_byte_wave.dat} which must be bundled in
   * assets.
   *
   * @param context The application context.
   * @return The wave table data.
   * @throws MidiDecoderException If there is an error while loading the file.
   */
  public static short[] loadWaveTableData(Context context) throws MidiDecoderException {
    byte[] bytes = loadWaveTableFile(context);
    // Use last offset and size to allocate the sample array. It is not clear why but Sonivox does
    // this.
    int numBytesWaveSizes =
        WAVE_OFFSETS[WAVE_OFFSETS.length - 1] + WAVES_SIZES[WAVES_SIZES.length - 1];
    int numSamples = max(bytes.length, numBytesWaveSizes);
    short[] waveTableData = new short[numSamples];
    for (int i = 0; i < bytes.length; i++) {
      // Extend to 16-bits.
      waveTableData[i] = (short) (bytes[i] << 8);
    }
    return waveTableData;
  }

  private static byte[] loadWaveTableFile(Context context) throws MidiDecoderException {
    try (InputStream inputStream =
        checkStateNotNull(context.getAssets().open("sonivox_wave_data.dat"))) {
      StringBuilder fileContents = new StringBuilder();
      BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
      String line;
      while ((line = reader.readLine()) != null) {
        if (line.startsWith("/") || line.startsWith(" ")) {
          continue;
        }
        fileContents.append(line);
      }
      return BaseEncoding.base16().decode(fileContents.toString());
    } catch (IOException e) {
      throw new MidiDecoderException("Could not load wave table data", e);
    }
  }

  public static String[] getProgramNames() {
    return new String[] {
      // Index 0 is called program "1" by musicians.
      /* 1 */ "Acoustic Grand",
      /* 2 */ "Bright Acoustic",
      /* 3 */ "Electric Grand",
      /* 4 */ "Honky-Tonk",
      /* 5 */ "Electric Piano 1",
      /* 6 */ "Electric Piano 2",
      /* 7 */ "Harpsichord",
      /* 8 */ "Clav",
      /* 9 */ "Celesta",
      /* 10 */ "Glockenspiel",
      /* 11 */ "Music Box",
      /* 12 */ "Vibraphone",
      /* 13 */ "Marimba",
      /* 14 */ "Xylophone",
      /* 15 */ "Tubular Bells",
      /* 16 */ "Dulcimer",
      /* 17 */ "Drawbar Organ",
      /* 18 */ "Percussive Organ",
      /* 19 */ "Rock Organ",
      /* 20 */ "Church Organ",
      /* 21 */ "Reed Organ",
      /* 22 */ "Accordian",
      /* 23 */ "Harmonica",
      /* 24 */ "Tango Accordian",
      /* 25 */ "Acoustic Guitar(nylon)",
      /* 26 */ "Acoustic Guitar(steel)",
      /* 27 */ "Electric Guitar(jazz)",
      /* 28 */ "Electric Guitar(clean)",
      /* 29 */ "Electric Guitar(muted)",
      /* 30 */ "Overdriven Guitar",
      /* 31 */ "Distortion Guitar",
      /* 32 */ "Guitar Harmonics",
      /* 33 */ "Acoustic Bass",
      /* 34 */ "Electric Bass(finger)",
      /* 35 */ "Electric Bass(pick)",
      /* 36 */ "Fretless Bass",
      /* 37 */ "Slap Bass 1",
      /* 38 */ "Slap Bass 2",
      /* 39 */ "Synth Bass 1",
      /* 40 */ "Synth Bass 2",
      /* 41 */ "Violin",
      /* 42 */ "Viola",
      /* 43 */ "Cello",
      /* 44 */ "Contrabass",
      /* 45 */ "Tremolo Strings",
      /* 46 */ "Pizzicato Strings",
      /* 47 */ "Orchestral Harp",
      /* 48 */ "Timpani",
      /* 49 */ "String Ensembles 1",
      /* 50 */ "String Ensembles 2",
      /* 51 */ "SynthStrings 1",
      /* 52 */ "SynthStrings 2",
      /* 53 */ "Choir Aahs",
      /* 54 */ "Voice Oohs",
      /* 55 */ "Synth Voice",
      /* 56 */ "Orchestra Hit",
      /* 57 */ "Trumpet",
      /* 58 */ "Trombone",
      /* 59 */ "Tuba",
      /* 60 */ "Muted Trumpet",
      /* 61 */ "French Horn",
      /* 62 */ "Brass Section",
      /* 63 */ "SynthBrass 1",
      /* 64 */ "SynthBrass 2",
      /* 65 */ "Soprano Sax",
      /* 66 */ "Alto Sax",
      /* 67 */ "Tenor Sax",
      /* 68 */ "Baritone Sax",
      /* 69 */ "Oboe",
      /* 70 */ "English Horn",
      /* 71 */ "Bassoon",
      /* 72 */ "Clarinet",
      /* 73 */ "Piccolo",
      /* 74 */ "Flute",
      /* 75 */ "Recorder",
      /* 76 */ "Pan Flute",
      /* 77 */ "Blown Bottle",
      /* 78 */ "Skakuhachi",
      /* 79 */ "Whistle",
      /* 80 */ "Ocarina",
      /* 81 */ "Lead 1 (square)",
      /* 82 */ "Lead 2 (sawtooth)",
      /* 83 */ "Lead 3 (calliope)",
      /* 84 */ "Lead 4 (chiff)",
      /* 85 */ "Lead 5 (charang)",
      /* 86 */ "Lead 6 (voice)",
      /* 87 */ "Lead 7 (fifths)",
      /* 88 */ "Lead 8 (bass+lead)",
      /* 89 */ "Pad 1 (new age)",
      /* 90 */ "Pad 2 (warm)",
      /* 91 */ "Pad 3 (polysynth)",
      /* 92 */ "Pad 4 (choir)",
      /* 93 */ "Pad 5 (bowed)",
      /* 94 */ "Pad 6 (metallic)",
      /* 95 */ "Pad 7 (halo)",
      /* 96 */ "Pad 8 (sweep)",
      /* 97 */ "FX 1 (rain)",
      /* 98 */ "FX 2 (soundtrack)",
      /* 99 */ "FX 3 (crystal)",
      /* 100 */ "FX 4 (atmosphere)",
      /* 101 */ "FX 5 (brightness)",
      /* 102 */ "FX 6 (goblins)",
      /* 103 */ "FX 7 (echoes)",
      /* 104 */ "FX 8 (sci-fi)",
      /* 105 */ "Sitar",
      /* 106 */ "Banjo",
      /* 107 */ "Shamisen",
      /* 108 */ "Koto",
      /* 109 */ "Kalimba",
      /* 110 */ "Bagpipe",
      /* 111 */ "Fiddle",
      /* 112 */ "Shanai",
      /* 113 */ "Tinkle Bell",
      /* 114 */ "Agogo",
      /* 115 */ "Steel Drums",
      /* 116 */ "Woodblock",
      /* 117 */ "Taiko Drum",
      /* 118 */ "Melodic Tom",
      /* 119 */ "Synth Drum",
      /* 120 */ "Reverse Cymbal",
      /* 121 */ "Guitar Fret Noise",
      /* 122 */ "Breath Noise",
      /* 123 */ "Seashore",
      /* 124 */ "Bird Tweet",
      /* 125 */ "Telephone Ring",
      /* 126 */ "Helicopter",
      /* 127 */ "Applause",
      /* 128 */ "Gunshot",
      /* 0x80 is special program for GM2 drum channels */ "Rhythm"
    };
  }

  public static WavetableRegion extractRegion(int index) {
    int cursor = index * REGION_NUM_VALUES;
    int keyGroupAndFlags = EAS_REGIONS[cursor++];
    byte rangeLow = (byte) EAS_REGIONS[cursor++];
    byte rangeHigh = (byte) EAS_REGIONS[cursor++];
    short tuning = (short) EAS_REGIONS[cursor++];
    short gain = (short) EAS_REGIONS[cursor++];
    int loopStart = EAS_REGIONS[cursor++];
    int loopEnd = EAS_REGIONS[cursor++];
    int waveIndex = EAS_REGIONS[cursor++];
    int artIndex = EAS_REGIONS[cursor];
    return new WavetableRegion(
        loopStart,
        loopEnd,
        waveIndex,
        artIndex,
        keyGroupAndFlags,
        rangeLow,
        rangeHigh,
        tuning,
        gain);
  }

  public static class Envelope {
    public final short attackTime; // Add to gain every 128 samples, 0 to 32767
    public final short decayTime; // 1.15 scaler for gain every 128 samples
    public final short sustainLevel; // Gain from 0 to 32767
    public final short releaseTime; // 1.15 scaler for gain every 128 samples

    public Envelope(short attackTime, short decayTime, short sustainLevel, short releaseTime) {
      checkArgument(attackTime != 0); // used later as denominator
      this.attackTime = attackTime;
      this.decayTime = decayTime;
      this.sustainLevel = sustainLevel;
      this.releaseTime = releaseTime;
    }

    /**
     * From Sonivox Code: attackTime is the fixed-point integer value from the Sonivox instrument.
     * (attackTime/32768) * (T*SR/128) = 1.0 // 0 to full scale in T seconds (T*SR/128) = 1.0 /
     * (attackTime/32768) T = 128 * 32768 / SR * attackTime
     */
    public double getAttackTimeInSeconds() {
      double numerator = 128L * 32768;
      int denominator = SONIVOX_SAMPLE_RATE * attackTime;
      double time = numerator / denominator;
      // Do not use less than 5 msec because it will be too clicky.
      return max(0.005, time);
    }

    public double getDecayTimeInSeconds() {
      return convertScalerToSeconds(decayTime);
    }

    public double getReleaseTimeInSeconds() {
      return max(0.02, convertScalerToSeconds(releaseTime));
    }

    public double getSustainLevel() {
      return sustainLevel * (1.0 / 32768);
    }

    /**
     * Based on a reverse engineering of the Sonivox code. (decayTime / 32768)^(T*SR/128) = -90dB
     * log((decayTime / 32768)^(T*SR/128)) = log(DB90) (T*SR/128) * log(decayTime / 32768) =
     * log(DB90) (T*SR/128) * log(decayTime / 32768) = log(1.0/32768) T = log(1.0/32768) * 128 / (SR
     * * log(decayTime/32768)
     */
    private static double convertScalerToSeconds(int scaler) {
      double numerator = Math.log(1.0 / 32768) * 128;
      double denominator = SONIVOX_SAMPLE_RATE * Math.log(scaler / 32768.0);
      return numerator / denominator;
    }
  }

  public static class Articulation {
    public final Envelope eg1;
    public final Envelope eg2;
    public final short lfoToPitch; // pitchCents += lfo * lfoToPitch / 32768.0
    public final short lfoDelay; // T = lfoDelay * 128 / SR
    public final short lfoFreq; // phaseIncrement = lfoFreq*128 from 0 to 32767
    public final short eg2ToPitch; // pitchCents += eg2 * eg2ToPitch / 32768.0
    public final short eg2ToFc; // pitchCents += eg2 * eg2ToFc / 32768.0
    public final short filterCutoff; // pitchCents
    public final byte lfoToGain; // logarithmic
    public final short filterQ;
    public final byte pan;

    public Articulation(
        Envelope eg1,
        Envelope eg2,
        short lfoToPitch,
        short lfoDelay,
        short lfoFreq,
        short eg2ToPitch,
        short eg2ToFc,
        short filterCutoff,
        byte lfoToGain,
        short filterQ,
        byte pan) {
      this.eg1 = eg1;
      this.eg2 = eg2;
      this.lfoToPitch = lfoToPitch;
      this.lfoDelay = lfoDelay;
      this.lfoFreq = lfoFreq;
      this.eg2ToPitch = eg2ToPitch;
      this.eg2ToFc = eg2ToFc;
      this.filterCutoff = filterCutoff;
      this.lfoToGain = lfoToGain;
      this.filterQ = filterQ;
      this.pan = pan;
    }
  }

  public static Articulation extractArticulation(int index) {
    int cursor = index * ARTICULATION_NUM_VALUES;
    // Extraction order must match the order of the members of the structure.
    return new Articulation(
        /* eg1= */ new Envelope(
            /* attackTime= */ EAS_ARTICULATIONS[cursor++],
            /* decayTime= */ EAS_ARTICULATIONS[cursor++],
            /* sustainLevel= */ EAS_ARTICULATIONS[cursor++],
            /* releaseTime= */ EAS_ARTICULATIONS[cursor++]),
        /* eg2= */ new Envelope(
            /* attackTime= */ EAS_ARTICULATIONS[cursor++],
            /* decayTime= */ EAS_ARTICULATIONS[cursor++],
            /* sustainLevel= */ EAS_ARTICULATIONS[cursor++],
            /* releaseTime= */ EAS_ARTICULATIONS[cursor++]),
        /* lfoToPitch= */ EAS_ARTICULATIONS[cursor++],
        /* lfoDelay= */ EAS_ARTICULATIONS[cursor++],
        /* lfoFreq= */ EAS_ARTICULATIONS[cursor++],
        /* eg2ToPitch= */ EAS_ARTICULATIONS[cursor++],
        /* eg2ToFc= */ EAS_ARTICULATIONS[cursor++],
        /* filterCutoff= */ EAS_ARTICULATIONS[cursor++],
        /* lfoToGain= */ (byte) EAS_ARTICULATIONS[cursor++],
        /* filterQ= */ EAS_ARTICULATIONS[cursor++],
        /* pan= */ (byte) EAS_ARTICULATIONS[cursor]);
  }

  public static int getProgramRegion(int index) {
    return PROGRAM_REGIONS[index];
  }

  private static final int[] PROGRAM_REGIONS = {
    291, 324, 314, 334, 202, 319, 95, 195, // 0
    107, 92, 371, 89, 87, 85, 135, 82, // 8
    200, 192, 130, 267, 193, 302, 207, 210,
    128, 125, 190, 120, 118, 213, 221, 271,
    80, 78, 308, 164, 220, 310, 166, 167, // 32
    186, 182, 181, 179, 160, 178, 176, 115,
    155, 153, 151, 149, 75, 73, 374, 111,
    252, 254, 258, 305, 256, 157, 146, 137,
    249, 237, 245, 241, 274, 262, 260, 265, // 64
    172, 171, 309, 277, 284, 307, 136, 344,
    173, 168, 345, 353, 346, 70, 110, 311,
    357, 144, 104, 67, 364, 367, 64, 288,
    142, 140, 98, 355, 133, 123, 61, 113, // 96
    285, 280, 279, 278, 370, 286, 359, 283,
    101, 236, 163, 235, 234, 233, 232, 231,
    162, 363, 230, 281, 165, 229, 109, 228,
    0 // start of rhythm regions
  };

  public static int getWaveSize(int index) {
    return WAVES_SIZES[index];
  }

  private static final int[] WAVES_SIZES = {
    16820, 16708, 16592, 11754, 10954, 10295, 9922, 7489,
    5779, 5462, 4452, 3779, 3115, 3093, 3057, 3024,
    2818, 2776, 2171, 2168, 2052, 1902, 1835, 1614,
    1603, 1528, 1517, 1480, 1455, 1424, 1387, 1302,
    1262, 1254, 1230, 1227, 1185, 1181, 1178, 1168,
    1132, 1120, 1034, 1033, 1018, 994, 964, 926,
    907, 886, 881, 866, 830, 817, 816, 813,
    749, 748, 739, 720, 652, 610, 610, 583,
    564, 561, 556, 549, 542, 535, 530, 530,
    516, 508, 492, 478, 461, 448, 437, 431,
    423, 418, 403, 402, 400, 394, 387, 387,
    367, 357, 347, 347, 341, 336, 334, 329,
    325, 312, 294, 284, 277, 265, 255, 233,
    230, 213, 207, 205, 194, 193, 184, 181,
    181, 167, 164, 158, 152, 152, 145, 139,
    128, 103, 100, 88, 87, 84, 84, 72,
    71, 55, 46, 45, 43, 40, 40, 40,
    37, 35, 32, 32, 30, 29, 27, 23,
    22, 21, 21, 21, 21, 20
  };

  public static int getWaveCount() {
    return WAVE_OFFSETS.length;
  }

  public static int getWaveOffset(int index) {
    return WAVE_OFFSETS[index];
  }

  private static final int[] WAVE_OFFSETS = {
    0x00000000,
    0x000041b4,
    0x000082f8,
    0x0000c3c8,
    0x0000f1b2,
    0x00011c7c,
    0x000144b3,
    0x00016b75, // 0
    0x000188b6,
    0x00019f49,
    0x0001b49f,
    0x0001c603,
    0x0001d4c6,
    0x0001e0f1,
    0x0001ed06,
    0x0001f8f7, // 8
    0x000204c7,
    0x00020fc9,
    0x00021aa1,
    0x0002231c,
    0x00022b94,
    0x00023398,
    0x00023b06,
    0x00024231, // 16
    0x0002487f,
    0x00024ec2,
    0x000254ba,
    0x00025aa7,
    0x0002606f,
    0x0002661e,
    0x00026bae,
    0x00027119,
    0x0002762f,
    0x00027b1d,
    0x00028003,
    0x000284d1,
    0x0002899c,
    0x00028e3d,
    0x000292da,
    0x00029774, // 32
    0x00029c04,
    0x0002a070,
    0x0002a4d0,
    0x0002a8da,
    0x0002ace3,
    0x0002b0dd,
    0x0002b4bf,
    0x0002b883,
    0x0002bc21,
    0x0002bfac,
    0x0002c322,
    0x0002c693,
    0x0002c9f5,
    0x0002cd33,
    0x0002d064,
    0x0002d394,
    0x0002d6c1,
    0x0002d9ae,
    0x0002dc9a,
    0x0002df7d,
    0x0002e24d,
    0x0002e4d9,
    0x0002e73b,
    0x0002e99d,
    0x0002ebe4,
    0x0002ee18,
    0x0002f049,
    0x0002f275,
    0x0002f49a,
    0x0002f6b8,
    0x0002f8cf,
    0x0002fae1, // 64
    0x0002fcf3,
    0x0002fef7,
    0x000300f3,
    0x000302df,
    0x000304bd,
    0x0003068a,
    0x0003084a,
    0x000309ff,
    0x00030bae,
    0x00030d55,
    0x00030ef7,
    0x0003108a,
    0x0003121c,
    0x000313ac,
    0x00031536,
    0x000316b9,
    0x0003183c,
    0x000319ab,
    0x00031b10,
    0x00031c6b,
    0x00031dc6,
    0x00031f1b,
    0x0003206b,
    0x000321b9,
    0x00032302,
    0x00032447,
    0x0003257f,
    0x000326a5,
    0x000327c1,
    0x000328d6,
    0x000329df,
    0x00032ade,
    0x00032bc7,
    0x00032cad,
    0x00032d82,
    0x00032e51,
    0x00032f1e,
    0x00032fe0,
    0x000330a1,
    0x00033159,
    0x0003320e,
    0x000332c3,
    0x0003336a,
    0x0003340e,
    0x000334ac,
    0x00033544,
    0x000335dc,
    0x0003366d,
    0x000336f8,
    0x00033778,
    0x000337df,
    0x00033843,
    0x0003389b,
    0x000338f2,
    0x00033946,
    0x0003399a,
    0x000339e2,
    0x00033a29,
    0x00033a60,
    0x00033a8e,
    0x00033abb,
    0x00033ae6,
    0x00033b0e,
    0x00033b36, // 128
    0x00033b5e,
    0x00033b83,
    0x00033ba6,
    0x00033bc6,
    0x00033be6,
    0x00033c04,
    0x00033c21,
    0x00033c3c, // 136-143
    0x00033c53,
    0x00033c69,
    0x00033c7e,
    0x00033c93,
    0x00033ca8,
    0x00033cbd // 144-149
  };

  /**
   * This region data is stored as integers in a big uniform array. Region objects can be built from
   * this array.
   */
  private static final int[] EAS_REGIONS = {
    // keyGroupAndFlags, rangeLow, rangeHigh, tuning, gain, loopStart, loopEnd, waveIndex, artIndex
    0, 27, 27, -2868, 16422, 0, 0, 81, 0, /* region 0 */
    0, 28, 28, -3568, 32767, 0, 0, 40, 0, /* region 1 */
    0, 29, 29, -4553, 32767, 0, 0, 32, 1, /* region 2 */
    0, 30, 30, -4853, 32767, 0, 0, 32, 2, /* region 3 */
    0, 31, 31, -3868, 23197, 0, 0, 48, 3, /* region 4 */
    1536, 32, 32, -3200, 20675, 0, 0, 137, 4, /* region 5 */
    1537, 33, 33, -3703, 20675, 792, 879, 50, 5, /* region 6 */
    1537, 34, 34, -3803, 16422, 792, 879, 50, 6, /* region 7 */
    0, 35, 35, -4968, 32767, 0, 0, 83, 7, /* region 8 */
    0, 36, 36, -4968, 32767, 0, 0, 83, 7, /* region 9 */
    0, 37, 37, -4051, 18426, 0, 0, 53, 8, /* region 10 */
    0, 38, 38, -4151, 23197, 0, 0, 16, 9, /* region 11 */
    0, 39, 39, -3568, 32767, 0, 0, 40, 10, /* region 12 */
    0, 40, 40, -4151, 23197, 0, 0, 16, 4, /* region 13 */
    1, 41, 41, -5855, 26028, 798, 993, 45, 11, /* region 14 */
    257, 42, 42, -4200, 26028, 4288, 7488, 7, 12, /* region 15 */
    1, 43, 43, -5755, 26028, 798, 993, 45, 13, /* region 16 */
    257, 44, 44, -4400, 26028, 4288, 7488, 7, 14, /* region 17 */
    1, 45, 45, -5755, 26028, 798, 993, 45, 15, /* region 18 */
    257, 46, 46, -4600, 26028, 4288, 7488, 7, 16, /* region 19 */
    1, 47, 47, -5455, 26028, 798, 993, 45, 17, /* region 20 */
    1, 48, 48, -5355, 26028, 798, 993, 45, 18, /* region 21 */
    1, 49, 49, -5200, 16422, 1294, 5778, 8, 19, /* region 22 */
    1, 50, 50, -5255, 26028, 798, 993, 45, 20, /* region 23 */
    1, 51, 51, -5268, 16422, 6592, 9921, 6, 21, /* region 24 */
    1, 52, 52, -5600, 32767, 1294, 5778, 8, 22, /* region 25 */
    1, 53, 53, -5418, 14636, 6592, 9921, 6, 23, /* region 26 */
    0, 54, 54, -5751, 26028, 0, 0, 39, 24, /* region 27 */
    1, 55, 55, -5300, 32767, 1294, 5778, 8, 25, /* region 28 */
    0, 56, 56, -7255, 32767, 0, 0, 90, 26, /* region 29 */
    1, 57, 57, -5700, 32767, 1294, 5778, 8, 27, /* region 30 */
    1, 58, 58, -7053, 23197, 0, 166, 113, 28, /* region 31 */
    1, 59, 59, -5968, 16422, 6592, 9921, 6, 29, /* region 32 */
    1, 60, 60, -6453, 23197, 432, 582, 63, 30, /* region 33 */
    1, 61, 61, -6853, 16422, 432, 582, 63, 30, /* region 34 */
    1, 62, 62, -7253, 20675, 432, 582, 63, 31, /* region 35 */
    1, 63, 63, -7353, 23197, 432, 582, 63, 32, /* region 36 */
    1, 64, 64, -7953, 23197, 432, 582, 63, 33, /* region 37 */
    0, 65, 65, -7555, 32767, 0, 0, 14, 34, /* region 38 */
    0, 66, 66, -7955, 20675, 0, 0, 14, 34, /* region 39 */
    512, 67, 67, -7155, 18426, 0, 0, 90, 35, /* region 40 */
    512, 68, 68, -7755, 18426, 0, 0, 90, 35, /* region 41 */
    0, 69, 69, -7755, 32767, 0, 0, 86, 36, /* region 42 */
    0, 70, 70, -6855, 21900, 0, 0, 86, 37, /* region 43 */
    769, 71, 71, -6355, 23197, 0, 1226, 35, 38, /* region 44 */
    769, 72, 72, -6955, 26028, 0, 1226, 35, 38, /* region 45 */
    1024, 73, 73, -7955, 32767, 0, 0, 22, 39, /* region 46 */
    1024, 74, 74, -8455, 32767, 0, 0, 22, 40, /* region 47 */
    1, 75, 75, -7900, 23197, 0, 31, 139, 41, /* region 48 */
    0, 76, 76, -10455, 23197, 0, 0, 134, 42, /* region 49 */
    0, 77, 77, -10055, 23197, 0, 0, 134, 43, /* region 50 */
    0, 78, 78, -8853, 16422, 0, 0, 89, 44, /* region 51 */
    0, 79, 79, -10253, 16422, 0, 0, 89, 45, /* region 52 */
    1281, 80, 80, -6300, 13045, 209, 230, 103, 46, /* region 53 */
    1281, 81, 81, -6400, 16422, 209, 230, 103, 47, /* region 54 */
    0, 82, 82, -8455, 20675, 0, 0, 87, 48, /* region 55 */
    0, 83, 83, -8900, 32767, 0, 0, 13, 49, /* region 56 */
    1, 84, 84, -8400, 23197, 0, 10294, 5, 50, /* region 57 */
    0, 85, 85, -9655, 32767, 0, 0, 135, 4, /* region 58 */
    0, 86, 86, -9068, 16422, 0, 0, 24, 51, /* region 59 */
    32769, 87, 87, -9168, 32767, 1335, 1603, 24, 52, /* region 60 */
    1, 12, 67, -6605, 23197, 437, 16584, 2, 48, /* region 61 */
    1, 68, 73, -7196, 23197, 452, 16803, 0, 48, /* region 62 */
    32769, 74, 108, -8467, 23197, 404, 16698, 1, 48, /* region 63 */
    1, 12, 78, -6605, 16422, 437, 16584, 2, 48, /* region 64 */
    1, 79, 91, -7196, 16422, 452, 16803, 0, 48, /* region 65 */
    32769, 92, 108, -8467, 16422, 404, 16698, 1, 48, /* region 66 */
    1, 12, 78, -6605, 16422, 437, 16584, 2, 48, /* region 67 */
    1, 79, 91, -7196, 16422, 452, 16803, 0, 48, /* region 68 */
    32769, 92, 108, -8467, 16422, 404, 16698, 1, 48, /* region 69 */
    1, 12, 70, -6600, 23197, 437, 16584, 2, 48, /* region 70 */
    1, 71, 88, -7191, 23197, 452, 16803, 0, 48, /* region 71 */
    32769, 89, 108, -8462, 23197, 404, 16698, 1, 48, /* region 72 */
    1, 12, 54, -5956, 13045, 639, 4368, 10, 48, /* region 73 */
    32769, 55, 108, -6351, 18426, 702, 3112, 12, 48, /* region 74 */
    1, 12, 66, -6611, 23197, 437, 16584, 2, 48, /* region 75 */
    1, 67, 87, -7202, 23197, 452, 16803, 0, 48, /* region 76 */
    32769, 88, 108, -8473, 16422, 404, 16698, 1, 48, /* region 77 */
    1, 12, 43, -3055, 23197, 920, 1383, 30, 59, /* region 78 */
    32769, 44, 96, -5060, 18426, 885, 1176, 37, 59, /* region 79 */
    1, 12, 48, -3461, 18426, 1148, 1514, 26, 60, /* region 80 */
    32769, 49, 96, -6253, 16422, 1347, 1420, 29, 60, /* region 81 */
    1, 33, 56, -5600, 26028, 1064, 1170, 38, 61, /* region 82 */
    1, 57, 72, -6000, 26028, 930, 1014, 44, 61, /* region 83 */
    32769, 73, 108, -7600, 26028, 726, 826, 52, 61, /* region 84 */
    1, 36, 96, -7600, 20675, 635, 735, 58, 62, /* region 85 */
    32769, 97, 108, -10108, 13045, 0, 31, 139, 62, /* region 86 */
    1, 36, 96, -7600, 14636, 635, 735, 58, 0, /* region 87 */
    32769, 97, 108, -10108, 13045, 0, 31, 139, 0, /* region 88 */
    1, 36, 83, -6006, 13045, 838, 922, 47, 63, /* region 89 */
    1, 84, 93, -8406, 14636, 209, 230, 103, 63, /* region 90 */
    32769, 94, 108, -10108, 13045, 0, 31, 139, 63, /* region 91 */
    1, 36, 83, -6006, 13045, 838, 922, 47, 64, /* region 92 */
    1, 84, 93, -8406, 13045, 209, 230, 103, 64, /* region 93 */
    32769, 94, 108, -10108, 13045, 0, 31, 139, 64, /* region 94 */
    1, 21, 56, -5595, 23197, 1064, 1170, 38, 65, /* region 95 */
    1, 57, 72, -5995, 23197, 930, 1014, 44, 65, /* region 96 */
    32769, 73, 108, -7598, 23197, 726, 826, 52, 65, /* region 97 */
    1, 12, 83, -6006, 16422, 838, 922, 47, 66, /* region 98 */
    1, 84, 93, -8406, 16422, 209, 230, 103, 66, /* region 99 */
    32769, 94, 108, -10108, 16422, 0, 31, 139, 66, /* region 100 */
    1, 24, 83, -6006, 16422, 838, 922, 47, 67, /* region 101 */
    1, 84, 93, -8406, 16422, 209, 230, 103, 67, /* region 102 */
    32769, 94, 108, -10108, 16422, 0, 31, 139, 67, /* region 103 */
    1, 12, 83, -6020, 16422, 0, 83, 126, 68, /* region 104 */
    1, 84, 90, -8482, 16422, 0, 20, 145, 68, /* region 105 */
    32769, 91, 108, -9101, 16422, 6, 20, 147, 68, /* region 106 */
    1, 21, 75, -7241, 16422, 419, 460, 76, 69, /* region 107 */
    32769, 76, 108, -9690, 14636, 254, 264, 101, 69, /* region 108 */
    32769, 36, 84, -7755, 16422, 0, 2775, 17, 70, /* region 109 */
    32769, 12, 108, -6655, 23197, 30, 276, 100, 71, /* region 110 */
    0, 12, 60, -7914, 26028, 0, 0, 15, 72, /* region 111 */
    32768, 61, 96, -7914, 26028, 0, 0, 15, 73, /* region 112 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 74, /* region 113 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 74, /* region 114 */
    1, 12, 35, -5355, 16422, 2869, 3778, 11, 75, /* region 115 */
    1, 36, 48, -6555, 20675, 2869, 3778, 11, 75, /* region 116 */
    32769, 49, 72, -6555, 20675, 2869, 3778, 11, 76, /* region 117 */
    1, 16, 55, -6224, 20675, 1045, 1119, 41, 77, /* region 118 */
    32769, 56, 96, -6718, 20675, 907, 963, 46, 77, /* region 119 */
    1, 16, 53, -5994, 29204, 1140, 1479, 27, 78, /* region 120 */
    1, 54, 70, -7171, 29204, 726, 812, 55, 78, /* region 121 */
    32769, 71, 108, -7788, 29204, 718, 748, 56, 78, /* region 122 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 79, /* region 123 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 79, /* region 124 */
    1, 16, 54, -5727, 20675, 5362, 5461, 9, 80, /* region 125 */
    1, 55, 63, -5851, 26028, 1362, 1454, 28, 80, /* region 126 */
    32769, 64, 108, -6744, 16422, 311, 366, 88, 80, /* region 127 */
    1, 16, 48, -4798, 20675, 1132, 1301, 31, 81, /* region 128 */
    32769, 49, 108, -5988, 20675, 1099, 1184, 36, 81, /* region 129 */
    1, 21, 68, -8458, 20675, 87, 2170, 18, 82, /* region 130 */
    1, 69, 82, -8960, 20675, 120, 2167, 19, 82, /* region 131 */
    32769, 83, 108, -10160, 20675, 376, 2041, 20, 82, /* region 132 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 83, /* region 133 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 83, /* region 134 */
    32769, 55, 108, -7368, 20675, 0, 477, 75, 84, /* region 135 */
    32769, 36, 96, -6900, 14636, 101, 151, 116, 85, /* region 136 */
    1, 24, 83, -6020, 13045, 0, 83, 126, 86, /* region 137 */
    1, 84, 90, -8482, 13045, 0, 20, 145, 86, /* region 138 */
    32769, 91, 108, -9101, 13045, 6, 20, 147, 86, /* region 139 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 87, /* region 140 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 87, /* region 141 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 88, /* region 142 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 88, /* region 143 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 89, /* region 144 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 89, /* region 145 */
    1, 24, 83, -6020, 13045, 0, 83, 126, 90, /* region 146 */
    1, 84, 90, -8482, 13045, 0, 20, 145, 90, /* region 147 */
    32769, 91, 108, -9101, 13045, 6, 20, 147, 90, /* region 148 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 91, /* region 149 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 91, /* region 150 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 92, /* region 151 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 92, /* region 152 */
    1, 12, 62, -7053, 16422, 23, 10953, 4, 93, /* region 153 */
    32769, 63, 108, -7755, 20675, 11, 11753, 3, 93, /* region 154 */
    1, 12, 62, -7053, 16422, 23, 10953, 4, 94, /* region 155 */
    32769, 63, 108, -7755, 16422, 11, 11753, 3, 94, /* region 156 */
    1, 24, 79, -6020, 13045, 0, 83, 126, 95, /* region 157 */
    1, 80, 90, -8482, 13045, 0, 20, 145, 95, /* region 158 */
    32769, 91, 108, -9101, 13045, 6, 20, 147, 95, /* region 159 */
    1, 12, 65, -7053, 13045, 23, 10953, 4, 96, /* region 160 */
    32769, 66, 108, -7755, 16422, 11, 11753, 3, 96, /* region 161 */
    32768, 36, 84, -7500, 20675, 0, 0, 25, 97, /* region 162 */
    32769, 36, 96, -8855, 20675, 1482, 1613, 23, 98, /* region 163 */
    32769, 12, 96, -4366, 32767, 818, 1033, 42, 99, /* region 164 */
    32769, 36, 84, -8568, 18426, 0, 293, 98, 100, /* region 165 */
    32769, 12, 96, -6020, 26028, 0, 83, 125, 101, /* region 166 */
    32769, 12, 96, -6020, 20675, 0, 83, 125, 102, /* region 167 */
    1, 12, 83, -6020, 13045, 0, 83, 125, 104, /* region 168 */
    1, 84, 90, -8482, 13045, 0, 20, 146, 104, /* region 169 */
    32769, 91, 108, -9101, 13045, 6, 20, 148, 104, /* region 170 */
    32769, 36, 108, -8570, 32767, 472, 491, 74, 105, /* region 171 */
    32769, 36, 108, -8570, 20675, 472, 491, 74, 106, /* region 172 */
    1, 12, 72, -6012, 7336, 2, 86, 124, 107, /* region 173 */
    1, 73, 101, -8500, 8231, 2, 22, 143, 107, /* region 174 */
    32769, 102, 108, -9683, 20675, 173, 183, 110, 107, /* region 175 */
    1, 21, 96, -7768, 13045, 477, 507, 73, 108, /* region 176 */
    32769, 97, 108, -9683, 13045, 173, 183, 110, 109, /* region 177 */
    32769, 12, 108, -7771, 16422, 477, 507, 73, 110, /* region 178 */
    1, 12, 53, -4971, 16422, 388, 541, 68, 111, /* region 179 */
    32769, 54, 60, -5949, 11626, 473, 560, 65, 111, /* region 180 */
    32769, 36, 72, -5949, 16422, 473, 560, 65, 112, /* region 181 */
    1, 48, 58, -7053, 16422, 356, 402, 82, 113, /* region 182 */
    1, 59, 65, -7574, 16422, 514, 548, 67, 113, /* region 183 */
    1, 66, 78, -8174, 16422, 505, 529, 71, 113, /* region 184 */
    32769, 79, 96, -9233, 16422, 178, 191, 109, 113, /* region 185 */
    1, 55, 60, -7053, 16422, 356, 402, 82, 114, /* region 186 */
    1, 61, 69, -7574, 16422, 514, 548, 67, 114, /* region 187 */
    1, 70, 79, -8174, 16422, 505, 529, 71, 114, /* region 188 */
    32769, 80, 108, -9233, 16422, 178, 191, 109, 114, /* region 189 */
    1, 16, 82, -8029, 23197, 180, 206, 106, 115, /* region 190 */
    32769, 83, 108, -7240, 18426, 3, 44, 131, 115, /* region 191 */
    32769, 21, 108, -8869, 20675, 483, 515, 72, 116, /* region 192 */
    1, 21, 89, -7205, 18426, 3, 45, 130, 117, /* region 193 */
    32769, 90, 108, -9101, 10362, 6, 20, 148, 117, /* region 194 */
    1, 21, 42, -4686, 20675, 0, 180, 111, 118, /* region 195 */
    1, 43, 51, -5286, 23197, 0, 127, 120, 118, /* region 196 */
    1, 52, 58, -6292, 26028, 0, 71, 127, 118, /* region 197 */
    1, 59, 68, -7468, 23197, 0, 36, 136, 118, /* region 198 */
    32769, 69, 108, -8574, 20675, 0, 19, 149, 118, /* region 199 */
    1, 21, 89, -7199, 20675, 3, 45, 130, 119, /* region 200 */
    32769, 90, 108, -9101, 14636, 6, 20, 148, 119, /* region 201 */
    1, 21, 46, -5651, 26028, 236, 340, 92, 120, /* region 202 */
    1, 47, 71, -6563, 20675, 824, 885, 49, 120, /* region 203 */
    1, 72, 88, -7907, 18426, 719, 747, 57, 120, /* region 204 */
    1, 89, 93, -8876, 16422, 83, 99, 122, 120, /* region 205 */
    32769, 94, 108, -9689, 16422, 173, 183, 110, 120, /* region 206 */
    1, 60, 71, -7205, 16422, 0, 42, 132, 121, /* region 207 */
    1, 72, 78, -7903, 16422, 0, 28, 141, 121, /* region 208 */
    32769, 79, 96, -8405, 16422, 0, 21, 144, 121, /* region 209 */
    1, 48, 65, -6316, 11626, 0, 70, 128, 122, /* region 210 */
    1, 66, 79, -7724, 14636, 0, 31, 138, 122, /* region 211 */
    32769, 80, 96, -8030, 11626, 0, 26, 142, 122, /* region 212 */
    1, 16, 44, -5868, 14636, 163, 254, 102, 123, /* region 213 */
    1, 45, 51, -6418, 16422, 261, 393, 85, 123, /* region 214 */
    1, 52, 58, -7333, 18426, 190, 229, 104, 123, /* region 215 */
    1, 59, 66, -8100, 18426, 168, 193, 108, 123, /* region 216 */
    1, 67, 70, -8576, 18426, 138, 157, 115, 123, /* region 217 */
    1, 71, 80, -9103, 18426, 166, 180, 112, 123, /* region 218 */
    32769, 81, 108, -10074, 18426, 135, 151, 117, 123, /* region 219 */
    32769, 12, 96, -5004, 23197, 570, 719, 59, 124, /* region 220 */
    1, 12, 48, -5868, 14636, 163, 254, 102, 125, /* region 221 */
    1, 49, 54, -6418, 16422, 261, 393, 85, 125, /* region 222 */
    1, 55, 63, -7333, 18426, 190, 229, 104, 125, /* region 223 */
    1, 64, 70, -8100, 18426, 168, 193, 108, 125, /* region 224 */
    1, 71, 75, -8576, 18426, 138, 157, 115, 125, /* region 225 */
    1, 76, 82, -9103, 18426, 166, 180, 112, 125, /* region 226 */
    32769, 83, 108, -10074, 18426, 135, 151, 117, 125, /* region 227 */
    32770, 36, 84, -7200, 29204, 0, 0, 0, 126, /* region 228 */
    32770, 36, 84, -7600, 8231, 0, 0, 0, 127, /* region 229 */
    32770, 36, 84, -7200, 20675, 0, 0, 0, 128, /* region 230 */
    32769, 36, 84, -6000, -24285, 1294, 5778, 8, 129, /* region 231 */
    32769, 36, 84, -6555, 29204, 798, 993, 45, 130, /* region 232 */
    32769, 36, 84, -6855, 20675, 798, 993, 45, 131, /* region 233 */
    32769, 36, 84, -7755, 29204, 798, 993, 45, 132, /* region 234 */
    32768, 36, 84, -8155, 32767, 0, 0, 133, 133, /* region 235 */
    32768, 36, 84, -6555, 20675, 0, 0, 91, 134, /* region 236 */
    1, 24, 62, -7000, 23197, 286, 333, 94, 135, /* region 237 */
    1, 63, 66, -7364, 26028, 297, 335, 93, 135, /* region 238 */
    1, 67, 72, -7722, 23197, 368, 399, 84, 135, /* region 239 */
    32769, 73, 96, -8310, 23197, 116, 138, 119, 135, /* region 240 */
    1, 24, 48, -5141, 23197, 309, 447, 77, 136, /* region 241 */
    1, 49, 56, -6266, 26028, 211, 283, 99, 136, /* region 242 */
    1, 57, 63, -7000, 26028, 286, 333, 94, 136, /* region 243 */
    32769, 64, 84, -7722, 23197, 368, 399, 84, 136, /* region 244 */
    1, 24, 56, -6266, 29204, 211, 283, 99, 137, /* region 245 */
    1, 57, 63, -7000, 29204, 286, 333, 94, 137, /* region 246 */
    1, 64, 69, -7722, 29204, 368, 399, 84, 137, /* region 247 */
    32769, 70, 96, -8310, 29204, 116, 138, 119, 137, /* region 248 */
    1, 24, 68, -7722, 18426, 368, 399, 84, 138, /* region 249 */
    1, 69, 76, -8310, 26028, 116, 138, 119, 138, /* region 250 */
    32769, 77, 108, -8758, 23197, 127, 144, 118, 138, /* region 251 */
    1, 24, 82, -7613, 23197, 389, 422, 80, 139, /* region 252 */
    32769, 83, 108, -8764, 26028, 146, 163, 114, 139, /* region 253 */
    1, 12, 58, -6898, 29204, 386, 436, 78, 140, /* region 254 */
    32769, 59, 96, -7371, 26028, 290, 328, 95, 140, /* region 255 */
    1, 12, 58, -6898, 16422, 386, 436, 78, 141, /* region 256 */
    32769, 59, 96, -7371, 18426, 290, 328, 95, 141, /* region 257 */
    1, 12, 48, -6898, -28771, 386, 436, 78, 142, /* region 258 */
    32769, 49, 84, -7371, 29204, 290, 328, 95, 142, /* region 259 */
    1, 12, 60, -5453, 20675, 314, 430, 79, 143, /* region 260 */
    32769, 61, 84, -6553, 18426, 263, 324, 96, 143, /* region 261 */
    1, 24, 60, -6553, 16422, 263, 324, 96, 144, /* region 262 */
    1, 61, 70, -7669, 20675, 279, 311, 97, 144, /* region 263 */
    32769, 71, 96, -8098, 23197, 179, 204, 107, 144, /* region 264 */
    1, 24, 84, -8483, 20675, 191, 211, 105, 145, /* region 265 */
    32769, 85, 108, -9683, 20675, 92, 102, 121, 145, /* region 266 */
    1, 21, 69, -6553, 13045, 263, 324, 96, 146, /* region 267 */
    1, 70, 94, -7669, 20675, 279, 311, 97, 146, /* region 268 */
    1, 95, 96, -8098, -24285, 179, 204, 107, 146, /* region 269 */
    32769, 97, 108, -9683, -24285, 173, 183, 110, 146, /* region 270 */
    1, 16, 55, -8100, 20675, 168, 193, 108, 147, /* region 271 */
    1, 56, 74, -8576, 26028, 138, 157, 115, 147, /* region 272 */
    32769, 75, 96, -10074, 26028, 135, 151, 117, 147, /* region 273 */
    1, 24, 72, -8098, 26028, 179, 204, 107, 148, /* region 274 */
    1, 73, 85, -8483, 20675, 191, 211, 105, 148, /* region 275 */
    32769, 86, 108, -9683, 18426, 92, 102, 121, 148, /* region 276 */
    32769, 36, 108, -7730, 18426, 1839, 1901, 21, 149, /* region 277 */
    32769, 24, 108, -7273, 20675, 494, 534, 69, 150, /* region 278 */
    32769, 12, 108, -7273, 20675, 494, 534, 69, 151, /* region 279 */
    32769, 24, 108, -7273, 20675, 494, 534, 69, 152, /* region 280 */
    1, 36, 60, -4900, 5193, 2, 22, 143, 153, /* region 281 */
    32769, 61, 84, -6083, 20675, 173, 183, 110, 153, /* region 282 */
    32769, 24, 96, -6553, 14636, 263, 324, 96, 154, /* region 283 */
    32769, 36, 96, -7730, 26028, 1839, 1901, 21, 155, /* region 284 */
    32769, 24, 108, -7273, 20675, 494, 534, 69, 156, /* region 285 */
    1, 24, 58, -7851, 14636, 0, 29, 140, 157, /* region 286 */
    32769, 59, 96, -7851, 14636, 0, 29, 140, 157, /* region 287 */
    1, 12, 83, -6020, 13045, 0, 83, 125, 158, /* region 288 */
    1, 84, 90, -8482, 13045, 0, 20, 146, 158, /* region 289 */
    32769, 91, 108, -9101, 13045, 6, 20, 148, 158, /* region 290 */
    1, 21, 42, -4663, 26028, 1047, 1229, 34, 159, /* region 291 */
    1, 43, 48, -5456, 29204, 1138, 1253, 33, 159, /* region 292 */
    1, 49, 53, -5845, 26028, 559, 651, 60, 159, /* region 293 */
    1, 54, 60, -6732, 26028, 508, 563, 64, 159, /* region 294 */
    1, 61, 65, -7080, 32767, 819, 864, 51, 159, /* region 295 */
    1, 66, 70, -6866, 26942, 981, 1032, 43, 159, /* region 296 */
    1, 71, 76, -8166, 26028, 790, 814, 54, 159, /* region 297 */
    1, 77, 82, -8766, 26028, 592, 609, 61, 159, /* region 298 */
    1, 83, 87, -9517, 23197, 543, 554, 66, 159, /* region 299 */
    1, 88, 96, -10071, 18426, 601, 609, 62, 159, /* region 300 */
    32769, 97, 108, -10566, 18426, 523, 529, 70, 159, /* region 301 */
    1, 48, 69, -6313, 14636, 0, 70, 128, 160, /* region 302 */
    1, 70, 79, -7724, 18426, 0, 31, 138, 160, /* region 303 */
    32769, 80, 96, -8030, 14636, 0, 26, 142, 160, /* region 304 */
    1, 36, 72, -7134, 29204, 0, 87, 123, 161, /* region 305 */
    32769, 73, 96, -7960, 29204, 0, 54, 129, 161, /* region 306 */
    32769, 36, 96, -7730, 26028, 1839, 1901, 21, 162, /* region 307 */
    32769, 12, 96, -4372, 32767, 818, 1033, 42, 163, /* region 308 */
    32769, 36, 108, -8570, 26028, 472, 491, 74, 164, /* region 309 */
    32769, 12, 96, -5004, 29204, 570, 719, 59, 165, /* region 310 */
    1, 12, 83, -6020, 13045, 0, 83, 125, 166, /* region 311 */
    1, 84, 90, -8482, 13045, 0, 20, 146, 166, /* region 312 */
    32769, 91, 108, -9101, 13045, 6, 20, 148, 166, /* region 313 */
    1, 21, 46, -5651, 32767, 236, 340, 92, 167, /* region 314 */
    1, 47, 75, -6563, 26028, 824, 885, 49, 167, /* region 315 */
    1, 76, 84, -7907, 23197, 719, 747, 57, 167, /* region 316 */
    1, 85, 93, -8876, 20675, 83, 99, 122, 167, /* region 317 */
    32769, 94, 108, -9689, 20675, 173, 183, 110, 167, /* region 318 */
    1, 21, 46, -5651, 26028, 236, 340, 92, 168, /* region 319 */
    1, 47, 71, -6563, 20675, 824, 885, 49, 168, /* region 320 */
    1, 72, 88, -7907, 18426, 719, 747, 57, 168, /* region 321 */
    1, 89, 93, -8876, 16422, 83, 99, 122, 168, /* region 322 */
    32769, 94, 108, -9689, 16422, 173, 183, 110, 168, /* region 323 */
    1, 21, 45, -4663, 26028, 1047, 1229, 34, 169, /* region 324 */
    1, 46, 51, -5456, 29204, 1138, 1253, 33, 169, /* region 325 */
    1, 52, 54, -5845, 26028, 559, 651, 60, 169, /* region 326 */
    1, 55, 63, -6732, 26028, 508, 563, 64, 169, /* region 327 */
    1, 64, 68, -7080, 32767, 819, 864, 51, 169, /* region 328 */
    1, 69, 73, -6866, 26942, 981, 1032, 43, 169, /* region 329 */
    1, 74, 79, -8166, 26028, 790, 814, 54, 169, /* region 330 */
    1, 80, 88, -8766, 23197, 592, 609, 61, 169, /* region 331 */
    1, 89, 99, -10071, 18426, 601, 609, 62, 169, /* region 332 */
    32769, 100, 108, -10566, 18426, 523, 529, 70, 169, /* region 333 */
    1, 21, 45, -4663, 26028, 1047, 1229, 34, 170, /* region 334 */
    1, 46, 51, -5456, 29204, 1138, 1253, 33, 170, /* region 335 */
    1, 52, 54, -5845, 26028, 559, 651, 60, 170, /* region 336 */
    1, 55, 63, -6732, 26028, 508, 563, 64, 170, /* region 337 */
    1, 64, 68, -7080, 32767, 819, 864, 51, 170, /* region 338 */
    1, 69, 73, -6866, 26942, 981, 1032, 43, 170, /* region 339 */
    1, 74, 79, -8166, 26028, 790, 814, 54, 170, /* region 340 */
    1, 80, 88, -8766, 23197, 592, 609, 61, 170, /* region 341 */
    1, 89, 99, -10071, 18426, 601, 609, 62, 171, /* region 342 */
    32769, 100, 108, -10566, 18426, 523, 529, 70, 172, /* region 343 */
    32769, 36, 108, -8570, 20675, 472, 491, 74, 173, /* region 344 */
    32769, 12, 108, -7730, 20675, 1839, 1901, 21, 174, /* region 345 */
    1, 12, 44, -5868, 18426, 163, 254, 102, 175, /* region 346 */
    1, 45, 51, -6418, 20675, 261, 393, 85, 175, /* region 347 */
    1, 52, 58, -7333, 23197, 190, 229, 104, 175, /* region 348 */
    1, 59, 66, -8100, 23197, 168, 193, 108, 175, /* region 349 */
    1, 67, 70, -8576, 23197, 138, 157, 115, 175, /* region 350 */
    1, 71, 80, -9103, 23197, 166, 180, 112, 175, /* region 351 */
    32769, 81, 108, -10074, 23197, 135, 151, 117, 175, /* region 352 */
    1, 12, 65, -7053, 16422, 23, 10953, 4, 176, /* region 353 */
    32769, 66, 108, -7755, 20675, 11, 11753, 3, 176, /* region 354 */
    1, 12, 48, -4798, 29204, 1132, 1301, 31, 177, /* region 355 */
    32769, 49, 108, -5988, 29204, 1099, 1184, 36, 177, /* region 356 */
    1, 12, 83, -7241, 20675, 419, 460, 76, 178, /* region 357 */
    32769, 84, 108, -10123, 20675, 0, 31, 139, 178, /* region 358 */
    1, 55, 60, -7053, 18426, 356, 402, 82, 179, /* region 359 */
    1, 61, 69, -7574, 18426, 514, 548, 67, 179, /* region 360 */
    1, 70, 79, -8174, 18426, 505, 529, 71, 179, /* region 361 */
    32769, 80, 108, -9233, 23197, 178, 191, 109, 179, /* region 362 */
    32769, 36, 96, -7730, -24285, 1839, 1901, 21, 180, /* region 363 */
    1, 12, 83, -6006, 16422, 838, 922, 47, 181, /* region 364 */
    1, 84, 93, -8406, 18426, 209, 230, 103, 181, /* region 365 */
    32769, 94, 108, -10108, 16422, 0, 31, 139, 181, /* region 366 */
    1, 12, 56, -5595, 23197, 1064, 1170, 38, 182, /* region 367 */
    1, 57, 72, -5995, 23197, 930, 1014, 44, 182, /* region 368 */
    32769, 73, 108, -7598, 23197, 726, 826, 52, 182, /* region 369 */
    32769, 24, 108, -7600, 23197, 635, 735, 58, 62, /* region 370 */
    1, 36, 83, -6006, 13045, 838, 922, 47, 183, /* region 371 */
    1, 84, 93, -8406, 13045, 209, 230, 103, 183, /* region 372 */
    32769, 94, 108, -10108, 13045, 0, 31, 139, 183, /* region 373 */
    1, 12, 66, -6611, 23197, 437, 16584, 2, 184, /* region 374 */
    1, 67, 87, -7202, 23197, 452, 16803, 0, 184, /* region 375 */
    32769, 88, 108, -8473, 16422, 404, 16698, 1, 184 /* region 376 */
  }; /* end Regions */

  private static final short[] EAS_ARTICULATIONS = {
    /* articulation 0 */
    32767,
    30725,
    0,
    30725,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 1 */
    32767,
    26863,
    0,
    26863,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 2 */
    32767,
    30484,
    0,
    30668,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 3 */
    32767,
    26439,
    0,
    26439,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 4 */
    32767,
    0,
    32767,
    32715,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 5 */
    32767,
    21333,
    0,
    21333,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 6 */
    32767,
    31882,
    0,
    31938,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 7 */
    32767,
    32663,
    32767,
    32663,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 8 */
    32767,
    0,
    32767,
    0,
    32767,
    1902,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 9 */
    32767,
    32349,
    0,
    32349,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 10 */
    32767,
    0,
    32767,
    17213,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -1,
    /* articulation 11 */
    32767,
    31730,
    0,
    31730,
    32767,
    761,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -44,
    /* articulation 12 */
    32767,
    23749,
    0,
    23749,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    56,
    /* articulation 13 */
    32767,
    31730,
    0,
    31730,
    32767,
    761,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -31,
    /* articulation 14 */
    9511,
    21333,
    0,
    21333,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    56,
    /* articulation 15 */
    32767,
    31617,
    0,
    31617,
    32767,
    761,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -6,
    /* articulation 16 */
    32767,
    32123,
    0,
    32194,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    56,
    /* articulation 17 */
    32767,
    31550,
    0,
    31550,
    32767,
    761,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    6,
    /* articulation 18 */
    32767,
    31391,
    0,
    31391,
    32767,
    951,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    31,
    /* articulation 19 */
    32767,
    31964,
    0,
    31964,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    25,
    /* articulation 20 */
    32767,
    31056,
    0,
    31056,
    32767,
    951,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    63,
    /* articulation 21 */
    32767,
    32289,
    0,
    32271,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -37,
    /* articulation 22 */
    19021,
    31882,
    0,
    31911,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -37,
    /* articulation 23 */
    32767,
    31988,
    0,
    32032,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -37,
    /* articulation 24 */
    32767,
    0,
    32767,
    32663,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    12,
    /* articulation 25 */
    32767,
    31352,
    0,
    31352,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -25,
    /* articulation 26 */
    32767,
    0,
    32767,
    32663,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    25,
    /* articulation 27 */
    32767,
    31817,
    0,
    31781,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -25,
    /* articulation 28 */
    32767,
    30725,
    0,
    30725,
    32767,
    95,
    0,
    0,
    0,
    0,
    951,
    240,
    0,
    0,
    0,
    0,
    -56,
    /* articulation 29 */
    32767,
    32230,
    0,
    32218,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -37,
    /* articulation 30 */
    32767,
    26439,
    0,
    26439,
    32767,
    3804,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    50,
    /* articulation 31 */
    32767,
    23749,
    0,
    23749,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -50,
    /* articulation 32 */
    32767,
    29434,
    0,
    29434,
    32767,
    3804,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -50,
    /* articulation 33 */
    32767,
    30240,
    0,
    30234,
    32767,
    3804,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -44,
    /* articulation 34 */
    32767,
    32558,
    0,
    32558,
    32767,
    254,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    25,
    /* articulation 35 */
    32767,
    0,
    32767,
    32663,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -63,
    /* articulation 36 */
    3804,
    23749,
    0,
    23749,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -63,
    /* articulation 37 */
    32767,
    23749,
    0,
    23749,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -59,
    /* articulation 38 */
    32767,
    30725,
    0,
    30725,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    50,
    /* articulation 39 */
    32767,
    28809,
    0,
    28809,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    44,
    /* articulation 40 */
    1902,
    30725,
    0,
    30725,
    32767,
    380,
    0,
    0,
    0,
    0,
    951,
    -100,
    0,
    0,
    0,
    0,
    44,
    /* articulation 41 */
    32767,
    9042,
    0,
    9042,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    25,
    /* articulation 42 */
    32767,
    29889,
    0,
    29889,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    63,
    /* articulation 43 */
    32767,
    30240,
    0,
    30234,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    63,
    /* articulation 44 */
    19021,
    19970,
    0,
    19970,
    951,
    32767,
    32767,
    0,
    0,
    0,
    951,
    100,
    0,
    0,
    0,
    0,
    -25,
    /* articulation 45 */
    3804,
    17213,
    0,
    17213,
    951,
    32767,
    32767,
    0,
    0,
    0,
    951,
    500,
    0,
    0,
    0,
    0,
    -25,
    /* articulation 46 */
    32767,
    17213,
    0,
    17213,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -56,
    /* articulation 47 */
    32767,
    30725,
    0,
    30725,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    -56,
    /* articulation 48 */
    32767,
    0,
    32767,
    0,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 49 */
    32767,
    31180,
    0,
    31180,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 50 */
    19021,
    31964,
    0,
    32071,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 51 */
    32767,
    29669,
    0,
    29669,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 52 */
    32767,
    31742,
    0,
    31352,
    32767,
    294,
    0,
    0,
    0,
    0,
    951,
    0,
    10000,
    7121,
    0,
    0,
    0,
    /* articulation 53 */
    32767,
    0,
    32767,
    31391,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    1555,
    0,
    -2300,
    11920,
    0,
    0,
    0,
    /* articulation 54 */
    1174,
    0,
    32767,
    31988,
    32767,
    127,
    0,
    0,
    0,
    0,
    1555,
    0,
    2000,
    10721,
    0,
    8,
    15,
    /* articulation 55 */
    1174,
    0,
    32767,
    31988,
    951,
    127,
    0,
    0,
    0,
    0,
    1555,
    0,
    2000,
    9023,
    0,
    5,
    15,
    /* articulation 56 */
    7608,
    0,
    32767,
    30237,
    32767,
    69,
    5898,
    0,
    0,
    0,
    1555,
    0,
    6000,
    9080,
    0,
    0,
    -2,
    /* articulation 57 */
    32767,
    0,
    32767,
    29337,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    1555,
    0,
    0,
    0,
    0,
    0,
    1,
    /* articulation 58 */
    5141,
    0,
    32767,
    30194,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    1555,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 59 */
    32767,
    32558,
    0,
    26439,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 60 */
    32767,
    32349,
    0,
    26439,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 61 */
    32767,
    32072,
    0,
    32072,
    32767,
    95,
    0,
    0,
    0,
    34,
    989,
    0,
    2400,
    9521,
    0,
    0,
    0,
    /* articulation 62 */
    32767,
    30234,
    0,
    30234,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    11738,
    0,
    16,
    0,
    /* articulation 63 */
    32767,
    32349,
    0,
    30073,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 64 */
    32767,
    31730,
    0,
    31476,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 65 */
    32767,
    32418,
    0,
    25329,
    32767,
    95,
    0,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 66 */
    32767,
    32052,
    0,
    31964,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 67 */
    32767,
    31938,
    0,
    31938,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 68 */
    9511,
    32663,
    18820,
    23749,
    1902,
    57,
    13107,
    0,
    0,
    0,
    989,
    0,
    6000,
    5535,
    0,
    4,
    0,
    /* articulation 69 */
    32767,
    31754,
    0,
    31730,
    32767,
    1902,
    0,
    0,
    0,
    52,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 70 */
    127,
    32686,
    3811,
    32349,
    95,
    38,
    32767,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 71 */
    4755,
    32663,
    3566,
    28809,
    3804,
    32767,
    32767,
    0,
    0,
    0,
    989,
    100,
    0,
    11919,
    0,
    0,
    0,
    /* articulation 72 */
    32767,
    31935,
    0,
    31935,
    32767,
    335,
    0,
    0,
    0,
    17,
    989,
    0,
    7000,
    9023,
    0,
    0,
    0,
    /* articulation 73 */
    32767,
    31391,
    0,
    31391,
    32767,
    335,
    0,
    0,
    0,
    2,
    951,
    0,
    7000,
    9023,
    0,
    0,
    0,
    /* articulation 74 */
    32767,
    32628,
    6208,
    31935,
    380,
    95,
    0,
    0,
    0,
    0,
    989,
    0,
    3840,
    8302,
    0,
    8,
    0,
    /* articulation 75 */
    32767,
    32072,
    0,
    32171,
    32767,
    380,
    0,
    0,
    0,
    0,
    989,
    0,
    5000,
    8321,
    0,
    0,
    0,
    /* articulation 76 */
    32767,
    31935,
    0,
    31935,
    32767,
    380,
    0,
    0,
    0,
    0,
    951,
    0,
    5000,
    7934,
    0,
    0,
    0,
    /* articulation 77 */
    32767,
    32117,
    0,
    30685,
    32767,
    63,
    0,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 78 */
    32767,
    32245,
    0,
    23749,
    32767,
    1902,
    0,
    0,
    0,
    172,
    989,
    0,
    1000,
    11107,
    0,
    0,
    0,
    /* articulation 79 */
    32767,
    32663,
    6208,
    31935,
    95,
    95,
    0,
    0,
    0,
    34,
    1622,
    0,
    3560,
    8834,
    1,
    8,
    0,
    /* articulation 80 */
    32767,
    32362,
    0,
    26439,
    32767,
    190,
    0,
    0,
    0,
    17,
    989,
    0,
    6000,
    9907,
    0,
    0,
    0,
    /* articulation 81 */
    32767,
    32245,
    0,
    23749,
    32767,
    63,
    0,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 82 */
    32767,
    31730,
    18820,
    9042,
    32767,
    32767,
    32767,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 83 */
    32767,
    32715,
    128,
    32168,
    32767,
    127,
    0,
    0,
    0,
    0,
    989,
    0,
    0,
    11920,
    0,
    8,
    0,
    /* articulation 84 */
    32767,
    32072,
    0,
    32072,
    32767,
    67,
    0,
    0,
    3,
    0,
    572,
    0,
    5000,
    5535,
    0,
    0,
    0,
    /* articulation 85 */
    3804,
    32663,
    18820,
    23749,
    32767,
    2024,
    0,
    0,
    10,
    34,
    1008,
    -30,
    0,
    0,
    0,
    0,
    0,
    /* articulation 86 */
    19021,
    32663,
    18820,
    23749,
    761,
    95,
    0,
    0,
    0,
    34,
    989,
    0,
    4473,
    7131,
    0,
    8,
    0,
    /* articulation 87 */
    1902,
    32628,
    6208,
    32171,
    634,
    38,
    16384,
    0,
    0,
    0,
    989,
    0,
    2987,
    7877,
    0,
    12,
    0,
    /* articulation 88 */
    32767,
    32593,
    0,
    31935,
    32767,
    95,
    0,
    0,
    0,
    0,
    1162,
    0,
    4053,
    7930,
    2,
    12,
    0,
    /* articulation 89 */
    380,
    32684,
    6208,
    31935,
    32767,
    112,
    0,
    0,
    0,
    0,
    989,
    0,
    0,
    8887,
    0,
    0,
    0,
    /* articulation 90 */
    19021,
    32663,
    18820,
    23749,
    1268,
    95,
    0,
    0,
    0,
    34,
    989,
    0,
    5113,
    7981,
    0,
    4,
    0,
    /* articulation 91 */
    1902,
    32663,
    6208,
    30725,
    1902,
    127,
    0,
    0,
    0,
    34,
    989,
    0,
    3500,
    7877,
    0,
    5,
    0,
    /* articulation 92 */
    1902,
    32663,
    6208,
    30725,
    1268,
    95,
    0,
    0,
    0,
    34,
    951,
    0,
    4773,
    8355,
    0,
    5,
    0,
    /* articulation 93 */
    476,
    32663,
    10809,
    31935,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 94 */
    3804,
    32663,
    18820,
    30234,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 95 */
    7608,
    32663,
    18820,
    17213,
    2536,
    261,
    0,
    0,
    0,
    34,
    989,
    0,
    1200,
    11690,
    0,
    4,
    0,
    /* articulation 96 */
    32767,
    32468,
    15076,
    30234,
    32767,
    32767,
    32767,
    0,
    0,
    36,
    2183,
    0,
    0,
    11919,
    1,
    0,
    0,
    /* articulation 97 */
    32767,
    0,
    32767,
    32663,
    380,
    32767,
    32767,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 98 */
    32767,
    31391,
    0,
    31391,
    32767,
    634,
    0,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 99 */
    32767,
    32558,
    0,
    23749,
    1268,
    190,
    13107,
    0,
    0,
    34,
    989,
    0,
    3200,
    8321,
    0,
    0,
    0,
    /* articulation 100 */
    32767,
    0,
    32767,
    0,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 101 */
    32767,
    32072,
    0,
    23749,
    32767,
    1087,
    0,
    0,
    0,
    34,
    989,
    0,
    8187,
    5535,
    0,
    5,
    0,
    /* articulation 102 */
    32767,
    32558,
    0,
    29434,
    32767,
    190,
    7667,
    0,
    5,
    0,
    989,
    0,
    6053,
    5535,
    0,
    5,
    0,
    /* articulation 103 */
    32767,
    32663,
    18820,
    23749,
    1902,
    95,
    0,
    0,
    0,
    0,
    989,
    0,
    2700,
    9852,
    0,
    0,
    0,
    /* articulation 104 */
    32767,
    32663,
    18820,
    27897,
    1902,
    95,
    0,
    0,
    0,
    0,
    989,
    0,
    2700,
    9852,
    0,
    0,
    0,
    /* articulation 105 */
    32767,
    32663,
    18820,
    23749,
    32767,
    1268,
    0,
    0,
    0,
    52,
    951,
    0,
    2500,
    10490,
    1,
    8,
    0,
    /* articulation 106 */
    32767,
    32663,
    23493,
    23749,
    32767,
    380,
    0,
    0,
    0,
    34,
    988,
    0,
    4000,
    10223,
    1,
    4,
    0,
    /* articulation 107 */
    32767,
    32663,
    18820,
    27897,
    32767,
    126,
    7667,
    0,
    0,
    0,
    989,
    0,
    1813,
    9154,
    0,
    0,
    0,
    /* articulation 108 */
    32767,
    31730,
    0,
    31730,
    32767,
    380,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 109 */
    32767,
    31180,
    0,
    30484,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    11690,
    0,
    0,
    0,
    /* articulation 110 */
    32767,
    30725,
    0,
    30725,
    32767,
    380,
    0,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 111 */
    32767,
    32349,
    18820,
    27897,
    32767,
    95,
    0,
    0,
    12,
    34,
    951,
    0,
    3000,
    10223,
    0,
    0,
    0,
    /* articulation 112 */
    32767,
    32349,
    18820,
    27897,
    32767,
    63,
    0,
    0,
    12,
    34,
    951,
    0,
    1900,
    10031,
    0,
    0,
    0,
    /* articulation 113 */
    32767,
    32663,
    18820,
    26439,
    32767,
    63,
    0,
    0,
    12,
    34,
    988,
    0,
    1000,
    11107,
    0,
    0,
    0,
    /* articulation 114 */
    32767,
    32663,
    18820,
    26439,
    32767,
    63,
    0,
    0,
    12,
    34,
    988,
    0,
    2000,
    11107,
    0,
    0,
    0,
    /* articulation 115 */
    32767,
    32505,
    0,
    26439,
    32767,
    190,
    0,
    0,
    0,
    17,
    989,
    0,
    4000,
    8321,
    0,
    0,
    0,
    /* articulation 116 */
    32767,
    31832,
    19893,
    9042,
    32767,
    476,
    0,
    0,
    0,
    34,
    1452,
    0,
    0,
    11919,
    0,
    0,
    0,
    /* articulation 117 */
    19021,
    32072,
    23493,
    9042,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    1355,
    0,
    0,
    11877,
    1,
    0,
    0,
    /* articulation 118 */
    32767,
    32468,
    0,
    23749,
    32767,
    190,
    0,
    0,
    0,
    34,
    989,
    0,
    3500,
    9023,
    0,
    0,
    0,
    /* articulation 119 */
    32767,
    17213,
    23493,
    0,
    32767,
    32767,
    32767,
    0,
    0,
    17,
    1521,
    0,
    0,
    10925,
    1,
    0,
    0,
    /* articulation 120 */
    32767,
    32505,
    0,
    26439,
    32767,
    190,
    0,
    0,
    0,
    52,
    989,
    0,
    3200,
    8721,
    0,
    4,
    0,
    /* articulation 121 */
    3804,
    32663,
    18820,
    23749,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    1,
    0,
    0,
    /* articulation 122 */
    9511,
    32663,
    18820,
    25329,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    989,
    0,
    0,
    11877,
    0,
    8,
    0,
    /* articulation 123 */
    32767,
    32663,
    18820,
    23749,
    32767,
    32,
    0,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 124 */
    32767,
    32558,
    0,
    23749,
    32767,
    380,
    0,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 125 */
    32767,
    32663,
    18820,
    23749,
    32767,
    24,
    0,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 126 */
    32767,
    30725,
    0,
    30725,
    32767,
    761,
    0,
    0,
    0,
    0,
    989,
    0,
    3000,
    10223,
    0,
    8,
    0,
    /* articulation 127 */
    127,
    0,
    32767,
    32349,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    1522,
    0,
    0,
    11423,
    4,
    0,
    0,
    /* articulation 128 */
    951,
    32422,
    0,
    32387,
    32767,
    19,
    0,
    0,
    0,
    0,
    989,
    0,
    0,
    11423,
    0,
    0,
    0,
    /* articulation 129 */
    391,
    0,
    0,
    31180,
    190,
    32767,
    32767,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 130 */
    32767,
    30725,
    0,
    30725,
    32767,
    761,
    0,
    0,
    0,
    0,
    989,
    1200,
    0,
    0,
    0,
    0,
    0,
    /* articulation 131 */
    32767,
    31730,
    0,
    31935,
    32767,
    380,
    0,
    0,
    0,
    0,
    989,
    50,
    0,
    0,
    0,
    0,
    0,
    /* articulation 132 */
    32767,
    32072,
    0,
    32072,
    32767,
    19021,
    0,
    0,
    0,
    0,
    989,
    0,
    4700,
    7769,
    0,
    0,
    0,
    /* articulation 133 */
    32767,
    30073,
    0,
    30073,
    32767,
    32767,
    0,
    0,
    0,
    0,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 134 */
    32767,
    32558,
    32767,
    32558,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 135 */
    32767,
    32663,
    18820,
    17213,
    32767,
    190,
    0,
    0,
    10,
    34,
    951,
    0,
    2000,
    10696,
    0,
    0,
    0,
    /* articulation 136 */
    32767,
    32663,
    10809,
    17213,
    32767,
    190,
    0,
    0,
    12,
    34,
    982,
    0,
    0,
    10910,
    0,
    0,
    0,
    /* articulation 137 */
    32767,
    32663,
    18820,
    17213,
    32767,
    190,
    0,
    0,
    10,
    34,
    951,
    0,
    1200,
    10218,
    0,
    0,
    0,
    /* articulation 138 */
    32767,
    32663,
    18820,
    17213,
    32767,
    190,
    0,
    0,
    10,
    34,
    951,
    0,
    1100,
    9525,
    0,
    0,
    0,
    /* articulation 139 */
    19021,
    32558,
    18820,
    23749,
    32767,
    19,
    0,
    0,
    10,
    34,
    988,
    0,
    2000,
    10962,
    0,
    0,
    0,
    /* articulation 140 */
    32767,
    32349,
    18820,
    23749,
    19021,
    634,
    0,
    0,
    10,
    32,
    1008,
    0,
    1200,
    10090,
    0,
    0,
    0,
    /* articulation 141 */
    2536,
    0,
    32767,
    27897,
    1902,
    380,
    0,
    0,
    7,
    34,
    988,
    0,
    1620,
    8933,
    0,
    0,
    0,
    /* articulation 142 */
    32767,
    32349,
    10809,
    23749,
    32767,
    380,
    0,
    0,
    7,
    34,
    988,
    0,
    2200,
    8994,
    0,
    0,
    0,
    /* articulation 143 */
    32767,
    32663,
    15076,
    23749,
    32767,
    1902,
    0,
    0,
    10,
    34,
    982,
    0,
    2500,
    9525,
    0,
    0,
    0,
    /* articulation 144 */
    32767,
    32663,
    15076,
    23749,
    32767,
    190,
    0,
    0,
    10,
    34,
    951,
    0,
    1500,
    11423,
    0,
    0,
    0,
    /* articulation 145 */
    32767,
    32663,
    18820,
    23749,
    32767,
    1902,
    0,
    0,
    9,
    34,
    982,
    0,
    1500,
    9521,
    0,
    0,
    0,
    /* articulation 146 */
    3804,
    0,
    32767,
    28809,
    32767,
    32767,
    32767,
    0,
    0,
    0,
    1521,
    0,
    0,
    9521,
    0,
    0,
    0,
    /* articulation 147 */
    32767,
    32558,
    0,
    23749,
    32767,
    19021,
    0,
    0,
    0,
    17,
    989,
    0,
    5000,
    10223,
    0,
    0,
    0,
    /* articulation 148 */
    32767,
    32663,
    18820,
    23749,
    32767,
    63,
    0,
    0,
    10,
    34,
    951,
    0,
    1500,
    9907,
    0,
    0,
    0,
    /* articulation 149 */
    32767,
    32698,
    11682,
    23749,
    32767,
    1902,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 150 */
    32767,
    32072,
    0,
    32072,
    32767,
    380,
    0,
    0,
    0,
    17,
    989,
    0,
    3440,
    9260,
    0,
    0,
    0,
    /* articulation 151 */
    32767,
    30234,
    0,
    30725,
    32767,
    1902,
    0,
    0,
    0,
    17,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 152 */
    32767,
    31730,
    0,
    30725,
    32767,
    380,
    0,
    0,
    0,
    17,
    989,
    0,
    4000,
    7823,
    0,
    0,
    0,
    /* articulation 153 */
    32767,
    32558,
    3566,
    23749,
    783,
    32767,
    32767,
    0,
    100,
    0,
    1522,
    500,
    0,
    11877,
    0,
    0,
    0,
    /* articulation 154 */
    32767,
    32663,
    18820,
    17213,
    32767,
    1902,
    0,
    0,
    8,
    34,
    989,
    -22,
    0,
    0,
    0,
    0,
    0,
    /* articulation 155 */
    19021,
    29007,
    6784,
    23749,
    32767,
    1902,
    0,
    0,
    0,
    34,
    951,
    0,
    5000,
    9521,
    1,
    0,
    0,
    /* articulation 156 */
    32767,
    32558,
    0,
    31935,
    1902,
    254,
    16384,
    0,
    0,
    52,
    989,
    0,
    3627,
    10547,
    0,
    5,
    0,
    /* articulation 157 */
    3804,
    0,
    32767,
    23749,
    1902,
    1902,
    0,
    0,
    0,
    34,
    989,
    27,
    0,
    11919,
    0,
    0,
    0,
    /* articulation 158 */
    32767,
    0,
    32767,
    31730,
    76,
    66,
    10092,
    0,
    5,
    0,
    989,
    0,
    8007,
    5535,
    0,
    8,
    0,
    /* articulation 159 */
    32767,
    32468,
    0,
    29434,
    32767,
    127,
    0,
    0,
    0,
    52,
    989,
    0,
    2500,
    9032,
    0,
    0,
    0,
    /* articulation 160 */
    9511,
    32663,
    10809,
    25329,
    32767,
    32767,
    32767,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 161 */
    19021,
    32558,
    18820,
    23749,
    32767,
    190,
    0,
    0,
    10,
    34,
    988,
    0,
    2600,
    9513,
    0,
    0,
    0,
    /* articulation 162 */
    32767,
    32106,
    9568,
    23749,
    2348,
    391,
    0,
    0,
    10,
    52,
    980,
    0,
    6500,
    9023,
    0,
    0,
    0,
    /* articulation 163 */
    32767,
    32558,
    0,
    26439,
    32767,
    63,
    0,
    0,
    0,
    34,
    989,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 164 */
    32767,
    32072,
    15076,
    17213,
    32767,
    1268,
    0,
    0,
    0,
    0,
    951,
    0,
    2000,
    10223,
    0,
    0,
    0,
    /* articulation 165 */
    32767,
    32558,
    0,
    23749,
    32767,
    380,
    0,
    0,
    0,
    34,
    989,
    0,
    3000,
    9366,
    0,
    0,
    0,
    /* articulation 166 */
    32767,
    32663,
    18820,
    23749,
    1902,
    127,
    10879,
    0,
    0,
    0,
    989,
    0,
    6000,
    7121,
    0,
    4,
    0,
    /* articulation 167 */
    32767,
    32505,
    0,
    26439,
    32767,
    21,
    0,
    0,
    0,
    52,
    989,
    0,
    3500,
    6236,
    0,
    5,
    0,
    /* articulation 168 */
    32767,
    32505,
    0,
    26439,
    32767,
    190,
    0,
    0,
    0,
    52,
    989,
    0,
    2800,
    7121,
    0,
    0,
    0,
    /* articulation 169 */
    32767,
    32418,
    0,
    29434,
    32767,
    127,
    0,
    0,
    0,
    52,
    989,
    0,
    2100,
    9626,
    0,
    0,
    0,
    /* articulation 170 */
    32767,
    32349,
    0,
    30234,
    32767,
    127,
    0,
    0,
    0,
    52,
    989,
    0,
    3000,
    9626,
    0,
    0,
    0,
    /* articulation 171 */
    32767,
    32288,
    0,
    28400,
    32767,
    127,
    0,
    0,
    0,
    52,
    989,
    0,
    1000,
    9032,
    0,
    0,
    0,
    /* articulation 172 */
    32767,
    32072,
    0,
    28809,
    32767,
    127,
    0,
    0,
    0,
    52,
    989,
    0,
    1000,
    9032,
    0,
    0,
    0,
    /* articulation 173 */
    3804,
    32072,
    15076,
    17213,
    32767,
    1268,
    0,
    0,
    0,
    52,
    991,
    0,
    0,
    11107,
    0,
    8,
    0,
    /* articulation 174 */
    32767,
    32349,
    15076,
    23749,
    7608,
    147,
    0,
    0,
    0,
    0,
    989,
    0,
    4500,
    9521,
    0,
    8,
    0,
    /* articulation 175 */
    32767,
    32663,
    18820,
    23749,
    32767,
    95,
    0,
    0,
    0,
    0,
    989,
    0,
    2000,
    8321,
    0,
    8,
    0,
    /* articulation 176 */
    32767,
    32715,
    128,
    29669,
    32767,
    1729,
    0,
    0,
    0,
    0,
    989,
    0,
    6000,
    7823,
    0,
    8,
    0,
    /* articulation 177 */
    19021,
    32448,
    0,
    31882,
    32767,
    95,
    0,
    0,
    0,
    0,
    989,
    0,
    4500,
    7121,
    0,
    8,
    0,
    /* articulation 178 */
    32767,
    32560,
    3646,
    32107,
    32767,
    190,
    0,
    0,
    0,
    0,
    989,
    0,
    4000,
    8321,
    0,
    8,
    0,
    /* articulation 179 */
    32767,
    32602,
    13644,
    26439,
    32767,
    63,
    0,
    0,
    12,
    34,
    988,
    0,
    2000,
    11107,
    0,
    0,
    0,
    /* articulation 180 */
    19021,
    30484,
    0,
    23749,
    32767,
    1902,
    0,
    0,
    0,
    0,
    989,
    0,
    5000,
    8321,
    1,
    0,
    0,
    /* articulation 181 */
    261,
    32466,
    0,
    31938,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 182 */
    32767,
    32418,
    0,
    31742,
    2348,
    39,
    0,
    0,
    0,
    34,
    989,
    0,
    3600,
    7121,
    0,
    4,
    0,
    /* articulation 183 */
    32767,
    32090,
    0,
    32090,
    32767,
    634,
    0,
    0,
    0,
    34,
    951,
    0,
    0,
    0,
    0,
    0,
    0,
    /* articulation 184 */
    1669,
    32715,
    19242,
    30194,
    32767,
    296,
    0,
    0,
    0,
    0,
    1555,
    0,
    3000,
    9907,
    0,
    4,
    0
  };

  private SonivoxWaveData() {}
}
