/*
 * Copyright 2016, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.owntracks.android.test

import androidx.test.espresso.IdlingResource
import androidx.test.espresso.IdlingResource.ResourceCallback
import java.util.concurrent.atomic.AtomicBoolean
import org.jetbrains.annotations.NotNull

/**
 * A very simple implementation of [IdlingResource].
 *
 * Consider using CountingIdlingResource from espresso-contrib package if you use this class from
 * multiple threads or need to keep a count of pending operations.
 */
class SimpleIdlingResource(private val resourceName: @NotNull String, initialIdlingState: Boolean) :
    IdlingResource {
  @Volatile private var mCallback: ResourceCallback? = null

  // Idleness is controlled with this boolean.
  private val mIsIdleNow = AtomicBoolean(initialIdlingState)

  override fun getName(): String {
    return this.resourceName
  }

  override fun isIdleNow(): Boolean {
    return mIsIdleNow.get()
  }

  override fun registerIdleTransitionCallback(callback: ResourceCallback) {
    mCallback = callback
  }

  /**
   * Sets the new idle state, if isIdleNow is true, it pings the [ResourceCallback].
   *
   * @param isIdleNow false if there are pending operations, true if idle.
   */
  fun setIdleState(isIdleNow: Boolean) {
    mIsIdleNow.set(isIdleNow)
    if (isIdleNow && mCallback != null) {
      mCallback!!.onTransitionToIdle()
    }
  }
}
