"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageChannel = void 0;
const MessageStream_1 = require("./MessageStream");
class MessageChannel {
    constructor() {
        this.__activeChannels = new Set();
        this.__pendingErrorMessages = new Map();
    }
    registerListener(context) {
        const { referer } = context.headers;
        if (!referer) {
            throw new Error('Cannot set up a message channel without a referer.');
        }
        const messageChannel = new MessageStream_1.MessageStream(referer, context.socket, context);
        this.__activeChannels.add(messageChannel);
        messageChannel.addListener('closed', () => {
            this.__activeChannels.delete(messageChannel);
        });
        messageChannel.open();
        const now = Date.now();
        // send error messages that occurred within 1sec before opening the new
        // message channel. this helps catch errors that happen while loading a
        // page when the message channel is not set up yet
        this.__pendingErrorMessages.forEach((timestamp, message) => {
            if (now - timestamp <= 1000) {
                messageChannel.sendMessage('error-message', message);
            }
        });
        this.__pendingErrorMessages.clear();
    }
    sendMessage({ name, data, listeners }) {
        if (name === 'error-message') {
            if (data) {
                this.__pendingErrorMessages.set(data, Date.now());
            }
        }
        for (const channel of this.__activeChannels) {
            if (!listeners || listeners.includes(channel.referer)) {
                channel.sendMessage(name, data);
            }
        }
    }
}
exports.MessageChannel = MessageChannel;
//# sourceMappingURL=MessageChannel.js.map