"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createConfig = void 0;
const tslib_1 = require("tslib");
const path_1 = tslib_1.__importDefault(require("path"));
const utils_1 = require("./utils/utils");
const constants_1 = require("./constants");
/**
 * Creates dev server config with default settings
 */
function createConfig(config) {
    const { babel = false, babelConfig, babelExclude = [], babelModernExclude = [], babelModuleExclude = [], customBabelInclude = [], customBabelExclude = [], basePath, compress = true, fileExtensions: fileExtensionsArg, hostname, http2 = false, logStartup, open = false, cors = false, port, sslCert, sslKey, watch = false, logErrorsToBrowser = false, polyfillsLoader, plugins = [], responseTransformers = [], debug = false, nodeResolve: nodeResolveArg = false, eventStream = true, dedupeModules, moduleDirs = ['node_modules', 'web_modules'], preserveSymlinks = false, onServerStart, } = config;
    if (debug) {
        utils_1.setDebug(true);
    }
    let { compatibility = constants_1.compatibilityModes.AUTO } = config;
    if (compatibility === 'modern' || compatibility === 'all') {
        /* eslint-disable-next-line no-console */
        console.warn('[es-dev-server] Compatibility mode "modern" and "all" are deprecated, and combined into "auto".' +
            '"auto" mode is turned on by default.');
        compatibility = constants_1.compatibilityModes.AUTO;
    }
    if (!Object.values(constants_1.compatibilityModes).includes(compatibility)) {
        throw new Error(`Unknown compatibility mode: ${compatibility}`);
    }
    // middlewares used to be called customMiddlewares
    const middlewares = [
        // use .middleware for fowards compatibility
        ...(config.middleware || []),
        ...(config.middlewares || config.customMiddlewares || []),
    ];
    let { appIndex, rootDir = process.cwd() } = config;
    let appIndexDir;
    // ensure rootDir is a fully resolved path, for example if you set ../../
    // in the config or cli, it's resolved relative to the current working directory
    rootDir = path_1.default.resolve(rootDir);
    // resolve appIndex relative to rootDir and transform it to a browser path
    if (appIndex) {
        if (path_1.default.isAbsolute(appIndex)) {
            appIndex = `/${utils_1.toBrowserPath(path_1.default.relative(rootDir, appIndex))}`;
        }
        else if (!appIndex.startsWith('/')) {
            appIndex = `/${utils_1.toBrowserPath(appIndex)}`;
        }
        else {
            appIndex = utils_1.toBrowserPath(appIndex);
        }
        appIndexDir = `${appIndex.substring(0, appIndex.lastIndexOf('/'))}`;
    }
    // parse `open` option, based on whether it's a boolean or a string
    let openPath;
    if (typeof open === 'string' && open !== '') {
        // user-provided open path
        openPath = open;
    }
    else if (appIndex) {
        // if an appIndex was provided, use it's directory as open path
        openPath = `${basePath || ''}${appIndexDir}/`;
    }
    else {
        openPath = basePath ? `${basePath}/` : '/';
    }
    const fileExtensions = [
        ...(fileExtensionsArg || []),
        '.mjs',
        '.js',
        '.cjs',
        '.jsx',
        '.ts',
        '.tsx',
    ];
    let nodeResolve = nodeResolveArg;
    // some node resolve options can be set separately for convenience, primarily
    // for the command line args. we merge them into a node resolve options object
    if (nodeResolveArg != null &&
        nodeResolveArg !== false &&
        (moduleDirs != null || preserveSymlinks != null || dedupeModules != null)) {
        nodeResolve = {
            // user provided options, if any
            ...(typeof nodeResolveArg === 'object' ? nodeResolveArg : {}),
            customResolveOptions: {
                moduleDirectory: moduleDirs,
                preserveSymlinks,
            },
        };
        if (dedupeModules) {
            nodeResolve.dedupe =
                dedupeModules === true ? importee => !['.', '/'].includes(importee[0]) : dedupeModules;
        }
    }
    const polyfillsLoaderConfig = typeof polyfillsLoader === 'boolean' ? undefined : polyfillsLoader;
    return {
        appIndex,
        appIndexDir,
        babelExclude,
        babelModernExclude,
        babelModuleExclude,
        customBabelInclude,
        customBabelExclude,
        basePath,
        compatibilityMode: compatibility,
        polyfillsLoader: polyfillsLoader !== false,
        polyfillsLoaderConfig,
        compress,
        customBabelConfig: babelConfig,
        customMiddlewares: middlewares,
        responseTransformers,
        eventStream,
        fileExtensions,
        hostname,
        http2,
        plugins,
        logStartup: !!logStartup,
        nodeResolve,
        openBrowser: open === true || typeof open === 'string',
        openPath,
        cors,
        port,
        readUserBabelConfig: babel,
        rootDir,
        sslCert,
        sslKey,
        watch,
        logErrorsToBrowser,
        watchDebounce: 100,
        onServerStart,
    };
}
exports.createConfig = createConfig;
//# sourceMappingURL=config.js.map