"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseStackTrace = void 0;
function createRawFrame(raw) {
    return {
        column: -1,
        fileName: "",
        line: -1,
        name: "",
        raw: raw,
        sourceColumn: -1,
        sourceFileName: "",
        sourceLine: -1,
        type: "",
    };
}
var FIREFOX_WEBKIT = /([^@]+|^)@(.*):(\d+):(\d+)/;
var WEBKIT_ADDRESS_UNNAMED = /^(http(s)?:\/\/.*):(\d+):(\d+)$/;
var WEBKIT_NATIVE_UNNAMED = "[native code]";
function parsWebkit(str) {
    if (str.includes(WEBKIT_NATIVE_UNNAMED)) {
        return {
            line: -1,
            column: -1,
            type: "native",
            fileName: "",
            name: "",
            raw: str,
            sourceColumn: -1,
            sourceFileName: "",
            sourceLine: -1,
        };
    }
    var match = str.match(WEBKIT_ADDRESS_UNNAMED);
    if (match) {
        var line = match[3] ? +match[3] : -1;
        var column = match[4] ? +match[4] : -1;
        var fileName = match[1] ? match[1] : "";
        return {
            line: line,
            column: column,
            type: "",
            fileName: fileName,
            name: "",
            raw: str,
            sourceColumn: -1,
            sourceFileName: "",
            sourceLine: -1,
        };
    }
    return createRawFrame(str);
}
function parseFirefoxWebkit(lines) {
    return lines.map(function (str) {
        var match = str.match(FIREFOX_WEBKIT);
        if (!match) {
            return parsWebkit(str);
        }
        var line = match[3] ? +match[3] : -1;
        var column = match[4] ? +match[4] : -1;
        var fileName = match[2] ? match[2] : "";
        return {
            line: line,
            column: column,
            type: match[0] ? "" : "native",
            fileName: fileName,
            name: (match[1] || "").trim(),
            raw: str,
            sourceColumn: -1,
            sourceFileName: "",
            sourceLine: -1,
        };
    });
}
// foo.bar.js:123:39
// foo.bar.js:123:39 <- original.js:123:34
var CHROME_MAPPED = /(.*?):(\d+):(\d+)(\s<-\s(.+):(\d+):(\d+))?/;
function parseMapped(frame, maybeMapped) {
    var match = maybeMapped.match(CHROME_MAPPED);
    if (match) {
        frame.fileName = match[1];
        frame.line = +match[2];
        frame.column = +match[3];
        if (match[5])
            frame.sourceFileName = match[5];
        if (match[6])
            frame.sourceLine = +match[6];
        if (match[7])
            frame.sourceColumn = +match[7];
    }
}
// at <SomeFramework>
var CHROME_IE_NATIVE_NO_LINE = /^at\s(<.*>)$/;
// at <SomeFramework>:123:39
var CHROME_IE_NATIVE = /^\s*at\s(<.*>):(\d+):(\d+)$/;
// at foo.bar(bob) (foo.bar.js:123:39)
// at foo.bar(bob) (foo.bar.js:123:39 <- original.js:123:34)
var CHROME_IE_FUNCTION = /^at\s(.*)\s\((.*)\)$/;
// >= Chrome 88
// spy() at Component.Foo [as constructor] (original.js:123:34)
// spy() at Component.Foo [as constructor] (foo.bar.js:123:39 <- original.js:123:34)
var CHROME_IE_FUNCTION_WITH_CALL = /^([\w\W]*)\s\((.*)\)/;
var CHROME_IE_DETECTOR = /\s*at\s.+/;
// at about:blank:1:7
// >= Chrome 99
// at /projects/foo.test.js:689:1 <- /projects/foo.test.js:10:1
var CHROME_BLANK = /\s*at\s(.*):(\d+):(\d+)$/;
function parseChromeIe(lines) {
    // Many frameworks mess with error.stack. So we use this check
    // to find the first line of the actual stack
    var start = lines.findIndex(function (line) { return CHROME_IE_DETECTOR.test(line); });
    if (start === -1) {
        return [];
    }
    var frames = [];
    for (var i = start; i < lines.length; i++) {
        var str = lines[i].replace(/^\s+|\s+$/g, "");
        var frame = createRawFrame(lines[i]);
        var nativeNoLine = str.match(CHROME_IE_NATIVE_NO_LINE);
        if (nativeNoLine) {
            frame.fileName = nativeNoLine[1];
            frame.type = "native";
            frames.push(frame);
            continue;
        }
        var native = str.match(CHROME_IE_NATIVE);
        if (native) {
            frame.fileName = native[1];
            frame.type = "native";
            if (native[2])
                frame.line = +native[2];
            if (native[3])
                frame.column = +native[3];
            frames.push(frame);
            continue;
        }
        var withFn = str.match(CHROME_IE_FUNCTION);
        if (withFn) {
            frame.name = withFn[1];
            parseMapped(frame, withFn[2]);
            frames.push(frame);
            continue;
        }
        var blank = str.match(CHROME_BLANK);
        if (blank) {
            frame.fileName = blank[1];
            frame.line = +blank[2];
            frame.column = +blank[3];
            parseMapped(frame, blank[1] + ":" + blank[2] + ":" + blank[3]);
            frames.push(frame);
            continue;
        }
        var withFnCall = str.match(CHROME_IE_FUNCTION_WITH_CALL);
        if (withFnCall) {
            frame.name = withFnCall[1];
            parseMapped(frame, withFnCall[2]);
            frames.push(frame);
            continue;
        }
        frames.push(frame);
    }
    return frames;
}
function parseStackTrace(stack) {
    if (!stack)
        return [];
    var lines = stack.split("\n").filter(Boolean);
    // Libraries like node's "assert" module mess with the stack trace by
    // prepending custom data. So we need to do a precheck, to determine
    // which browser the trace is coming from.
    if (lines.some(function (line) { return CHROME_IE_DETECTOR.test(line); })) {
        return parseChromeIe(lines);
    }
    return parseFirefoxWebkit(lines);
}
exports.parseStackTrace = parseStackTrace;
//# sourceMappingURL=index.js.map