"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TestSessionManager = void 0;
const EventEmitter_js_1 = require("../utils/EventEmitter.js");
function* filtered(it, filter) {
    while (true) {
        const { value, done } = it.next();
        if (done)
            return;
        if (filter(value))
            yield value;
    }
}
class TestSessionManager extends EventEmitter_js_1.EventEmitter {
    constructor(groups, sessions) {
        super();
        this.sessionsMap = new Map();
        this.debugSessions = new Map();
        this._groups = groups;
        for (const session of sessions) {
            this.sessionsMap.set(session.id, session);
        }
    }
    addDebug(...sessions) {
        for (const session of sessions) {
            this.debugSessions.set(session.id, session);
        }
    }
    updateStatus(session, status) {
        const updatedSession = Object.assign(Object.assign({}, session), { status });
        this.update(updatedSession);
        this.emit('session-status-updated', updatedSession);
    }
    update(session) {
        if (!this.sessionsMap.has(session.id)) {
            throw new Error(`Unknown session: ${session.id}`);
        }
        this.sessionsMap.set(session.id, session);
        this.emit('session-updated', undefined);
    }
    groups() {
        return this._groups;
    }
    get(id) {
        return this.sessionsMap.get(id);
    }
    all() {
        return this.sessionsMap.values();
    }
    filtered(filter) {
        return filtered(this.all(), filter);
    }
    forStatus(...statuses) {
        return this.filtered(s => statuses.includes(s.status));
    }
    forStatusAndTestFile(testFile, ...statuses) {
        return this.filtered(s => statuses.includes(s.status) && (!testFile || s.testFile === testFile));
    }
    forTestFile(...testFiles) {
        return this.filtered(s => testFiles.includes(s.testFile));
    }
    forBrowser(browser) {
        return this.filtered(s => s.browser === browser);
    }
    forGroup(groupName) {
        return this.filtered(s => s.group.name === groupName);
    }
    forBrowserName(browserName) {
        return this.filtered(s => s.browser.name === browserName);
    }
    forBrowserNames(browserNames) {
        return this.filtered(s => browserNames.includes(s.browser.name));
    }
    passed() {
        return this.filtered(s => s.passed);
    }
    failed() {
        return this.filtered(s => !s.passed);
    }
    getDebug(id) {
        return this.debugSessions.get(id);
    }
    getAllDebug() {
        return this.debugSessions.values();
    }
    setDebug(debugSession) {
        this.debugSessions.set(debugSession.id, debugSession);
    }
    removeDebug(id) {
        this.debugSessions.delete(id);
    }
}
exports.TestSessionManager = TestSessionManager;
//# sourceMappingURL=TestSessionManager.js.map