"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamicTerminal = void 0;
const log_update_1 = __importDefault(require("log-update"));
const cli_cursor_1 = __importDefault(require("cli-cursor"));
const BufferedConsole_js_1 = require("./BufferedConsole.js");
const EventEmitter_js_1 = require("../../utils/EventEmitter.js");
const CLEAR_COMMAND = process.platform === 'win32' ? '\x1B[2J\x1B[0f' : '\x1B[2J\x1B[3J\x1B[H';
class DynamicTerminal extends EventEmitter_js_1.EventEmitter {
    constructor() {
        super(...arguments);
        this.originalFunctions = {};
        this.previousDynamic = [];
        this.started = false;
        this.bufferedConsole = new BufferedConsole_js_1.BufferedConsole();
        this.pendingConsoleFlush = false;
        this.isInteractive = process.stdout.isTTY;
        this.onStdInData = (input) => {
            this.emit('input', input);
        };
    }
    start() {
        // start off with an empty line
        console.log('');
        this.interceptConsoleOutput();
        if (process.stdin.isTTY === true) {
            process.stdin.resume();
            process.stdin.setEncoding('utf8');
            process.stdin.addListener('data', this.onStdInData);
        }
        this.started = true;
    }
    observeDirectInput() {
        if (!this.isInteractive) {
            throw new Error('Cannot observe input in a non-interactive (TTY) terminal.');
        }
        if (typeof process.stdin.setRawMode === 'function') {
            process.stdin.setRawMode(true);
        }
        cli_cursor_1.default.hide();
    }
    observeConfirmedInput() {
        if (!this.isInteractive) {
            throw new Error('Cannot observe input in a non-interactive (TTY) terminal.');
        }
        if (typeof process.stdin.setRawMode === 'function') {
            process.stdin.setRawMode(false);
        }
        cli_cursor_1.default.show();
    }
    stop() {
        this.flushConsoleOutput();
        log_update_1.default.done();
        for (const [key, fn] of Object.entries(this.originalFunctions)) {
            // @ts-ignore
            console[key] = fn;
        }
        this.started = false;
        process.stdin.pause();
        process.stdin.removeListener('data', this.onStdInData);
    }
    clear() {
        process.stdout.write(CLEAR_COMMAND);
        this.relogDynamic();
    }
    logStatic(entriesOrEntry) {
        const entries = Array.isArray(entriesOrEntry) ? entriesOrEntry : [entriesOrEntry];
        if (entries.length === 0) {
            return;
        }
        console.log(entries.join('\n'));
    }
    logPendingUserInput(string) {
        process.stdout.write(string);
    }
    logDynamic(entriesOrEntry) {
        const entries = Array.isArray(entriesOrEntry) ? entriesOrEntry : [entriesOrEntry];
        if (!this.started) {
            return;
        }
        this.previousDynamic = entries;
        (0, log_update_1.default)(entries.join('\n'));
    }
    /**
     * Intercepts console output, piping all output to a buffered console instead.
     * Console messages are piped to the regular console at intervals. This is necessary
     * because when logging regular console messages the progress bar needs to be removes
     * and added back at the bottom. The time between this must be as minimal as possible.
     * Regular console logging can take a noticeable amount of time to compute object highlighting
     * and formatting. This causes the progress bar to flicker. Pre-computing the formatting
     * prevents this.
     */
    interceptConsoleOutput() {
        for (const key of Object.keys(console)) {
            if (typeof console[key] === 'function') {
                this.originalFunctions[key] = console[key];
                console[key] = new Proxy(console[key], {
                    apply: (target, thisArg, argArray) => {
                        this.bufferedConsole.console[key](...argArray);
                        if (this.pendingConsoleFlush) {
                            return;
                        }
                        this.pendingConsoleFlush = true;
                        setTimeout(() => {
                            this.flushConsoleOutput();
                        }, 0);
                    },
                });
            }
        }
    }
    flushConsoleOutput() {
        // clear progress bar
        log_update_1.default.clear();
        // log static console messages
        this.bufferedConsole.flush();
        // put progress bar back
        this.relogDynamic();
        this.pendingConsoleFlush = false;
    }
    relogDynamic() {
        this.logDynamic(this.previousDynamic);
    }
}
exports.DynamicTerminal = DynamicTerminal;
//# sourceMappingURL=DynamicTerminal.js.map