"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TestRunnerCli = exports.MENUS = void 0;
const code_frame_1 = require("@babel/code-frame");
const path_1 = __importDefault(require("path"));
const nanocolors_1 = require("nanocolors");
const open_1 = __importDefault(require("open"));
const writeCoverageReport_js_1 = require("./writeCoverageReport.js");
const getSelectFilesMenu_js_1 = require("./getSelectFilesMenu.js");
const getWatchCommands_js_1 = require("./getWatchCommands.js");
const DynamicTerminal_js_1 = require("./terminal/DynamicTerminal.js");
const BufferedLogger_js_1 = require("./BufferedLogger.js");
const getManualDebugMenu_js_1 = require("./getManualDebugMenu.js");
const TestSessionStatus_js_1 = require("../test-session/TestSessionStatus.js");
exports.MENUS = {
    NONE: 'none',
    OVERVIEW: 'overview',
    FOCUS_SELECT_FILE: 'focus',
    DEBUG_SELECT_FILE: 'debug',
    MANUAL_DEBUG: 'manual-debug',
};
const KEYCODES = {
    ENTER: '\r',
    ESCAPE: '\u001b',
    CTRL_C: '\u0003',
    CTRL_D: '\u0004',
};
class TestRunnerCli {
    constructor(config, runner) {
        this.terminal = new DynamicTerminal_js_1.DynamicTerminal();
        this.reportedFilesByTestRun = new Map();
        this.activeMenu = exports.MENUS.NONE;
        this.menuSucceededAndPendingFiles = [];
        this.menuFailedFiles = [];
        this.pendingReportPromises = [];
        this.lastStaticLog = -1;
        this.config = config;
        this.runner = runner;
        this.logger = this.config.logger;
        this.sessions = runner.sessions;
        this.localAddress = `${this.config.protocol}//${this.config.hostname}:${this.config.port}/`;
        if (config.watch && !this.terminal.isInteractive) {
            this.runner.stop(new Error('Cannot run watch mode in a non-interactive (TTY) terminal.'));
        }
    }
    start() {
        var _a;
        this.setupTerminalEvents();
        this.setupRunnerEvents();
        this.terminal.start();
        for (const reporter of this.config.reporters) {
            (_a = reporter.start) === null || _a === void 0 ? void 0 : _a.call(reporter, {
                config: this.config,
                sessions: this.sessions,
                testFiles: this.runner.testFiles,
                startTime: this.runner.startTime,
                browsers: this.runner.browsers,
                browserNames: this.runner.browserNames,
            });
        }
        this.switchMenu(this.config.manual ? exports.MENUS.MANUAL_DEBUG : exports.MENUS.OVERVIEW);
        if (this.config.watch || (this.config.manual && this.terminal.isInteractive)) {
            this.terminal.observeDirectInput();
        }
        if (this.config.staticLogging || !this.terminal.isInteractive) {
            this.logger.log((0, nanocolors_1.bold)(`Running ${this.runner.testFiles.length} test files...\n`));
        }
        if (this.config.open) {
            (0, open_1.default)(this.localAddress);
        }
    }
    setupTerminalEvents() {
        this.terminal.on('input', key => {
            var _a;
            if ([exports.MENUS.DEBUG_SELECT_FILE, exports.MENUS.FOCUS_SELECT_FILE].includes(this.activeMenu)) {
                const i = Number(key);
                if (!Number.isNaN(i)) {
                    this.focusTestFileNr(i);
                    return;
                }
            }
            switch (key.toUpperCase()) {
                case KEYCODES.CTRL_C:
                case KEYCODES.CTRL_D:
                case 'Q':
                    if (this.activeMenu === exports.MENUS.OVERVIEW ||
                        (this.config.manual && this.activeMenu === exports.MENUS.MANUAL_DEBUG)) {
                        this.runner.stop();
                    }
                    return;
                case 'D':
                    if (this.activeMenu === exports.MENUS.OVERVIEW) {
                        if (this.runner.focusedTestFile) {
                            this.runner.startDebugBrowser(this.runner.focusedTestFile);
                        }
                        else if (this.runner.testFiles.length === 1) {
                            this.runner.startDebugBrowser(this.runner.testFiles[0]);
                        }
                        else {
                            this.switchMenu(exports.MENUS.DEBUG_SELECT_FILE);
                        }
                    }
                    else if (this.activeMenu === exports.MENUS.MANUAL_DEBUG) {
                        (0, open_1.default)(this.localAddress);
                    }
                    return;
                case 'F':
                    if (this.activeMenu === exports.MENUS.OVERVIEW && this.runner.testFiles.length > 1) {
                        this.switchMenu(exports.MENUS.FOCUS_SELECT_FILE);
                    }
                    return;
                case 'C':
                    if (this.activeMenu === exports.MENUS.OVERVIEW && this.config.coverage) {
                        (0, open_1.default)(`file://${path_1.default.resolve((_a = this.config.coverageConfig.reportDir) !== null && _a !== void 0 ? _a : '', 'lcov-report', 'index.html')}`);
                    }
                    return;
                case 'M':
                    this.switchMenu(exports.MENUS.MANUAL_DEBUG);
                    return;
                case KEYCODES.ESCAPE:
                    if (this.activeMenu === exports.MENUS.OVERVIEW && this.runner.focusedTestFile) {
                        this.runner.focusedTestFile = undefined;
                        this.reportTestResults(true);
                        this.reportTestProgress();
                    }
                    else if (this.activeMenu === exports.MENUS.MANUAL_DEBUG) {
                        this.switchMenu(exports.MENUS.OVERVIEW);
                    }
                    return;
                case KEYCODES.ENTER:
                    this.runner.runTests(this.sessions.all());
                    return;
                default:
                    return;
            }
        });
    }
    setupRunnerEvents() {
        this.sessions.on('session-status-updated', session => {
            if (this.activeMenu !== exports.MENUS.OVERVIEW) {
                return;
            }
            if (session.status === TestSessionStatus_js_1.SESSION_STATUS.FINISHED) {
                this.reportTestResult(session.testFile);
                this.reportTestProgress();
            }
        });
        this.runner.on('test-run-started', ({ testRun }) => {
            var _a;
            for (const reporter of this.config.reporters) {
                (_a = reporter.onTestRunStarted) === null || _a === void 0 ? void 0 : _a.call(reporter, { testRun });
            }
            if (this.activeMenu !== exports.MENUS.OVERVIEW) {
                return;
            }
            if (testRun !== 0 && this.config.watch) {
                this.terminal.clear();
            }
            this.reportTestResults();
            this.reportTestProgress(false);
        });
        this.runner.on('test-run-finished', ({ testRun, testCoverage }) => {
            var _a;
            for (const reporter of this.config.reporters) {
                (_a = reporter.onTestRunFinished) === null || _a === void 0 ? void 0 : _a.call(reporter, {
                    testRun,
                    sessions: Array.from(this.sessions.all()),
                    testCoverage,
                    focusedTestFile: this.runner.focusedTestFile,
                });
            }
            if (this.activeMenu !== exports.MENUS.OVERVIEW) {
                return;
            }
            this.testCoverage = testCoverage;
            if (testCoverage && !this.runner.focusedTestFile) {
                this.writeCoverageReport(testCoverage);
            }
            this.reportSyntaxErrors();
            this.reportTestProgress();
        });
        this.runner.on('finished', () => {
            this.reportEnd();
        });
    }
    focusTestFileNr(i) {
        var _a;
        const focusedTestFile = (_a = this.menuFailedFiles[i - 1]) !== null && _a !== void 0 ? _a : this.menuSucceededAndPendingFiles[i - this.menuFailedFiles.length - 1];
        const debug = this.activeMenu === exports.MENUS.DEBUG_SELECT_FILE;
        if (focusedTestFile) {
            this.runner.focusedTestFile = focusedTestFile;
            this.switchMenu(exports.MENUS.OVERVIEW);
            if (debug) {
                this.runner.startDebugBrowser(focusedTestFile);
            }
        }
        else {
            this.terminal.clear();
            this.logSelectFilesMenu();
        }
    }
    reportTestResults(forceReport = false) {
        const { focusedTestFile } = this.runner;
        const testFiles = focusedTestFile ? [focusedTestFile] : this.runner.testFiles;
        for (const testFile of testFiles) {
            this.reportTestResult(testFile, forceReport);
        }
    }
    reportTestResult(testFile, forceReport = false) {
        var _a;
        const testRun = this.runner.testRun;
        const sessionsForTestFile = Array.from(this.sessions.forTestFile(testFile));
        const allFinished = sessionsForTestFile.every(s => s.status === TestSessionStatus_js_1.SESSION_STATUS.FINISHED);
        if (!allFinished) {
            // not all sessions for this file are finished
            return;
        }
        let reportedFiles = this.reportedFilesByTestRun.get(testRun);
        if (!reportedFiles) {
            reportedFiles = new Set();
            this.reportedFilesByTestRun.set(testRun, reportedFiles);
        }
        if (!forceReport && reportedFiles.has(testFile)) {
            // this was file was already reported
            return;
        }
        reportedFiles.add(testFile);
        const bufferedLogger = new BufferedLogger_js_1.BufferedLogger(this.logger);
        for (const reporter of this.config.reporters) {
            const sessionsForTestFile = Array.from(this.sessions.forTestFile(testFile));
            (_a = reporter.reportTestFileResults) === null || _a === void 0 ? void 0 : _a.call(reporter, {
                logger: bufferedLogger,
                sessionsForTestFile,
                testFile,
                testRun,
            });
        }
        // all the logs from the reportered were buffered, if they finished before a new test run
        // actually log them to the terminal here
        if (this.runner.testRun === testRun) {
            bufferedLogger.logBufferedMessages();
        }
    }
    reportTestProgress(final = false) {
        var _a;
        if (this.config.manual) {
            return;
        }
        const logStatic = this.config.staticLogging || !this.terminal.isInteractive;
        if (logStatic && !final) {
            // print a static progress log only once every 10000ms
            const now = Date.now();
            if (this.lastStaticLog !== -1 && now - this.lastStaticLog < 10000) {
                return;
            }
            this.lastStaticLog = now;
        }
        const reports = [];
        for (const reporter of this.config.reporters) {
            const report = (_a = reporter.getTestProgress) === null || _a === void 0 ? void 0 : _a.call(reporter, {
                sessions: Array.from(this.sessions.all()),
                testRun: this.runner.testRun,
                focusedTestFile: this.runner.focusedTestFile,
                testCoverage: this.testCoverage,
            });
            if (report) {
                reports.push(...report);
            }
        }
        if (this.config.watch) {
            if (this.runner.focusedTestFile) {
                reports.push(`Focused on test file: ${(0, nanocolors_1.cyan)(path_1.default.relative(process.cwd(), this.runner.focusedTestFile))}\n`);
            }
            reports.push(...(0, getWatchCommands_js_1.getWatchCommands)(!!this.config.coverage, this.runner.testFiles, !!this.runner.focusedTestFile), '');
        }
        if (logStatic) {
            this.terminal.logStatic(reports);
        }
        else {
            this.terminal.logDynamic(reports);
        }
    }
    reportSyntaxErrors() {
        var _a;
        // TODO: this special cases the logger of @web/test-runner which implements
        // logging of syntax errors. we need to make this more generic
        const logger = this.config.logger;
        const { loggedSyntaxErrors = new Map() } = logger;
        if (loggedSyntaxErrors.size === 0) {
            return;
        }
        const report = [];
        (_a = logger.clearLoggedSyntaxErrors) === null || _a === void 0 ? void 0 : _a.call(logger);
        for (const [filePath, errors] of loggedSyntaxErrors.entries()) {
            for (const error of errors) {
                const { message, code, line, column } = error;
                const result = (0, code_frame_1.codeFrameColumns)(code, { start: { line, column } }, { highlightCode: true });
                const relativePath = path_1.default.relative(process.cwd(), filePath);
                report.push((0, nanocolors_1.red)(`Error while transforming ${(0, nanocolors_1.cyan)(relativePath)}: ${message}`));
                report.push(result);
                report.push('');
            }
        }
        this.terminal.logStatic(report);
    }
    writeCoverageReport(testCoverage) {
        (0, writeCoverageReport_js_1.writeCoverageReport)(testCoverage, this.config.coverageConfig);
    }
    switchMenu(menu) {
        if (this.activeMenu === menu) {
            return;
        }
        this.activeMenu = menu;
        if (this.config.watch) {
            this.terminal.clear();
        }
        switch (menu) {
            case exports.MENUS.OVERVIEW:
                this.reportTestResults(true);
                this.reportTestProgress();
                if (this.config.watch) {
                    this.terminal.observeDirectInput();
                }
                break;
            case exports.MENUS.FOCUS_SELECT_FILE:
            case exports.MENUS.DEBUG_SELECT_FILE:
                this.logSelectFilesMenu();
                break;
            case exports.MENUS.MANUAL_DEBUG:
                this.logManualDebugMenu();
                break;
            default:
                break;
        }
    }
    logSelectFilesMenu() {
        this.menuSucceededAndPendingFiles = [];
        this.menuFailedFiles = [];
        for (const testFile of this.runner.testFiles) {
            const sessions = Array.from(this.sessions.forTestFile(testFile));
            if (sessions.every(t => t.status === TestSessionStatus_js_1.SESSION_STATUS.FINISHED && !t.passed)) {
                this.menuFailedFiles.push(testFile);
            }
            else {
                this.menuSucceededAndPendingFiles.push(testFile);
            }
        }
        const selectFilesEntries = (0, getSelectFilesMenu_js_1.getSelectFilesMenu)(this.menuSucceededAndPendingFiles, this.menuFailedFiles);
        this.terminal.logDynamic([]);
        this.terminal.logStatic(selectFilesEntries);
        this.terminal.logPendingUserInput(`Number of the file to ${this.activeMenu === exports.MENUS.FOCUS_SELECT_FILE ? 'focus' : 'debug'}: `);
        this.terminal.observeConfirmedInput();
    }
    logManualDebugMenu() {
        this.terminal.logDynamic((0, getManualDebugMenu_js_1.getManualDebugMenu)(this.config));
    }
    async reportEnd() {
        var _a;
        for (const reporter of this.config.reporters) {
            await ((_a = reporter.stop) === null || _a === void 0 ? void 0 : _a.call(reporter, {
                sessions: Array.from(this.sessions.all()),
                testCoverage: this.testCoverage,
                focusedTestFile: this.runner.focusedTestFile,
            }));
        }
        this.reportTestProgress(true);
        this.terminal.stop();
        this.runner.stop();
    }
}
exports.TestRunnerCli = TestRunnerCli;
//# sourceMappingURL=TestRunnerCli.js.map