"use strict";
/**
 * Copyright 2017 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTSConfig = exports.getPkgManagerCommand = exports.isYarnUsed = exports.nop = exports.readJsonp = exports.ncpp = exports.rimrafp = exports.readFilep = void 0;
const fs = require("fs");
const path = require("path");
const rimraf = require("rimraf");
const util_1 = require("util");
const ncp = require("ncp");
const JSON5 = require("json5");
exports.readFilep = (0, util_1.promisify)(fs.readFile);
exports.rimrafp = (0, util_1.promisify)(rimraf);
exports.ncpp = (0, util_1.promisify)(ncp.ncp);
async function readJsonp(jsonPath) {
    const contents = await (0, exports.readFilep)(jsonPath, { encoding: 'utf8' });
    return JSON5.parse(contents);
}
exports.readJsonp = readJsonp;
function nop() {
    /* empty */
}
exports.nop = nop;
/**
 * Recursively iterate through the dependency chain until we reach the end of
 * the dependency chain or encounter a circular reference
 * @param filePath Filepath of file currently being read
 * @param customReadFilep The file reading function being used
 * @param readFiles an array of the previously read files so we can check for
 * circular references
 * returns a ConfigFile object containing the data from all the dependencies
 */
async function getBase(filePath, customReadFilep, readFiles, currentDir) {
    customReadFilep = customReadFilep || exports.readFilep;
    filePath = path.resolve(currentDir, filePath);
    // An error is thrown if there is a circular reference as specified by the
    // TypeScript doc
    if (readFiles.has(filePath)) {
        throw new Error(`Circular reference in ${filePath}`);
    }
    readFiles.add(filePath);
    try {
        const json = await customReadFilep(filePath, 'utf8');
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        let contents;
        try {
            contents = JSON5.parse(json);
        }
        catch (e) {
            const err = e;
            err.message = `Unable to parse ${filePath}!\n${err.message}`;
            throw err;
        }
        if (contents.extends) {
            const nextFile = await getBase(contents.extends, customReadFilep, readFiles, path.dirname(filePath));
            contents = combineTSConfig(nextFile, contents);
        }
        return contents;
    }
    catch (e) {
        const err = e;
        err.message = `Error: ${filePath}\n${err.message}`;
        throw err;
    }
}
/**
 * Takes in 2 config files
 * @param base is loaded first
 * @param inherited is then loaded and overwrites base
 */
function combineTSConfig(base, inherited) {
    const result = { compilerOptions: {} };
    Object.assign(result, base, inherited);
    Object.assign(result.compilerOptions, base.compilerOptions, inherited.compilerOptions);
    delete result.extends;
    return result;
}
/**
 * Automatically defines npm or yarn is going to be used:
 * - If only yarn.lock exists, use yarn
 * - If only package-lock.json or both exist, use npm
 */
function isYarnUsed(existsSync = fs.existsSync) {
    if (existsSync('package-lock.json')) {
        return false;
    }
    return existsSync('yarn.lock');
}
exports.isYarnUsed = isYarnUsed;
function getPkgManagerCommand(isYarnUsed) {
    return ((isYarnUsed ? 'yarn' : 'npm') + (process.platform === 'win32' ? '.cmd' : ''));
}
exports.getPkgManagerCommand = getPkgManagerCommand;
/**
 * Find the tsconfig.json, read it, and return parsed contents.
 * @param rootDir Directory where the tsconfig.json should be found.
 * If the tsconfig.json file has an "extends" field hop down the dependency tree
 * until it ends or a circular reference is found in which case an error will be
 * thrown
 */
async function getTSConfig(rootDir, customReadFilep) {
    customReadFilep = (customReadFilep || exports.readFilep);
    const readArr = new Set();
    return getBase('tsconfig.json', customReadFilep, readArr, rootDir);
}
exports.getTSConfig = getTSConfig;
//# sourceMappingURL=util.js.map