"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUserAgentCompat = void 0;
const tslib_1 = require("tslib");
/* eslint-disable no-console */
const browserslist_useragent_1 = require("browserslist-useragent");
const browserslist_1 = tslib_1.__importDefault(require("browserslist"));
const caniuse = tslib_1.__importStar(require("caniuse-api"));
const utils_1 = require("./utils");
const is_modern_browser_1 = require("./is-modern-browser");
/**
 * We compile to es modules when the browser supports module scripts, dynamic imports
 * and import.meta.url. Unfortunately, caniuse doesn't report import.meta.url. Chrome 63
 * is the only browser which suppors dynamic imports but not import.meta.url.
 */
const moduleFeatures = ['es6-module', 'es6-module-dynamic-import'];
// Cache resolved user agents because we're going to request them a lot
const cache = new Map();
// Some browsers are resolved to a name unknown to browserslist
const browserAliases = {
    'UC Browser': 'UCAndroid',
};
/**
 * The user agent parser does not always return a valid version for samsung,
 * so we need to normalize it.
 */
function normalizeSamsungVersion(browser, version) {
    try {
        browserslist_1.default(`${browser} ${version}`);
        // browserslist didn't throw, return the valid version
        return version;
    }
    catch (error) {
        // we gave an invalid version to browserslist, so we try to
        // find the nearest matching major version for samsung browser
        const validVersions = [
            ...Object.keys(browserslist_1.default.versionAliases.samsung),
            ...browserslist_1.default.data.samsung.versions,
        ];
        return validVersions.find(validVersion => validVersion[0] === version[0]);
    }
}
/**
 * The user agent parser returns patch versions, which browserslist doesn't
 * know about. Mostly the major version is sufficient, except for safari.
 */
function getBrowserVersion(resolvedUA) {
    const version = resolvedUA.version.split('.');
    switch (resolvedUA.family) {
        case 'Safari':
        case 'iOS':
            return `${version[0]}.${version[1]}`;
        case 'Samsung':
            return normalizeSamsungVersion(resolvedUA.family, version);
        default:
            return version[0];
    }
}
function getBrowserName(browserName) {
    return browserAliases[browserName] || browserName;
}
/**
 * Returns whether this browser supports es modules. We count this when the browser
 * supports module syntax, scripts, dynamic imports. We can't feature detect
 * import.meta.url but any browsers which supports dynamic import supports import.meta.url,
 * except for chrome 63.
 */
function getSupportsEsm(browserTarget) {
    if (browserTarget.toLowerCase() === 'chrome 63') {
        return false;
    }
    return moduleFeatures.every(ft => caniuse.isSupported(ft, browserTarget));
}
/**
 * Calculates the user agent's compatibility.
 */
function calcUserAgentCompat(userAgent) {
    let resolvedUA;
    try {
        resolvedUA = browserslist_useragent_1.resolveUserAgent(userAgent);
        const browserTarget = `${getBrowserName(resolvedUA.family)} ${getBrowserVersion(resolvedUA)}`;
        const modern = is_modern_browser_1.isModernBrowser(userAgent);
        const supportsEsm = modern ? true : getSupportsEsm(browserTarget);
        return { browserTarget, modern, supportsEsm };
    }
    catch (error) {
        let message = '\n';
        if (!resolvedUA) {
            message = `Unable to resolve user agent: ${userAgent}.`;
        }
        else {
            message = `Resolved to invalid user agent: ${JSON.stringify(resolvedUA)}.`;
        }
        console.warn(`${message} Using max compatibility mode. Try updating your dependencies, ` +
            'or file an issue with a reproduction.\n');
        return {
            browserTarget: null,
            modern: false,
            supportsEsm: false,
        };
    }
}
function getUserAgentCompat(ctx) {
    const userAgent = ctx.get('user-agent');
    let compat = cache.get(userAgent);
    if (!compat) {
        compat = calcUserAgentCompat(userAgent);
        cache.set(userAgent, compat);
        utils_1.logDebug(`User agent: ${userAgent} detected as compatibility:`, JSON.stringify(compat));
    }
    return compat;
}
exports.getUserAgentCompat = getUserAgentCompat;
//# sourceMappingURL=user-agent-compat.js.map