"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.startServer = void 0;
const tslib_1 = require("tslib");
const open_1 = tslib_1.__importDefault(require("open"));
const chokidar_1 = tslib_1.__importDefault(require("chokidar"));
const portfinder_1 = tslib_1.__importDefault(require("portfinder"));
const url_1 = require("url");
const create_server_1 = require("./create-server");
const constants_1 = require("./constants");
function isValidURL(str) {
    try {
        return !!new url_1.URL(str);
    }
    catch (error) {
        return false;
    }
}
async function startServer(cfg, fileWatcher = chokidar_1.default.watch([])) {
    const result = create_server_1.createServer(cfg, fileWatcher);
    const { app, onServerStarted } = result;
    let { server } = result;
    const port = typeof cfg.port === 'number' ? cfg.port : await portfinder_1.default.getPortPromise();
    // cleanup after quit
    function closeFileWatcher() {
        if (fileWatcher) {
            fileWatcher.close();
        }
    }
    server.addListener('close', closeFileWatcher);
    async function stopServer() {
        if (server) {
            server.close();
        }
    }
    ['exit', 'SIGINT'].forEach(event => {
        // @ts-ignore
        process.on(event, stopServer);
    });
    process.on('uncaughtException', error => {
        /* eslint-disable-next-line no-console */
        console.error(error);
        stopServer();
    });
    // Deprecated: replaced by plugins
    if (cfg.onServerStart) {
        await cfg.onServerStart(cfg);
    }
    await onServerStarted(app, server);
    // start the server, open the browser and log messages
    await new Promise(resolve => server.listen({ port, host: cfg.hostname }, () => {
        const prettyHost = cfg.hostname || 'localhost';
        if (cfg.logStartup) {
            const msgs = [];
            msgs.push(`es-dev-server started on http${cfg.http2 ? 's' : ''}://${prettyHost}:${port}`);
            msgs.push(`  Serving files from '${cfg.rootDir}'.`);
            if (cfg.sslKey && cfg.sslCert) {
                msgs.push(`  using key '${cfg.sslKey}'`);
                msgs.push(`  and cert '${cfg.sslCert}'`);
            }
            if (cfg.openBrowser) {
                msgs.push(`  Opening browser on '${cfg.openPath}'`);
            }
            if (cfg.appIndex) {
                msgs.push(`  Using history API fallback, redirecting route requests to '${cfg.appIndex}'`);
            }
            if (cfg.compatibilityMode === constants_1.compatibilityModes.AUTO) {
                msgs.push(`  Using auto compatibility mode, transforming code on older browsers based on user agent.`);
            }
            else if (cfg.compatibilityMode === constants_1.compatibilityModes.ALWAYS) {
                msgs.push(`  Using always compatibility mode, transforming code based on user agent.`);
            }
            else if (cfg.compatibilityMode === constants_1.compatibilityModes.MIN) {
                msgs.push(`  Using minimum compatibility mode, always transforming code for compatiblity with modern browsers.`);
            }
            else if (cfg.compatibilityMode === constants_1.compatibilityModes.MAX) {
                msgs.push(`  Using maximum compatibility mode, always transforming code to es5 for compatiblity with older browsers.`);
            }
            const hasBabel = cfg.readUserBabelConfig || cfg.customBabelConfig;
            if (hasBabel) {
                msgs.push(`  Using a custom babel configuration.`);
            }
            if (hasBabel ||
                ![constants_1.compatibilityModes.NONE, constants_1.compatibilityModes.AUTO].includes(cfg.compatibilityMode)) {
                msgs.push(`  \nes-dev-server is configured to always compile code. For the fastest dev experience, use compatibility auto without any custom babel configuration.`);
            }
            /* eslint-disable-next-line no-console */
            console.log(msgs.join('\n'));
        }
        if (cfg.openBrowser) {
            const openPath = (() => {
                if (isValidURL(cfg.openPath)) {
                    return cfg.openPath;
                }
                return new url_1.URL(cfg.openPath, `http${cfg.http2 ? 's' : ''}://${prettyHost}:${port}`).href;
            })();
            open_1.default(openPath);
        }
        resolve();
    }));
    return {
        server,
        app,
    };
}
exports.startServer = startServer;
//# sourceMappingURL=start-server.js.map