"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchSourceMap = void 0;
const convert_source_map_1 = require("convert-source-map");
const path_1 = __importDefault(require("path"));
const request_js_1 = require("./request.js");
function is2xxResponse(status) {
    return typeof status === 'number' && status >= 200 && status < 300;
}
async function doFetchSourceMap(code, browserUrl, reqOpts) {
    var _a;
    try {
        const match = code.match(convert_source_map_1.mapFileCommentRegex);
        if (!match || !match[0]) {
            return;
        }
        const [sourceMapComment] = match;
        const sourceMapUrl = sourceMapComment.replace('//# sourceMappingURL=', '');
        if (sourceMapUrl.startsWith('data')) {
            // this is a data url
            return (_a = (0, convert_source_map_1.fromSource)(code)) !== null && _a !== void 0 ? _a : undefined;
        }
        // this is a source map pointing to another file, fetch it
        const dir = path_1.default.posix.dirname(browserUrl.split('?')[0].split('#')[0]);
        const sourcMapPath = path_1.default.join(dir, sourceMapUrl);
        const { response: sourceMapResponse, body: sourceMap } = await (0, request_js_1.request)(Object.assign(Object.assign({}, reqOpts), { path: encodeURI(sourcMapPath) }));
        if (!is2xxResponse(sourceMapResponse.statusCode) || !sourceMap) {
            return;
        }
        return (0, convert_source_map_1.fromJSON)(sourceMap);
    }
    catch (error) {
        console.error(`Error retrieving source map for ${browserUrl}`);
        console.error(error);
    }
}
/**
 * Fetches the source maps for a file by retreiving the original source code from the server, and
 * reading the source maps if there are any available.
 */
async function fetchSourceMap(args) {
    const headers = args.userAgent ? { 'user-agent': args.userAgent } : {};
    const reqOpts = {
        protocol: args.protocol,
        host: args.host,
        port: String(args.port),
        method: 'GET',
        headers,
        timeout: 5000,
    };
    // fetch the source code used by the browser, using the browser's user agent to
    // account for accurate transformation
    const { response, body: source } = await (0, request_js_1.request)(Object.assign(Object.assign({}, reqOpts), { path: encodeURI(args.browserUrl) }));
    // we couldn't retreive this file, this could be because it is a generated file
    // from a server plugin which no longer exists
    if (!is2xxResponse(response.statusCode) || !source) {
        return {};
    }
    const sourceMap = await doFetchSourceMap(source, args.browserUrl, reqOpts);
    if (!sourceMap) {
        return { source };
    }
    return { source, sourceMap };
}
exports.fetchSourceMap = fetchSourceMap;
//# sourceMappingURL=fetchSourceMap.js.map