import { defaultIntegrations as defaultIntegrations$1, init as init$1 } from '@sentry/node';
import { session } from 'electron';
import { getSdkInfo, getDefaultReleaseName, getDefaultEnvironment } from './context.js';
import { configureIPC } from './ipc.js';
import { defaultStackParser } from './stack-parse.js';
import { makeElectronOfflineTransport } from './transports/electron-offline-net.js';
import { ensureProcess } from '../common/process.js';
import { SentryMinidump } from './integrations/sentry-minidump/index.js';
import { ElectronBreadcrumbs } from './integrations/electron-breadcrumbs.js';
import { Net } from './integrations/net-breadcrumbs.js';
import { MainContext } from './integrations/main-context.js';
import { ChildProcess } from './integrations/child-process.js';
import { OnUncaughtException } from './integrations/onuncaughtexception.js';
import { PreloadInjection } from './integrations/preload-injection.js';
import { AdditionalContext } from './integrations/additional-context.js';
import { Screenshots } from './integrations/screenshots.js';
import { RendererProfiling } from './integrations/renderer-profiling.js';
import { MainProcessSession } from './integrations/main-process-session.js';
import { IPCMode } from '../common/mode.js';

ensureProcess('main');
const defaultIntegrations = [
    new SentryMinidump(),
    new ElectronBreadcrumbs(),
    new Net(),
    new MainContext(),
    new ChildProcess(),
    new OnUncaughtException(),
    new PreloadInjection(),
    new AdditionalContext(),
    new Screenshots(),
    new RendererProfiling(),
    ...defaultIntegrations$1.filter((integration) => integration.name !== 'OnUncaughtException' && integration.name !== 'Context'),
];
const defaultOptions = {
    _metadata: { sdk: getSdkInfo() },
    ipcMode: IPCMode.Both,
    getSessions: () => [session.defaultSession],
};
/**
 * Initialize Sentry in the Electron main process
 */
function init(userOptions) {
    const options = Object.assign(defaultOptions, userOptions);
    const defaults = defaultIntegrations;
    // If we don't set a release, @sentry/node will automatically fetch from environment variables
    if (options.release === undefined) {
        options.release = getDefaultReleaseName();
    }
    // If we don't set an environment, @sentry/core defaults to production
    if (options.environment === undefined) {
        options.environment = getDefaultEnvironment();
    }
    // Unless autoSessionTracking is specifically disabled, we track sessions as the
    // lifetime of the Electron main process
    if (options.autoSessionTracking !== false) {
        defaults.push(new MainProcessSession());
        // We don't want nodejs autoSessionTracking
        options.autoSessionTracking = false;
    }
    if (options.stackParser === undefined) {
        options.stackParser = defaultStackParser;
    }
    setDefaultIntegrations(defaults, options);
    if (options.dsn && options.transport === undefined) {
        options.transport = makeElectronOfflineTransport;
    }
    configureIPC(options);
    init$1(options);
}
/** A list of integrations which cause default integrations to be removed */
const INTEGRATION_OVERRIDES = [
    { override: 'ElectronMinidump', remove: 'SentryMinidump' },
    { override: 'BrowserWindowSession', remove: 'MainProcessSession' },
];
/** Sets the default integrations and ensures that multiple minidump or session integrations are not enabled */
function setDefaultIntegrations(defaults, options) {
    if (options.defaultIntegrations === undefined) {
        const removeDefaultsMatching = (user, defaults) => {
            const toRemove = INTEGRATION_OVERRIDES.filter(({ override }) => user.some((i) => i.name === override)).map(({ remove }) => remove);
            return defaults.filter((i) => !toRemove.includes(i.name));
        };
        if (Array.isArray(options.integrations)) {
            options.defaultIntegrations = removeDefaultsMatching(options.integrations, defaults);
            return;
        }
        else if (typeof options.integrations === 'function') {
            const originalFn = options.integrations;
            options.integrations = (integrations) => {
                const resultIntegrations = originalFn(integrations);
                return removeDefaultsMatching(resultIntegrations, resultIntegrations);
            };
        }
        options.defaultIntegrations = defaults;
    }
}

export { defaultIntegrations, init };
//# sourceMappingURL=sdk.js.map
