import { __awaiter } from 'tslib';
import { convertIntegrationFnToClass, Scope, getCurrentScope, applyScopeDataToEvent, captureEvent } from '@sentry/core';
import { SentryError, logger } from '@sentry/utils';
import { crashReporter, app } from 'electron';
import { getDefaultReleaseName, getDefaultEnvironment, getEventDefaults } from '../../context.js';
import { onRendererProcessGone, onChildProcessGone, EXIT_REASONS } from '../../electron-normalize.js';
import { getSentryCachePath } from '../../fs.js';
import { trackRendererProperties, getRendererProperties } from '../../renderers.js';
import { checkPreviousSession, sessionCrashed } from '../../sessions.js';
import { BufferedWriteStore } from '../../store.js';
import { getMinidumpLoader } from './minidump-loader.js';
import { mergeEvents } from '../../../common/merge.js';

const INTEGRATION_NAME = 'SentryMinidump';
const sentryMinidump = () => {
    /** Store to persist context information beyond application crashes. */
    let scopeStore;
    // We need to store the scope in a variable here so it can be attached to minidumps
    let scopeLastRun;
    let minidumpLoader;
    function startCrashReporter() {
        logger.log('Starting Electron crashReporter');
        crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: app.name || app.getName(),
            // Empty string doesn't work for Linux Crashpad and no submitURL doesn't work for older versions of Electron
            submitURL: 'https://f.a.k/e',
            uploadToServer: false,
            compress: true,
        });
    }
    function setupScopeListener(currentRelease, currentEnvironment) {
        const scopeChanged = (updatedScope) => {
            // Since the initial scope read is async, we need to ensure that any writes do not beat that
            // https://github.com/getsentry/sentry-electron/issues/585
            setImmediate(() => __awaiter(this, void 0, void 0, function* () {
                return scopeStore === null || scopeStore === void 0 ? void 0 : scopeStore.set({
                    scope: updatedScope.getScopeData(),
                    event: yield getEventDefaults(currentRelease, currentEnvironment),
                });
            }));
        };
        const scope = getCurrentScope();
        if (scope) {
            scope.addScopeListener(scopeChanged);
            // Ensure at least one event is written to disk
            scopeChanged(scope);
        }
    }
    function sendNativeCrashes(client, eventIn) {
        var _a, _b, _c, _d;
        return __awaiter(this, void 0, void 0, function* () {
            // Whenever we are called, assume that the crashes we are going to load down
            // below have occurred recently. This means, we can use the same event data
            // for all minidumps that we load now. There are two conditions:
            //
            //  1. The application crashed and we are just starting up. The stored
            //     breadcrumbs and context reflect the state during the application
            //     crash.
            //
            //  2. A renderer process crashed recently and we have just been notified
            //     about it. Just use the breadcrumbs and context information we have
            //     right now and hope that the delay was not too long.
            const event = eventIn;
            // If this is a native main process crash, we need to apply the scope and context from the previous run
            if (((_a = event.tags) === null || _a === void 0 ? void 0 : _a['event.process']) === 'browser') {
                const previousRun = yield scopeLastRun;
                if (previousRun) {
                    if (previousRun.scope) {
                        applyScopeDataToEvent(event, previousRun.scope);
                    }
                    event.release = ((_b = previousRun.event) === null || _b === void 0 ? void 0 : _b.release) || event.release;
                    event.environment = ((_c = previousRun.event) === null || _c === void 0 ? void 0 : _c.environment) || event.environment;
                    event.contexts = ((_d = previousRun.event) === null || _d === void 0 ? void 0 : _d.contexts) || event.contexts;
                }
            }
            if (!event) {
                return false;
            }
            // If the SDK is not enabled, tell the loader to delete all minidumps
            const deleteAll = client.getOptions().enabled === false;
            let minidumpSent = false;
            yield (minidumpLoader === null || minidumpLoader === void 0 ? void 0 : minidumpLoader(deleteAll, (attachment) => {
                captureEvent(event, { attachments: [attachment] });
                minidumpSent = true;
            }));
            // Unset to recover memory
            return minidumpSent;
        });
    }
    function sendRendererCrash(client, options, contents, details) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const { getRendererName, release, environment } = options;
            const crashedProcess = (getRendererName === null || getRendererName === void 0 ? void 0 : getRendererName(contents)) || 'renderer';
            logger.log(`'${crashedProcess}' process '${details.reason}'`);
            const event = mergeEvents(yield getEventDefaults(release, environment), {
                contexts: {
                    electron: {
                        crashed_url: ((_a = getRendererProperties(contents.id)) === null || _a === void 0 ? void 0 : _a.url) || 'unknown',
                        details,
                    },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': crashedProcess,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield sendNativeCrashes(client, event);
            if (found) {
                sessionCrashed();
            }
        });
    }
    function sendChildProcessCrash(client, options, details) {
        return __awaiter(this, void 0, void 0, function* () {
            logger.log(`${details.type} process has ${details.reason}`);
            const { release, environment } = options;
            const event = mergeEvents(yield getEventDefaults(release, environment), {
                contexts: {
                    electron: { details },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': details.type,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield sendNativeCrashes(client, event);
            if (found) {
                sessionCrashed();
            }
        });
    }
    return {
        name: INTEGRATION_NAME,
        setup(client) {
            // Mac AppStore builds cannot run the crash reporter due to the sandboxing
            // requirements. In this case, we prevent enabling native crashes entirely.
            // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
            if (process.mas) {
                return;
            }
            startCrashReporter();
            scopeStore = new BufferedWriteStore(getSentryCachePath(), 'scope_v3', {
                scope: new Scope().getScopeData(),
            });
            scopeLastRun = scopeStore.get();
            minidumpLoader = getMinidumpLoader();
            const options = client.getOptions();
            const currentRelease = (options === null || options === void 0 ? void 0 : options.release) || getDefaultReleaseName();
            const currentEnvironment = (options === null || options === void 0 ? void 0 : options.environment) || getDefaultEnvironment();
            setupScopeListener(currentRelease, currentEnvironment);
            if (!(options === null || options === void 0 ? void 0 : options.dsn)) {
                throw new SentryError('Attempted to enable Electron native crash reporter but no DSN was supplied');
            }
            trackRendererProperties();
            onRendererProcessGone(EXIT_REASONS, (contents, details) => sendRendererCrash(client, options, contents, details));
            onChildProcessGone(EXIT_REASONS, (details) => sendChildProcessCrash(client, options, details));
            // Start to submit recent minidump crashes. This will load breadcrumbs and
            // context information that was cached on disk prior to the crash.
            sendNativeCrashes(client, {
                level: 'fatal',
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': 'browser',
                    event_type: 'native',
                },
            })
                .then((minidumpsFound) => 
            // Check for previous uncompleted session. If a previous session exists
            // and no minidumps were found, its likely an abnormal exit
            checkPreviousSession(minidumpsFound))
                .catch((error) => logger.error(error));
        },
    };
};
/** Sends minidumps via the Sentry uploader */
// eslint-disable-next-line deprecation/deprecation
const SentryMinidump = convertIntegrationFnToClass(INTEGRATION_NAME, sentryMinidump);

export { SentryMinidump };
//# sourceMappingURL=index.js.map
