import { convertIntegrationFnToClass } from '@sentry/core';
import { app, screen } from 'electron';
import { cpus } from 'os';
import { whenAppReady } from '../electron-normalize.js';
import { mergeEvents } from '../../common/merge.js';

const DEFAULT_OPTIONS = {
    cpu: true,
    screen: true,
    memory: true,
    language: true,
};
const INTEGRATION_NAME = 'AdditionalContext';
const additionalContext = (userOptions = {}) => {
    const _lazyDeviceContext = {};
    const options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), userOptions);
    function _setPrimaryDisplayInfo() {
        const display = screen.getPrimaryDisplay();
        const width = Math.floor(display.size.width * display.scaleFactor);
        const height = Math.floor(display.size.height * display.scaleFactor);
        _lazyDeviceContext.screen_density = display.scaleFactor;
        _lazyDeviceContext.screen_resolution = `${width}x${height}`;
    }
    return {
        name: INTEGRATION_NAME,
        setup() {
            // Some metrics are only available after app ready so we lazily load them
            whenAppReady.then(() => {
                const { language, screen: screen$1 } = options;
                if (language) {
                    _lazyDeviceContext.language = app.getLocale();
                }
                if (screen$1) {
                    _setPrimaryDisplayInfo();
                    screen.on('display-metrics-changed', () => {
                        _setPrimaryDisplayInfo();
                    });
                }
            }, () => {
                // ignore
            });
        },
        processEvent(event) {
            const device = _lazyDeviceContext;
            const { memory, cpu } = options;
            if (memory) {
                const { total, free } = process.getSystemMemoryInfo();
                device.memory_size = total * 1024;
                device.free_memory = free * 1024;
            }
            if (cpu) {
                const cpuInfo = cpus();
                if (cpuInfo && cpuInfo.length) {
                    const firstCpu = cpuInfo[0];
                    device.processor_count = cpuInfo.length;
                    device.cpu_description = firstCpu.model;
                    device.processor_frequency = firstCpu.speed;
                    if (app.runningUnderARM64Translation) {
                        device.machine_arch = 'arm64';
                    }
                }
            }
            return mergeEvents(event, { contexts: { device } });
        },
    };
};
/** Adds Electron context to events and normalises paths. */
// eslint-disable-next-line deprecation/deprecation
const AdditionalContext = convertIntegrationFnToClass(INTEGRATION_NAME, additionalContext);

export { AdditionalContext };
//# sourceMappingURL=additional-context.js.map
