"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createServer = void 0;
const tslib_1 = require("tslib");
const koa_1 = tslib_1.__importDefault(require("koa"));
const path_1 = tslib_1.__importDefault(require("path"));
const http_1 = tslib_1.__importDefault(require("http"));
const http2_1 = tslib_1.__importDefault(require("http2"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const chokidar_1 = tslib_1.__importDefault(require("chokidar"));
const net_1 = tslib_1.__importDefault(require("net"));
const create_middlewares_1 = require("./create-middlewares");
/**
 * A request handler that returns a 301 HTTP Redirect to the same location as the original
 * request but using the https protocol
 */
function httpsRedirect(req, res) {
    const { host } = req.headers;
    res.writeHead(301, { Location: `https://${host}${req.url}` });
    res.end();
}
/**
 * Creates a koa server with middlewares, but does not start it. Returns the koa app and
 * http server instances.
 */
function createServer(cfg, fileWatcher = chokidar_1.default.watch([])) {
    const { middlewares, onServerStarted } = create_middlewares_1.createMiddlewares(cfg, fileWatcher);
    const app = new koa_1.default();
    middlewares.forEach(middleware => {
        app.use(middleware);
    });
    let server;
    if (cfg.http2) {
        const dir = path_1.default.join(__dirname, '..');
        const options = {
            key: fs_1.default.readFileSync(cfg.sslKey ? cfg.sslKey : path_1.default.join(dir, '.self-signed-dev-server-ssl.key')),
            cert: fs_1.default.readFileSync(cfg.sslCert ? cfg.sslCert : path_1.default.join(dir, '.self-signed-dev-server-ssl.cert')),
            allowHTTP1: true,
        };
        const httpsRedirectServer = http_1.default.createServer(httpsRedirect);
        const appServer = http2_1.default.createSecureServer(options, app.callback());
        let appServerPort;
        let httpsRedirectServerPort;
        /**
         * A connection handler that checks if the connection is using TLS
         */
        const httpRedirectProxy = (socket) => {
            socket.once('data', buffer => {
                // A TLS handshake record starts with byte 22.
                const address = buffer[0] === 22 ? appServerPort : httpsRedirectServerPort;
                //@ts-ignore
                const proxy = net_1.default.createConnection(address, () => {
                    proxy.write(buffer);
                    socket.pipe(proxy).pipe(socket);
                });
            });
        };
        server = net_1.default.createServer(httpRedirectProxy);
        server.addListener('close', () => {
            httpsRedirectServer.close();
            appServer.close();
        });
        server.addListener('listening', () => {
            const info = server.address();
            if (!info || typeof info === 'string') {
                return;
            }
            const { address, port } = info;
            appServerPort = port + 1;
            httpsRedirectServerPort = port + 2;
            appServer.listen({ address, port: appServerPort });
            httpsRedirectServer.listen({ address, port: httpsRedirectServerPort });
        });
        const serverListen = server.listen.bind(server);
        //@ts-ignore
        server.listen = (config, callback) => {
            appServer.addListener('listening', callback);
            serverListen(config);
            return server;
        };
    }
    else {
        server = http_1.default.createServer(app.callback());
    }
    return {
        server,
        app,
        onServerStarted,
    };
}
exports.createServer = createServer;
//# sourceMappingURL=create-server.js.map