const core = require('@sentry/core');
const electronNormalize = require('../electron-normalize.js');

const DEFAULT_OPTIONS = {
    breadcrumbs: electronNormalize.EXIT_REASONS,
    events: ['abnormal-exit', 'launch-failed', 'integrity-failure'],
};
/** Gets message and severity */
function getMessageAndSeverity(reason, proc) {
    const message = `'${proc}' process exited with '${reason}'`;
    switch (reason) {
        case 'abnormal-exit':
        case 'killed':
            return { message, level: 'warning' };
        case 'crashed':
        case 'oom':
        case 'launch-failed':
        case 'integrity-failure':
            return { message, level: 'fatal' };
        default:
            return { message, level: 'debug' };
    }
}
const INTEGRATION_NAME = 'ChildProcess';
const childProcess = (userOptions = {}) => {
    const { breadcrumbs, events } = userOptions;
    const options = {
        breadcrumbs: Array.isArray(breadcrumbs) ? breadcrumbs : breadcrumbs == false ? [] : DEFAULT_OPTIONS.breadcrumbs,
        events: Array.isArray(events) ? events : events == false ? [] : DEFAULT_OPTIONS.events,
    };
    return {
        name: INTEGRATION_NAME,
        setup(client) {
            const { breadcrumbs, events } = options;
            const allReasons = Array.from(new Set([...breadcrumbs, ...events]));
            // only hook these events if we're after more than just the unresponsive event
            if (allReasons.length > 0) {
                const clientOptions = client.getOptions();
                electronNormalize.onChildProcessGone(allReasons, (details) => {
                    const { reason } = details;
                    // Capture message first
                    if (events.includes(reason)) {
                        const { message, level } = getMessageAndSeverity(details.reason, details.type);
                        core.captureMessage(message, { level, tags: { 'event.process': details.type } });
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        core.addBreadcrumb(Object.assign(Object.assign({ type: 'process', category: 'child-process' }, getMessageAndSeverity(details.reason, details.type)), { data: details }));
                    }
                });
                electronNormalize.onRendererProcessGone(allReasons, (contents, details) => {
                    var _a;
                    const { reason } = details;
                    const name = ((_a = clientOptions === null || clientOptions === void 0 ? void 0 : clientOptions.getRendererName) === null || _a === void 0 ? void 0 : _a.call(clientOptions, contents)) || 'renderer';
                    // Capture message first
                    if (events.includes(reason)) {
                        const { message, level } = getMessageAndSeverity(details.reason, name);
                        core.captureMessage(message, level);
                    }
                    // And then add breadcrumbs for subsequent events
                    if (breadcrumbs.includes(reason)) {
                        core.addBreadcrumb(Object.assign(Object.assign({ type: 'process', category: 'child-process' }, getMessageAndSeverity(details.reason, name)), { data: details }));
                    }
                });
            }
        },
    };
};
/** Adds breadcrumbs for Electron events. */
// eslint-disable-next-line deprecation/deprecation
const ChildProcess = core.convertIntegrationFnToClass(INTEGRATION_NAME, childProcess);

exports.ChildProcess = ChildProcess;
//# sourceMappingURL=child-process.js.map
