import { convertIntegrationFnToClass, addBreadcrumb } from '@sentry/core';
import { screen, powerMonitor, app, autoUpdater } from 'electron';
import { whenAppReady, onBrowserWindowCreated, onWebContentsCreated } from '../electron-normalize.js';
import { trackRendererProperties, getRendererProperties } from '../renderers.js';

const DEFAULT_OPTIONS = {
    // We exclude events starting with remote as they can be quite verbose
    app: (name) => !name.startsWith('remote-'),
    autoUpdater: () => true,
    webContents: (name) => ['dom-ready', 'context-menu', 'load-url', 'destroyed'].includes(name),
    browserWindow: (name) => [
        'closed',
        'close',
        'unresponsive',
        'responsive',
        'show',
        'blur',
        'focus',
        'hide',
        'maximize',
        'minimize',
        'restore',
        'enter-full-screen',
        'leave-full-screen',
    ].includes(name),
    screen: () => true,
    powerMonitor: () => true,
    captureWindowTitles: false,
};
/** Converts all user supplied options to function | false */
function normalizeOptions(options) {
    return Object.keys(options).reduce((obj, k) => {
        if (k === 'captureWindowTitles') {
            obj[k] = !!options[k];
        }
        else {
            const val = options[k];
            if (Array.isArray(val)) {
                obj[k] = (name) => val.includes(name);
            }
            else if (typeof val === 'function' || val === false) {
                obj[k] = val;
            }
        }
        return obj;
    }, {});
}
const INTEGRATION_NAME = 'ElectronBreadcrumbs';
const electronBreadcrumbs = (userOptions = {}) => {
    const options = Object.assign(Object.assign({}, DEFAULT_OPTIONS), normalizeOptions(userOptions));
    function patchEventEmitter(emitter, category, shouldCapture, id) {
        const emit = emitter.emit.bind(emitter);
        emitter.emit = (event, ...args) => {
            var _a, _b;
            if (shouldCapture && shouldCapture(event)) {
                const breadcrumb = {
                    category: 'electron',
                    message: `${category}.${event}`,
                    timestamp: new Date().getTime() / 1000,
                    type: 'ui',
                };
                if (id) {
                    breadcrumb.data = Object.assign({}, getRendererProperties(id));
                    if (!options.captureWindowTitles && ((_a = breadcrumb.data) === null || _a === void 0 ? void 0 : _a.title)) {
                        (_b = breadcrumb.data) === null || _b === void 0 ? true : delete _b.title;
                    }
                }
                addBreadcrumb(breadcrumb);
            }
            return emit(event, ...args);
        };
    }
    return {
        name: INTEGRATION_NAME,
        setup(client) {
            const clientOptions = client.getOptions();
            trackRendererProperties();
            whenAppReady.then(() => {
                // We can't access these until app 'ready'
                if (options.screen) {
                    patchEventEmitter(screen, 'screen', options.screen);
                }
                if (options.powerMonitor) {
                    patchEventEmitter(powerMonitor, 'powerMonitor', options.powerMonitor);
                }
            }, () => {
                // ignore
            });
            if (options.app) {
                patchEventEmitter(app, 'app', options.app);
            }
            if (options.autoUpdater) {
                patchEventEmitter(autoUpdater, 'autoUpdater', options.autoUpdater);
            }
            if (options.browserWindow) {
                onBrowserWindowCreated((window) => {
                    var _a;
                    const id = window.webContents.id;
                    const windowName = ((_a = clientOptions === null || clientOptions === void 0 ? void 0 : clientOptions.getRendererName) === null || _a === void 0 ? void 0 : _a.call(clientOptions, window.webContents)) || 'window';
                    patchEventEmitter(window, windowName, options.browserWindow, id);
                });
            }
            if (options.webContents) {
                onWebContentsCreated((contents) => {
                    var _a;
                    const id = contents.id;
                    const webContentsName = ((_a = clientOptions === null || clientOptions === void 0 ? void 0 : clientOptions.getRendererName) === null || _a === void 0 ? void 0 : _a.call(clientOptions, contents)) || 'renderer';
                    patchEventEmitter(contents, webContentsName, options.webContents, id);
                });
            }
        },
    };
};
/** Adds breadcrumbs for Electron events. */
// eslint-disable-next-line deprecation/deprecation
const ElectronBreadcrumbs = convertIntegrationFnToClass(INTEGRATION_NAME, electronBreadcrumbs);

export { ElectronBreadcrumbs, normalizeOptions };
//# sourceMappingURL=electron-breadcrumbs.js.map
