function getCompatibility() {
  if (process.argv.find(arg => arg.includes('--legacy'))) {
    /* eslint-disable-next-line no-console */
    console.warn(`testing-karma --legacy flag has been renamed to --compatibility`);
    return 'all';
  }

  const indexOf = process.argv.indexOf('--compatibility');
  return indexOf === -1 ? 'auto' : process.argv[indexOf + 1];
}

const compatibility = getCompatibility();
const coverage = !!process.argv.find(arg => arg.includes('--coverage'));
const updateSnapshots = !!process.argv.find(arg => arg.includes('--update-snapshots'));
const pruneSnapshots = !!process.argv.find(arg => arg.includes('--prune-snapshots'));

/**
 * Creates base karma configuration.
 * @param {import("karma").Config} config
 * @return {import('karma').ConfigOptions & { coverageReporter: any }}
 */
module.exports = config => ({
  browsers: ['ChromeHeadlessNoSandbox'],

  files: [
    {
      pattern: '__snapshots__/**/*.md',
      // snapshot preprocessor will rewrite content of .md files with JS wrappers
      // but karma will try and determine file type based on extension if we do not
      // specify it, so force snapshot files to be js type to avoid karma complaints
      type: 'js',
    },
    require.resolve('axe-core/axe.min.js'),
  ],

  customLaunchers: {
    ChromeHeadlessNoSandbox: {
      base: 'ChromeHeadless',
      flags: ['--no-sandbox', '--disable-setuid-sandbox'],
    },
  },

  plugins: [
    // resolve plugins relative to this config so that they don't always need to exist
    // at the top level
    require.resolve('@open-wc/karma-esm'),
    require.resolve('karma-mocha'),
    require.resolve('karma-mocha-reporter'),
    require.resolve('karma-source-map-support'),
    require.resolve('karma-coverage'),
    require.resolve('karma-snapshot'),
    require.resolve('karma-mocha-snapshot'),
    require.resolve('karma-chrome-launcher'),

    // fallback: resolve any karma- plugins
    'karma-*',
  ],

  frameworks: ['esm', 'mocha', 'snapshot', 'mocha-snapshot', 'source-map-support'],

  esm: {
    coverage,
    compatibility,
    // prevent compiling es5 libs
    babelExclude: [
      '**/node_modules/mocha/**/*',
      '**/node_modules/chai/**/*',
      '**/node_modules/sinon-chai/**/*',
      '**/node_modules/chai-dom/**/*',
      '**/node_modules/core-js-bundle/**/*',
    ],
    // sinon is not completely es5...
    babelModernExclude: ['**/node_modules/sinon/**/*'],
    // prevent compiling non-module libs
    babelModuleExclude: ['**/node_modules/mocha/**/*', '**/node_modules/core-js-bundle/**/*'],
    exclude: ['**/__snapshots__/**/*'],
  },

  preprocessors: {
    '**/__snapshots__/**/*.md': ['snapshot'],
  },

  reporters: coverage ? ['mocha', 'coverage'] : ['mocha'],

  mochaReporter: {
    showDiff: true,
  },

  restartOnFileChange: false,

  client: {
    mocha: {
      reporter: 'html',
    },
  },

  colors: true,

  // possible values: LOG_DISABLE || LOG_ERROR || LOG_WARN || LOG_INFO || LOG_DEBUG
  logLevel: config.LOG_INFO,

  // ## code coverage config
  coverageReporter: {
    reporters: [{ type: 'html' }, { type: 'lcovonly' }, { type: 'text-summary' }],
    dir: 'coverage',
    thresholds: {
      global: {
        statements: 80,
        branches: 80,
        functions: 80,
        lines: 80,
      },
    },
  },

  snapshot: {
    update: updateSnapshots,
    prune: pruneSnapshots,
    // only warn about unused snapshots when running all tests
    limitUnusedSnapshotsInWarning: config.grep ? 0 : -1,
    pathResolver(basePath, suiteName) {
      return `${basePath}/__snapshots__/${suiteName}.md`;
    },
  },

  autoWatch: false,
  singleRun: true,
  concurrency: Infinity,
});
