/*
 * Copyright (c) 2010-2024 Contributors to the openHAB project
 *
 * See the NOTICE file(s) distributed with this work for additional
 * information.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 */

package org.openhab.habdroid.ui

import android.location.Location
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.view.isGone
import com.google.android.gms.maps.CameraUpdateFactory
import com.google.android.gms.maps.GoogleMap
import com.google.android.gms.maps.MapsInitializer
import com.google.android.gms.maps.model.LatLng
import com.google.android.gms.maps.model.LatLngBounds
import com.google.android.gms.maps.model.Marker
import com.google.android.gms.maps.model.MarkerOptions
import java.util.Locale
import org.openhab.habdroid.databinding.BottomSheetMapBinding
import org.openhab.habdroid.model.Item
import org.openhab.habdroid.model.Widget

object MapViewHelper {
    fun createViewHolder(initData: WidgetAdapter.ViewHolderInitData): WidgetAdapter.ViewHolder {
        MapsInitializer.initialize(initData.inflater.context)
        return GoogleMapsViewHolder(initData)
    }

    private class GoogleMapsViewHolder(initData: WidgetAdapter.ViewHolderInitData) :
        WidgetAdapter.AbstractMapViewHolder(initData) {
        private var map: GoogleMap? = null

        init {
            binding.mapview.onCreate(null)
        }

        override fun bindAfterDataSaverCheck(widget: Widget) {
            super.bindAfterDataSaverCheck(widget)
            withLoadedMap { map ->
                map.clear()
                map.applyPositionAndLabel(widget, 15.0f, false)
            }
        }

        override fun onStart() {
            super.onStart()
            binding.mapview.apply {
                onStart()
                onResume()
            }
        }

        override fun onStop() {
            super.onStop()
            binding.mapview.apply {
                onPause()
                onStop()
            }
        }

        override fun openPopup() {
            val widget = boundWidget ?: return
            fragmentPresenter.showBottomSheet(MapBottomSheet(), widget)
        }

        private fun withLoadedMap(callback: (map: GoogleMap) -> Unit) {
            val loadedMap = this.map
            if (loadedMap != null) {
                callback(loadedMap)
                return
            }

            binding.mapview.getMapAsync { map ->
                this.map = map
                with(map.uiSettings) {
                    setAllGesturesEnabled(false)
                    isMapToolbarEnabled = false
                }
                map.setOnMarkerClickListener {
                    openPopup()
                    true
                }
                map.setOnMapClickListener { openPopup() }
                callback(map)
            }
        }
    }
}

fun GoogleMap.applyPositionAndLabel(widget: Widget, zoomLevel: Float, allowDrag: Boolean) {
    val item = widget.item ?: return
    val canDragMarker = allowDrag && !item.readOnly
    if (item.members.isNotEmpty()) {
        val positionMap = item.members
            .mapNotNull { m -> m.state?.asLocation?.toLatLng()?.let { m to it } }
            .toMap()
        if (positionMap.isNotEmpty()) {
            val boundsBuilder = LatLngBounds.Builder()
            positionMap.forEach { (member, pos) ->
                setMarker(pos, member, member.label, canDragMarker)
                boundsBuilder.include(pos)
            }
            moveCamera(CameraUpdateFactory.newLatLngBounds(boundsBuilder.build(), 0))
            if (cameraPosition.zoom > zoomLevel) {
                moveCamera(CameraUpdateFactory.zoomTo(zoomLevel))
            }
        }
    } else {
        item.state?.asLocation?.toLatLng()?.let { pos ->
            setMarker(pos, item, widget.label, canDragMarker)
            moveCamera(CameraUpdateFactory.newLatLngZoom(pos, zoomLevel))
        }
    }
}

fun GoogleMap.setMarker(position: LatLng, item: Item, label: CharSequence?, canDrag: Boolean) {
    val marker = MarkerOptions()
        .draggable(canDrag)
        .position(position)
        .title(label?.toString())
    addMarker(marker)?.tag = item
}

fun Location.toLatLng() = LatLng(latitude, longitude)

fun Location.toMapsUrl() = "https://www.google.de/maps/@$latitude,$longitude,16z"

class MapBottomSheet :
    AbstractWidgetBottomSheet(),
    GoogleMap.OnMarkerDragListener {
    private lateinit var binding: BottomSheetMapBinding

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        binding = BottomSheetMapBinding.inflate(inflater, container, false)

        binding.title.apply {
            text = widget.label
            isGone = widget.label.isEmpty()
        }

        binding.mapview.onCreate(null)

        return binding.root
    }

    override fun onDestroyView() {
        binding.mapview.onDestroy()
        super.onDestroyView()
    }

    override fun onStart() {
        super.onStart()
        binding.mapview.onStart()
    }

    override fun onStop() {
        super.onStop()
        binding.mapview.onStop()
    }

    override fun onResume() {
        super.onResume()
        binding.mapview.onResume()

        binding.mapview.getMapAsync { map ->
            map.setOnMarkerDragListener(this@MapBottomSheet)
            map.applyPositionAndLabel(widget, 16.0f, true)
        }
    }

    override fun onPause() {
        binding.mapview.onPause()
        super.onPause()
    }

    override fun onMarkerDragStart(marker: Marker) {
        // no-op, we're interested in drag end only
    }

    override fun onMarkerDrag(marker: Marker) {
        // no-op, we're interested in drag end only
    }

    override fun onMarkerDragEnd(marker: Marker) {
        val newState = String.format(Locale.US, "%f,%f", marker.position.latitude, marker.position.longitude)
        connection?.httpClient?.sendItemCommand(marker.tag as Item?, newState)
    }
}
