package org.ojrandom.paiesque.ui;

import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.core.content.ContextCompat;
import androidx.recyclerview.widget.RecyclerView;

import org.ojrandom.paiesque.R;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.TextStyle;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public class ZoneAnalysisAdapter extends RecyclerView.Adapter<ZoneAnalysisAdapter.ViewHolder> {
    private final List<ZoneDayData> zoneData = new ArrayList<>();
    private final DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("MMM dd, yyyy");
    private final DateTimeFormatter shortDateFormatter = DateTimeFormatter.ofPattern("MMM dd");

    public static class ViewHolder extends RecyclerView.ViewHolder {
        TextView dateText, totalPaiText;
        TextView zone50_59, zone60_69, zone70_79, zone80_89, zone90Plus;
        TextView time50_59, time60_69, time70_79, time80_89, time90Plus;

        public ViewHolder(View itemView) {
            super(itemView);
            dateText = itemView.findViewById(R.id.text_date);
            totalPaiText = itemView.findViewById(R.id.text_total_pai);
            zone50_59 = itemView.findViewById(R.id.text_zone_50_59);
            zone60_69 = itemView.findViewById(R.id.text_zone_60_69);
            zone70_79 = itemView.findViewById(R.id.text_zone_70_79);
            zone80_89 = itemView.findViewById(R.id.text_zone_80_89);
            zone90Plus = itemView.findViewById(R.id.text_zone_90_plus);
            time50_59 = itemView.findViewById(R.id.text_time_50_59);
            time60_69 = itemView.findViewById(R.id.text_time_60_69);
            time70_79 = itemView.findViewById(R.id.text_time_70_79);
            time80_89 = itemView.findViewById(R.id.text_time_80_89);
            time90Plus = itemView.findViewById(R.id.text_time_90_plus);
        }
    }

    @Override
    public ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(parent.getContext())
                .inflate(R.layout.item_zone_day, parent, false);
        return new ViewHolder(view);
    }

    @Override
    public void onBindViewHolder(ViewHolder holder, int position) {
        ZoneDayData data = zoneData.get(position);

        String dayOfWeek = data.getDate().getDayOfWeek().getDisplayName(TextStyle.SHORT, Locale.getDefault());
        String dateText;
        if (data.getDate().getYear() == LocalDate.now().getYear()) {
            dateText = dayOfWeek + ", " + data.getDate().format(shortDateFormatter);
        } else {
            dateText = dayOfWeek + ", " + data.getDate().format(dateFormatter);
        }
        holder.dateText.setText(dateText);

        holder.totalPaiText.setText(String.format("%.0f", data.getTotalPai()));

        setZoneText(holder.zone50_59, data.getPaiZone50_59(), R.color.zone_blue);
        setZoneText(holder.zone60_69, data.getPaiZone60_69(), R.color.zone_green);
        setZoneText(holder.zone70_79, data.getPaiZone70_79(), R.color.zone_yellow);
        setZoneText(holder.zone80_89, data.getPaiZone80_89(), R.color.zone_orange);
        setZoneText(holder.zone90Plus, data.getPaiZone90Plus(), R.color.zone_red);

        setTimeText(holder.time50_59, data.getTimeZone50_59(), R.color.zone_blue);
        setTimeText(holder.time60_69, data.getTimeZone60_69(), R.color.zone_green);
        setTimeText(holder.time70_79, data.getTimeZone70_79(), R.color.zone_yellow);
        setTimeText(holder.time80_89, data.getTimeZone80_89(), R.color.zone_orange);
        setTimeText(holder.time90Plus, data.getTimeZone90Plus(), R.color.zone_red);

        if (data.getDate().equals(LocalDate.now())) {
            int highlightColor = ContextCompat.getColor(holder.itemView.getContext(), R.color.zone_blue);
            holder.itemView.setBackgroundColor(highlightColor & 0x1AFFFFFF);
        } else {
            holder.itemView.setBackgroundColor(0x00000000);
        }
    }

    @Override
    public int getItemCount() {
        return zoneData.size();
    }

    public void setData(List<ZoneDayData> newData) {
        this.zoneData.clear();
        if (newData != null) {
            this.zoneData.addAll(newData);
        }
        notifyDataSetChanged();
    }

    public void addData(ZoneDayData data) {
        this.zoneData.add(data);
        notifyItemInserted(zoneData.size() - 1);
    }

    public void clearData() {
        this.zoneData.clear();
        notifyDataSetChanged();
    }

    public List<ZoneDayData> getData() {
        return new ArrayList<>(zoneData);
    }

    private void setZoneText(TextView textView, double value, int colorResId) {
        if (value > 0) {
            textView.setText(String.format("%.1f", value));
            textView.setTextColor(ContextCompat.getColor(textView.getContext(), colorResId));
            textView.setAlpha(1.0f);
        } else {
            textView.setText("0.0");
            textView.setTextColor(ContextCompat.getColor(textView.getContext(), R.color.gray));
            textView.setAlpha(0.5f);
        }
    }

    private void setTimeText(TextView textView, double seconds, int colorResId) {
        if (seconds > 0) {
            int minutes = (int) Math.round(seconds / 60.0);
            if (minutes < 1 && seconds >= 30) {
                textView.setText(textView.getContext().getString(R.string.time_less_than_minute));
            } else if (minutes < 60) {
                textView.setText(textView.getContext().getString(R.string.time_minutes, minutes));
            } else {
                int hours = minutes / 60;
                int remainingMinutes = minutes % 60;
                if (remainingMinutes > 0) {
                    textView.setText(textView.getContext().getString(R.string.time_hours_minutes,
                            hours, " " + remainingMinutes + "m"));
                } else {
                    textView.setText(textView.getContext().getString(R.string.time_hours_only, hours));
                }
            }
            textView.setTextColor(ContextCompat.getColor(textView.getContext(), colorResId));
            textView.setAlpha(1.0f);
        } else {
            textView.setText(textView.getContext().getString(R.string.time_minutes, 0));
            textView.setTextColor(ContextCompat.getColor(textView.getContext(), R.color.gray));
            textView.setAlpha(0.5f);
        }
    }
}