package org.ojrandom.paiesque.ui;

import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import org.ojrandom.paiesque.PaiesqueApplication;
import org.ojrandom.paiesque.R;
import org.ojrandom.paiesque.data.DataManager;
import org.ojrandom.paiesque.data.SettingsManager;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.pai.HealthMetricsAnalyzer;
import org.ojrandom.paiesque.pai.ZoneCalculator;

import java.util.List;

public class ZoneAnalysisActivity extends AppCompatActivity {
    private static final String TAG = "ZoneAnalysisActivity";

    private SettingsManager settingsManager;
    private DataManager dataManager;
    private RecyclerView recyclerView;
    private TextView emptyStateText;
    private TextView analysisSummaryText;
    private LinearLayout healthInsightsHeader;
    private ViewGroup healthInsightsContent;
    private ImageView iconHealthInsightsExpand;
    private TextView textHealthInsights;
    private TextView textInsightsSummary;
    private ZoneAnalysisAdapter adapter;
    private HealthMetricsAnalyzer.HealthInsights currentInsights;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_zone_analysis);

        // FIX: Use singleton DataManager but ensure it's properly initialized
        PaiesqueApplication app = (PaiesqueApplication) getApplication();
        dataManager = app.getDataManager();

        // FIX: Get SettingsManager from DataManager
        settingsManager = dataManager.getSettingsManager();

        // FIX: Check if settings are available
        if (settingsManager == null) {
            AppLogger.e(TAG, "SettingsManager is null - cannot proceed");
            // FIX: Use application context for Toast
            Toast.makeText(this.getApplicationContext(), "Settings not available. Please configure settings first.", Toast.LENGTH_LONG).show();
            finish();
            return;
        }

        initializeViews();
        setupRecyclerView();
        setupHealthInsightsToggle();
        updateZoneHeaders();
        loadZoneData();
    }

    private void initializeViews() {
        recyclerView = findViewById(R.id.recycler_zone_analysis);
        emptyStateText = findViewById(R.id.text_empty_state);
        analysisSummaryText = findViewById(R.id.text_analysis_summary);

        healthInsightsHeader = findViewById(R.id.health_insights_header);
        healthInsightsContent = findViewById(R.id.health_insights_content);
        iconHealthInsightsExpand = findViewById(R.id.icon_health_insights_expand);
        textHealthInsights = findViewById(R.id.text_health_insights);
        textInsightsSummary = findViewById(R.id.text_insights_summary);

        AppLogger.d(TAG, "Health insights header: " + (healthInsightsHeader != null));
        AppLogger.d(TAG, "Health insights content: " + (healthInsightsContent != null));
        AppLogger.d(TAG, "Health insights expand icon: " + (iconHealthInsightsExpand != null));
        AppLogger.d(TAG, "Health insights text: " + (textHealthInsights != null));
        AppLogger.d(TAG, "Health insights summary: " + (textInsightsSummary != null));

        if (getSupportActionBar() != null) {
            getSupportActionBar().setDisplayHomeAsUpEnabled(true);
            getSupportActionBar().setTitle(R.string.zone_analysis_title);
        }
    }

    private void setupRecyclerView() {
        LinearLayoutManager layoutManager = new LinearLayoutManager(this);
        recyclerView.setLayoutManager(layoutManager);

        adapter = new ZoneAnalysisAdapter();
        recyclerView.setAdapter(adapter);
    }

    private void updateZoneHeaders() {
        // FIX: Use the singleton SettingsManager
        int restingHR = settingsManager.getRHR();
        int maxHR = settingsManager.getMaxHR();

        if (restingHR == 0) restingHR = 60;
        if (maxHR == 0) maxHR = 180;

        int measuredMaxHR = getMeasuredMaxHR();
        ZoneCalculator calculator = new ZoneCalculator(restingHR, maxHR, measuredMaxHR);

        updateZoneHeader(R.id.text_zone_50_59_header, calculator.getZone50_59());
        updateZoneHeader(R.id.text_zone_60_69_header, calculator.getZone60_69());
        updateZoneHeader(R.id.text_zone_70_79_header, calculator.getZone70_79());
        updateZoneHeader(R.id.text_zone_80_89_header, calculator.getZone80_89());
        updateZoneHeader(R.id.text_zone_90_plus_header, calculator.getZone90_100());
    }

    private void updateZoneHeader(int headerViewId, ZoneCalculator.ZoneRange zoneRange) {
        TextView headerView = findViewById(headerViewId);
        if (headerView != null) {
            headerView.setText(zoneRange.getCompactDisplayText());
        }
    }

    private int getMeasuredMaxHR() {
        try {
            return dataManager.getPaiesqueDBManager().getMeasuredMaxHR();
        } catch (Exception e) {
            AppLogger.e(TAG, "Error getting measured max HR", e);
            return 0;
        }
    }

    private void loadZoneData() {
        new Thread(() -> {
            try {
                int deviceId = getIntent().getIntExtra("DEVICE_ID", -1);

                // FIX: Use the working method that loads 30 days of historical data
                List<ZoneDayData> zoneData = dataManager.getPaiesqueDBManager()
                        .getZoneAnalysisData(deviceId, 30);

                AppLogger.d(TAG, "Zone analysis data loaded: " + (zoneData != null ? zoneData.size() : 0) + " days");

                if (zoneData != null && !zoneData.isEmpty()) {
                    for (ZoneDayData data : zoneData.subList(0, Math.min(3, zoneData.size()))) {
                        AppLogger.d(TAG, "Sample zone data: " + data.toString());
                    }
                }

                runOnUiThread(() -> {
                    if (zoneData != null && !zoneData.isEmpty()) {
                        adapter.setData(zoneData);
                        updateSummary(zoneData);
                        showHealthInsights(deviceId);
                        emptyStateText.setVisibility(View.GONE);
                        recyclerView.setVisibility(View.VISIBLE);
                        analysisSummaryText.setVisibility(View.VISIBLE);
                    } else {
                        showEmptyState();
                        AppLogger.d(TAG, "No zone data available - showing empty state");
                    }
                });

            } catch (Exception e) {
                AppLogger.e(TAG, "Error loading zone data", e);
                runOnUiThread(this::showEmptyState);
            }
        }).start();
    }

    private void updateSummary(List<ZoneDayData> zoneData) {
        if (zoneData.isEmpty()) return;

        int totalDays = zoneData.size();
        double totalPai = 0;
        int daysWithActivity = 0;

        for (ZoneDayData data : zoneData) {
            totalPai += data.getTotalPai();
            if (data.getTotalPai() > 0) {
                daysWithActivity++;
            }
        }

        double averagePai = totalPai / totalDays;
        String summary = getString(R.string.summary_format, totalDays, averagePai, daysWithActivity);

        AppLogger.d(TAG, "Setting summary text: " + summary);
        analysisSummaryText.setText(summary);
    }

    private void updateInsightsSummary(HealthMetricsAnalyzer.HealthInsights insights) {
        StringBuilder summary = new StringBuilder();

        if (insights.measuredRHR > 0) {
            int rhrDiff = insights.assumedRHR - insights.measuredRHR;
            if (rhrDiff == 0) {
                summary.append(getString(R.string.rhr_summary_matches));
            } else if (rhrDiff > 0) {
                summary.append(getString(R.string.rhr_summary_lower, rhrDiff));
            } else {
                summary.append(getString(R.string.rhr_summary_higher, -rhrDiff));
            }
        }

        if (insights.measuredMaxHR > 0) {
            if (summary.length() > 0) summary.append(" • ");
            int maxHrDiff = insights.measuredMaxHR - insights.assumedMaxHR;
            if (maxHrDiff == 0) {
                summary.append(getString(R.string.max_hr_summary_matches));
            } else if (maxHrDiff > 0) {
                summary.append(getString(R.string.max_hr_summary_higher, maxHrDiff));
            } else {
                summary.append(getString(R.string.max_hr_summary_lower, -maxHrDiff));
            }
        }

        if (summary.length() == 0) {
            textInsightsSummary.setText(R.string.health_insights_no_data);
        } else {
            textInsightsSummary.setText(summary.toString());
        }
    }

    private void showHealthInsights(int deviceId) {
        // FIX: Use the singleton SettingsManager
        var historicalRHR = dataManager.getPaiesqueDBManager().loadHistoricalRHR(deviceId);
        var database = dataManager.getPaiesqueDBManager().getWritableDatabase();

        // Create string resources for the analyzer
        HealthMetricsAnalyzer.HealthInsightsStrings strings =
                new HealthMetricsAnalyzer.HealthInsightsStrings(
                        getString(R.string.rhr_no_data),
                        getString(R.string.rhr_matches),
                        getString(R.string.rhr_lower),
                        getString(R.string.rhr_higher),
                        getString(R.string.max_hr_no_data),
                        getString(R.string.max_hr_matches),
                        getString(R.string.max_hr_higher),
                        getString(R.string.max_hr_lower),
                        getString(R.string.educational_note_lower_manual),
                        getString(R.string.educational_note_lower_formula),
                        getString(R.string.educational_note_higher_manual),
                        getString(R.string.educational_note_higher_formula),
                        getString(R.string.educational_note_matches_manual),
                        getString(R.string.educational_note_matches_formula)
                );

        HealthMetricsAnalyzer analyzer = new HealthMetricsAnalyzer();
        currentInsights = analyzer.analyzeHealthMetrics(historicalRHR, settingsManager, database, strings);
        updateInsightsSummary(currentInsights);
    }

    private void displayInsights(HealthMetricsAnalyzer.HealthInsights insights) {
        String insightText = getString(R.string.health_insights_your_profile) + "\n\n" +
                "• " + insights.rhrInsight + "\n\n" +
                "• " + insights.maxHrInsight + "\n\n" +
                "💡 " + insights.educationalNote;

        textHealthInsights.setText(insightText);
    }

    private void showEmptyState() {
        emptyStateText.setVisibility(View.VISIBLE);
        recyclerView.setVisibility(View.GONE);
        analysisSummaryText.setText(R.string.zone_analysis_empty_summary);
    }

    @Override
    public boolean onSupportNavigateUp() {
        finish();
        return true;
    }

    private void setupHealthInsightsToggle() {
        AppLogger.d(TAG, "Setting up health insights toggle");

        // Debug initial state
        AppLogger.d(TAG, "Before setup - Content visibility: " + healthInsightsContent.getVisibility());
        AppLogger.d(TAG, "Before setup - Content height: " + healthInsightsContent.getHeight());

        // Ensure content is completely gone and doesn't take space
        healthInsightsContent.setVisibility(View.GONE);

        // Force layout parameters to ensure it doesn't take space when collapsed
        ViewGroup.LayoutParams params = healthInsightsContent.getLayoutParams();
        if (params != null) {
            params.height = 0;
            healthInsightsContent.setLayoutParams(params);
            AppLogger.d(TAG, "Set content layout height to 0");
        }

        iconHealthInsightsExpand.setRotation(0);

        if (healthInsightsHeader != null) {
            healthInsightsHeader.setOnClickListener(v -> {
                AppLogger.d(TAG, "Health insights header clicked!");
                AppLogger.d(TAG, "Before toggle - Content visibility: " + healthInsightsContent.getVisibility());
                AppLogger.d(TAG, "Before toggle - Content height: " + healthInsightsContent.getHeight());
                toggleHealthInsights();
            });
            AppLogger.d(TAG, "ClickListener set on health insights header");
        }

        // Force layout update
        healthInsightsContent.requestLayout();
        healthInsightsHeader.requestLayout();

        AppLogger.d(TAG, "After setup - Content visibility: " + healthInsightsContent.getVisibility());
        AppLogger.d(TAG, "After setup - Content height: " + healthInsightsContent.getHeight());
    }

    private void toggleHealthInsights() {
        AppLogger.d(TAG, "toggleHealthInsights called");

        boolean isVisible = healthInsightsContent.getVisibility() == View.VISIBLE;
        AppLogger.d(TAG, "Current visibility: " + isVisible);
        AppLogger.d(TAG, "Current content height: " + healthInsightsContent.getHeight());

        if (isVisible) {
            // Collapse: hide content and set height to 0 to prevent space allocation
            healthInsightsContent.setVisibility(View.GONE);

            ViewGroup.LayoutParams params = healthInsightsContent.getLayoutParams();
            if (params != null) {
                params.height = 0;
                healthInsightsContent.setLayoutParams(params);
                AppLogger.d(TAG, "Collapsed - set content height to 0");
            }

            iconHealthInsightsExpand.setRotation(0);
            AppLogger.d(TAG, "Hiding insights content");
        } else {
            // Expand: show content and reset to wrap_content
            ViewGroup.LayoutParams params = healthInsightsContent.getLayoutParams();
            if (params != null) {
                params.height = ViewGroup.LayoutParams.WRAP_CONTENT;
                healthInsightsContent.setLayoutParams(params);
                AppLogger.d(TAG, "Expanding - set content height to WRAP_CONTENT");
            }

            healthInsightsContent.setVisibility(View.VISIBLE);
            iconHealthInsightsExpand.setRotation(180);
            AppLogger.d(TAG, "Showing insights content");

            // Always populate insights when expanding
            if (currentInsights != null) {
                AppLogger.d(TAG, "Populating insights content");
                displayInsights(currentInsights);
            } else {
                AppLogger.d(TAG, "No insights available to display");
                textHealthInsights.setText(R.string.health_insights_loading);

                // Load insights if we have a device ID
                int deviceId = getIntent().getIntExtra("DEVICE_ID", -1);
                if (deviceId != -1) {
                    showHealthInsights(deviceId);
                }
            }
        }

        // Force layout updates
        healthInsightsContent.requestLayout();
        healthInsightsHeader.requestLayout();

        // FIX: Use Handler with Looper to avoid deprecation
        new Handler(Looper.getMainLooper()).postDelayed(() -> {
            AppLogger.d(TAG, "After layout - Content visibility: " + healthInsightsContent.getVisibility());
            AppLogger.d(TAG, "After layout - Content height: " + healthInsightsContent.getHeight());
            AppLogger.d(TAG, "After layout - Content measured height: " + healthInsightsContent.getMeasuredHeight());
        }, 100);
    }

    @Override
    public void finish() {
        super.finish();
    }
}