package org.ojrandom.paiesque.ui;

import android.app.Activity;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import androidx.core.content.ContextCompat;

import org.ojrandom.paiesque.R;
import org.ojrandom.paiesque.logging.AppLogger;
import org.ojrandom.paiesque.rhr.RhrResult;

public class RhrViewManager {
    private final View rhrSummaryCard;
    private final TextView lastWeekView;
    private final TextView lastMonthView;
    private final TextView baselineView;
    private final TextView trendView;
    private final Activity activity;

    // Store the summary data separately since RhrResult is now simplified
    private int lastWeekRHR = -1;
    private int lastMonthRHR = -1;
    private int currentBaselineRHR = -1;

    public RhrViewManager(View rhrSummaryCard,
                          TextView valueLastWeekRhr, TextView valueLastMonthRhr,
                          TextView valueCurrentBaselineRhr, TextView textRhrTrend,
                          Activity activity) {
        this.rhrSummaryCard = rhrSummaryCard;
        this.lastWeekView = valueLastWeekRhr;
        this.lastMonthView = valueLastMonthRhr;
        this.baselineView = valueCurrentBaselineRhr;
        this.trendView = textRhrTrend;
        this.activity = activity;
    }

    /**
     * NEW METHOD: Update RHR summary data directly
     */
    public void updateRHRSummaryData(int lastWeekRHR, int lastMonthRHR, int currentBaselineRHR) {
        this.lastWeekRHR = lastWeekRHR;
        this.lastMonthRHR = lastMonthRHR;
        this.currentBaselineRHR = currentBaselineRHR;
        displayRHRSummary();
    }

    /**
     * OLD METHOD: Now deprecated - use updateRHRSummaryData instead
     */
    public void displayRHRSummary(RhrResult rhrResult) {
        AppLogger.w("RhrViewManager", "displayRHRSummary(RhrResult) is deprecated. Use updateRHRSummaryData() instead.");
        if (rhrResult != null) {
            // For backward compatibility, you could set some values from the single result
            // but this doesn't make much sense for the summary view
            setDefaultRHRValues();
        } else {
            setDefaultRHRValues();
        }
    }

    /**
     * Display the current stored RHR summary
     */
    public void displayRHRSummary() {
        activity.runOnUiThread(() -> {
            setRHRValue(lastWeekView, lastWeekRHR, "Last Week");
            setRHRValue(lastMonthView, lastMonthRHR, "Last Month");
            setRHRValue(baselineView, currentBaselineRHR, "Current Baseline");
            updateTrend(lastWeekRHR, currentBaselineRHR);
        });
    }

    private void setRHRValue(TextView textView, int value, String label) {
        if (value != -1) {
            textView.setText(String.valueOf(value));
            setRHRColor(textView, value);
        } else {
            textView.setText("--");
            textView.setTextColor(ContextCompat.getColor(textView.getContext(), android.R.color.darker_gray));
        }
    }

    private void setRHRColor(TextView textView, int rhrValue) {
        int colorRes;
        if (rhrValue < 50) {
            colorRes = R.color.excellent_green;        // <50: Typical of endurance athletes
        } else if (rhrValue < 60) {
            colorRes = R.color.good_blue;             // 50-59: Well above average fitness
        } else if (rhrValue < 70) {
            colorRes = R.color.ideal_color;           // 60-69: AHA's ideal range for lowest CV risk
        } else if (rhrValue < 80) {
            colorRes = R.color.average_yellow;        // 70-79: Normal population average
        } else if (rhrValue < 90) {
            colorRes = R.color.borderline_orange;     // 80-89: Borderline, increased health risks
        } else {
            colorRes = R.color.high_red;              // ≥90: Clinically significant risk elevation
        }
        textView.setTextColor(ContextCompat.getColor(textView.getContext(), colorRes));
    }

    private void updateTrend(int lastWeekRHR, int currentBaselineRHR) {
        if (lastWeekRHR != -1 && currentBaselineRHR != -1) {
            int deviation = lastWeekRHR - currentBaselineRHR;
            String trendText = getTrendText(deviation);
            trendView.setText(trendText);
            trendView.setTextColor(getTrendColor(deviation));
        } else {
            trendView.setText(activity.getString(R.string.rhr_trend_need_data));
            trendView.setTextColor(ContextCompat.getColor(trendView.getContext(), android.R.color.darker_gray));
        }
    }

    private String getTrendText(int deviationFromBaseline) {
        if (deviationFromBaseline > 5) return activity.getString(R.string.rhr_trend_above_high);
        if (deviationFromBaseline > 2) return activity.getString(R.string.rhr_trend_above_low);
        if (deviationFromBaseline < -5) return activity.getString(R.string.rhr_trend_below_high);
        if (deviationFromBaseline < -2) return activity.getString(R.string.rhr_trend_below_low);
        return activity.getString(R.string.rhr_trend_at_baseline);
    }

    private int getTrendColor(int deviation) {
        if (deviation > 5) return ContextCompat.getColor(trendView.getContext(), R.color.high_red);
        if (deviation > 2) return ContextCompat.getColor(trendView.getContext(), R.color.average_yellow);
        if (deviation < -5) return ContextCompat.getColor(trendView.getContext(), R.color.excellent_green);
        if (deviation < -2) return ContextCompat.getColor(trendView.getContext(), R.color.good_blue);
        return ContextCompat.getColor(trendView.getContext(), android.R.color.white);
    }

    private void setDefaultRHRValues() {
        lastWeekView.setText("--");
        lastMonthView.setText("--");
        baselineView.setText("--");
        trendView.setText(activity.getString(R.string.rhr_trend_no_data));

        lastWeekView.setTextColor(ContextCompat.getColor(lastWeekView.getContext(), android.R.color.darker_gray));
        lastMonthView.setTextColor(ContextCompat.getColor(lastMonthView.getContext(), android.R.color.darker_gray));
        baselineView.setTextColor(ContextCompat.getColor(baselineView.getContext(), android.R.color.darker_gray));
        trendView.setTextColor(ContextCompat.getColor(trendView.getContext(), android.R.color.darker_gray));
    }

    // Getters for the current values
    public int getLastWeekRHR() { return lastWeekRHR; }
    public int getLastMonthRHR() { return lastMonthRHR; }
    public int getCurrentBaselineRHR() { return currentBaselineRHR; }
}