package org.ojrandom.paiesque.ui;

import android.app.Activity;
import android.util.Log;
import android.view.View;
import android.widget.ProgressBar;
import android.widget.TextView;

import androidx.core.content.ContextCompat;

import org.ojrandom.paiesque.R;
import org.ojrandom.paiesque.data.AppConstants;
import org.ojrandom.paiesque.logging.AppLogger;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

public class PaiViewManager {
    private static final String TAG = "PaiViewManager";

    private final View paiSummaryCard;
    private final TextView valueTodayPai;
    private final TextView value7DayPai;
    private final ProgressBar progressPai;
    private final TextView textPaiProgressStatus;
    private final Activity activity;

    // Zone breakdown views
    private final View zoneBreakdownContainer;
    private final ProgressBar progressZone50_59;
    private final ProgressBar progressZone60_69;
    private final ProgressBar progressZone70_79;
    private final ProgressBar progressZone80_89;
    private final ProgressBar progressZone90Plus;
    private final TextView textZone50_59;
    private final TextView textZone60_69;
    private final TextView textZone70_79;
    private final TextView textZone80_89;
    private final TextView textZone90Plus;
    private final TextView textZoneSummary;

    private boolean showingDefaultZoneData = false;

    // Click listener for zone details
    private View.OnClickListener zoneDetailsClickListener;

    public PaiViewManager(View paiSummaryCard,
                          TextView valueTodayPai, TextView value7DayPai,
                          ProgressBar progressPai, TextView textPaiProgressStatus,
                          Activity activity) {
        this.paiSummaryCard = paiSummaryCard;
        this.valueTodayPai = valueTodayPai;
        this.value7DayPai = value7DayPai;
        this.progressPai = progressPai;
        this.textPaiProgressStatus = textPaiProgressStatus;
        this.activity = activity;

        // Initialize zone breakdown views
        this.zoneBreakdownContainer = safeFindViewById(paiSummaryCard, R.id.zone_breakdown_container);
        this.progressZone50_59 = safeFindViewById(paiSummaryCard, R.id.progress_zone_50_59);
        this.progressZone60_69 = safeFindViewById(paiSummaryCard, R.id.progress_zone_60_69);
        this.progressZone70_79 = safeFindViewById(paiSummaryCard, R.id.progress_zone_70_79);
        this.progressZone80_89 = safeFindViewById(paiSummaryCard, R.id.progress_zone_80_89);
        this.progressZone90Plus = safeFindViewById(paiSummaryCard, R.id.progress_zone_90_plus);
        this.textZone50_59 = safeFindViewById(paiSummaryCard, R.id.text_zone_50_59);
        this.textZone60_69 = safeFindViewById(paiSummaryCard, R.id.text_zone_60_69);
        this.textZone70_79 = safeFindViewById(paiSummaryCard, R.id.text_zone_70_79);
        this.textZone80_89 = safeFindViewById(paiSummaryCard, R.id.text_zone_80_89);
        this.textZone90Plus = safeFindViewById(paiSummaryCard, R.id.text_zone_90_plus);
        this.textZoneSummary = safeFindViewById(paiSummaryCard, R.id.text_zone_summary);

        // Initially hide zone breakdown until we have data
        if (zoneBreakdownContainer != null) {
            zoneBreakdownContainer.setVisibility(View.GONE);
        }

        setupClickListeners();
    }

    // Setup click listeners for zone details only
    private void setupClickListeners() {
        if (zoneBreakdownContainer != null) {
            zoneBreakdownContainer.setOnClickListener(v -> {
                AppLogger.d(TAG, "Zone breakdown container clicked");
                if (zoneDetailsClickListener != null) {
                    zoneDetailsClickListener.onClick(v);
                }
            });

            // Make it clear it's clickable
            zoneBreakdownContainer.setClickable(true);
            zoneBreakdownContainer.setFocusable(true);
        }
    }

    // Safe findViewById method to handle missing views
    private <T extends View> T safeFindViewById(View parent, int id) {
        try {
            return parent.findViewById(id);
        } catch (Exception e) {
            AppLogger.w(TAG, "View not found with ID: " + id, e);
            return null;
        }
    }

    // Set click listener for zone details
    public void setZoneDetailsClickListener(View.OnClickListener listener) {
        this.zoneDetailsClickListener = listener;
    }

    public void updatePaiSummary(List<Map<String, String>> rollingPAI) {
        AppLogger.d(TAG, "updatePaiSummary called with: " +
                (rollingPAI != null ? rollingPAI.size() + " items" : "null data"));

        if (rollingPAI == null || rollingPAI.isEmpty()) {
            AppLogger.w(TAG, "No PAI data available - showing empty state");
            activity.runOnUiThread(() -> {
                if (valueTodayPai != null) valueTodayPai.setText("--");
                if (value7DayPai != null) value7DayPai.setText("--");
                if (progressPai != null) progressPai.setProgress(0);
                if (textPaiProgressStatus != null) {
                    textPaiProgressStatus.setText("No data");
                    textPaiProgressStatus.setTextColor(ContextCompat.getColor(activity, R.color.gray));
                }
                hideZoneBreakdown();
            });
            return;
        }

        // Extract the data on the background thread FIRST
        int todayPai = 0;
        int sevenDayPai = 0;

        try {
            Map<String, String> latestData = rollingPAI.get(rollingPAI.size() - 1);
            todayPai = extractIntValue(latestData, "pai_day");
            sevenDayPai = extractIntValue(latestData, "pai_7_day");
            AppLogger.d(TAG, "Extracted values - Today: " + todayPai + ", 7-Day: " + sevenDayPai);
        } catch (Exception e) {
            AppLogger.e(TAG, "Error extracting PAI values", e);
        }

        final int finalTodayPai = todayPai;
        final int finalSevenDayPai = sevenDayPai;

        // NOW update UI on main thread
        activity.runOnUiThread(() -> {
            try {
                if (valueTodayPai != null) {
                    valueTodayPai.setText(String.valueOf(finalTodayPai));
                }
                if (value7DayPai != null) {
                    value7DayPai.setText(String.valueOf(finalSevenDayPai));
                }
                if (progressPai != null) {
                    int progress = Math.min(finalSevenDayPai, 100);
                    progressPai.setProgress(progress);
                }
                if (textPaiProgressStatus != null) {
                    int colorRes;
                    String statusText;

                    if (finalSevenDayPai >= 100) {
                        statusText = activity.getString(R.string.pai_status_optimal);
                        colorRes = R.color.green;
                    } else if (finalSevenDayPai >= 50) {
                        statusText = activity.getString(R.string.pai_status_good_progress, finalSevenDayPai);
                        colorRes = R.color.orange;
                    } else if (finalSevenDayPai > 0) {
                        statusText = activity.getString(R.string.pai_status_keep_going, finalSevenDayPai);
                        colorRes = R.color.orange;
                    } else {
                        statusText = activity.getString(R.string.pai_status_no_activity);
                        colorRes = R.color.gray;
                    }

                    textPaiProgressStatus.setText(statusText);
                    textPaiProgressStatus.setTextColor(ContextCompat.getColor(activity, colorRes));
                }

                AppLogger.d(TAG, "UI updated - Today: " + finalTodayPai + ", 7-Day: " + finalSevenDayPai);

            } catch (Exception e) {
                AppLogger.e(TAG, "Error updating PAI UI", e);
            }
        });
    }

    private int extractIntValue(Map<String, String> data, String... possibleKeys) {
        if (data == null) return 0;

        for (String key : possibleKeys) {
            if (data.containsKey(key)) {
                try {
                    String value = data.get(key);
                    if (value != null && !value.trim().isEmpty()) {
                        if (value.contains(".")) {
                            return (int) Math.round(Double.parseDouble(value));
                        }
                        return Integer.parseInt(value.trim());
                    }
                } catch (NumberFormatException e) {
                    AppLogger.w(TAG, "Failed to parse value for key '" + key + "': " + data.get(key));
                }
            }
        }

        AppLogger.w(TAG, "No valid key found in: " + data.keySet() + " - tried: " + Arrays.toString(possibleKeys));
        return 0;
    }

    public void updateZoneBreakdown(Map<String, Double> paiByZone, double totalPai) {
        AppLogger.d(TAG, "updateZoneBreakdown called with: " + paiByZone + ", totalPai: " + totalPai);

        activity.runOnUiThread(() -> {
            try {
                boolean hasValidData = (paiByZone != null && !paiByZone.isEmpty() && totalPai > 0);
                showingDefaultZoneData = !hasValidData;

                AppLogger.d(TAG, "Showing zone breakdown container - hasValidData: " + hasValidData);

                // Always show zone breakdown container
                if (zoneBreakdownContainer != null) {
                    zoneBreakdownContainer.setVisibility(View.VISIBLE);
                    AppLogger.d(TAG, "Zone breakdown container visibility set to VISIBLE");
                } else {
                    AppLogger.e(TAG, "zoneBreakdownContainer is null!");
                }

                if (hasValidData) {
                    updateZoneProgress(progressZone50_59, textZone50_59,
                            paiByZone.getOrDefault(AppConstants.ZONE_50_59, 0.0), totalPai);
                    updateZoneProgress(progressZone60_69, textZone60_69,
                            paiByZone.getOrDefault(AppConstants.ZONE_60_69, 0.0), totalPai);
                    updateZoneProgress(progressZone70_79, textZone70_79,
                            paiByZone.getOrDefault(AppConstants.ZONE_70_79, 0.0), totalPai);
                    updateZoneProgress(progressZone80_89, textZone80_89,
                            paiByZone.getOrDefault(AppConstants.ZONE_80_89, 0.0), totalPai);
                    updateZoneProgress(progressZone90Plus, textZone90Plus,
                            paiByZone.getOrDefault(AppConstants.ZONE_90_PLUS, 0.0), totalPai);

                    // Hide the entire summary line when data is available
                    if (textZoneSummary != null) {
                        textZoneSummary.setVisibility(View.GONE);
                        AppLogger.d(TAG, "Zone summary hidden - data available");
                    }
                } else {
                    updateZoneProgress(progressZone50_59, textZone50_59, 0.0, 1.0);
                    updateZoneProgress(progressZone60_69, textZone60_69, 0.0, 1.0);
                    updateZoneProgress(progressZone70_79, textZone70_79, 0.0, 1.0);
                    updateZoneProgress(progressZone80_89, textZone80_89, 0.0, 1.0);
                    updateZoneProgress(progressZone90Plus, textZone90Plus, 0.0, 1.0);

                    // Show the summary line only when no data
                    if (textZoneSummary != null) {
                        textZoneSummary.setVisibility(View.VISIBLE);
                        textZoneSummary.setText(activity.getString(R.string.zone_no_activity_data));
                        AppLogger.d(TAG, "Zone summary shown with empty state");
                    }
                }

                AppLogger.d(TAG, "Zone breakdown update completed successfully - showingDefault: " + showingDefaultZoneData);

            } catch (Exception e) {
                AppLogger.e(TAG, "Error updating zone breakdown", e);
                showDefaultZoneData();
            }
        });
    }

    private void updateZoneProgress(ProgressBar progressBar, TextView textView, double zonePai, double totalPai) {
        if (progressBar == null || textView == null) {
            AppLogger.w(TAG, "ProgressBar or TextView is null, skipping update");
            return;
        }

        if (totalPai > 0 && zonePai > 0) {
            int percentage = (int) Math.round((zonePai / totalPai) * 100);
            progressBar.setProgress(percentage);
            textView.setText(percentage + "%");
            AppLogger.d(TAG, "Zone progress updated: " + percentage + "%");
        } else {
            progressBar.setProgress(0);
            textView.setText("0%");
            AppLogger.d(TAG, "Zone progress set to 0% (no zone PAI)");
        }
    }

    private void showDefaultZoneData() {
        if (zoneBreakdownContainer != null) {
            zoneBreakdownContainer.setVisibility(View.VISIBLE);
        }

        updateZoneProgress(progressZone50_59, textZone50_59, 0.0, 1.0);
        updateZoneProgress(progressZone60_69, textZone60_69, 0.0, 1.0);
        updateZoneProgress(progressZone70_79, textZone70_79, 0.0, 1.0);
        updateZoneProgress(progressZone80_89, textZone80_89, 0.0, 1.0);
        updateZoneProgress(progressZone90Plus, textZone90Plus, 0.0, 1.0);

        if (textZoneSummary != null) {
            textZoneSummary.setText(activity.getString(R.string.zone_no_activity_data));
        }

        showingDefaultZoneData = true;
    }

    private void hideZoneBreakdown() {
        if (zoneBreakdownContainer != null) {
            zoneBreakdownContainer.setVisibility(View.GONE);
        }
    }

    public void showZoneBreakdown(boolean show) {
        activity.runOnUiThread(() -> {
            if (zoneBreakdownContainer != null) {
                zoneBreakdownContainer.setVisibility(show ? View.VISIBLE : View.GONE);
            }
        });
    }

    public void debugZoneData() {
        AppLogger.d(TAG, "Zone views status:");
        AppLogger.d(TAG, "zoneBreakdownContainer: " + (zoneBreakdownContainer != null ? "found" : "null"));
        AppLogger.d(TAG, "progressZone50_59: " + (progressZone50_59 != null ? "found" : "null"));
        AppLogger.d(TAG, "textZone50_59: " + (textZone50_59 != null ? "found" : "null"));
        AppLogger.d(TAG, "textZoneSummary: " + (textZoneSummary != null ? "found" : "null"));
    }
}